import pickle
import numpy as np
from common.standard_POD import standard_POD
from common.euler_method import euler_method_dense

### This file constructs a reduced model and computes a reduced solution via standard POD.


test_problem = 'SPE10'

# Load test problem, necessary FE data and high-fidelity solution:
FE_results = pickle.load(open('results/FE_data_problem='+test_problem+'.pickle','rb'))
FE_solution = FE_results['FE_solution']
mass_bc = FE_results['mass']
stiff_bc = FE_results['stiff']
stiffs_coeff_in_time_bc = FE_results['stiffs_coeff_in_time']
rhs_matrix = FE_results['rhs_matrix']
grid_t = FE_results['grid_t']
ht = grid_t[1]-grid_t[0]
T_start = grid_t[0]
T_finish = grid_t[-1]
nt = len(grid_t) - 1
u_0 = FE_results['u0_discrete']


# Choose number of time steps used for construction of reduced model:
nt_POD = 315
# Choose tolerance for error in Frobenius norm:
tol_POD = 1e-08
# Run reduced basis construction via POD:
singular_vals, red_basis = standard_POD(mass_bc,FE_solution,nt_POD,tol_POD)


# Construct reduced model:
mass_red = red_basis.T.dot(mass_bc.dot(red_basis))
stiffs_red = []
for i in range(*(1,len(stiffs_coeff_in_time_bc))):
    stiff_red_temp = red_basis.T.dot(stiffs_coeff_in_time_bc[i].dot(red_basis))
    stiffs_red.append(stiff_red_temp)
rhs_matrix_red = red_basis.T.dot(rhs_matrix)
u0_red = np.linalg.solve(red_basis.T.dot(mass_bc.dot(red_basis)),red_basis.T.dot(mass_bc.dot(u_0)))

# Compute reduced solution for full time interval with reduced model and implicit euler:
red_solution = euler_method_dense(u0_red,T_start,T_finish,nt,mass_red,stiffs_red,rhs_matrix_red)

# Generate function in high dimensional FE space to compare with high-fidelity FE solution:
red_solution = red_basis.dot(red_solution)


# Compute L2 errors in time:
L2_errors_over_time_relative = np.zeros(len(grid_t))

for j in range(len(grid_t)):
    error_temp = FE_solution[:, j] - red_solution[:,j]
    L2_norm_error_temp = np.sqrt(error_temp.T.dot(mass_bc.dot(error_temp)))
    L2_norm_FE_temp = np.sqrt(FE_solution[:,j].T.dot(mass_bc.dot(FE_solution[:,j])))
    # Avoid that an divison by zero error occurs (for instance if u_0 = 0)
    L2_errors_over_time_relative[j] = np.divide(L2_norm_error_temp, L2_norm_FE_temp, out=np.zeros_like(L2_norm_error_temp), where=L2_norm_FE_temp != 0)

# Store reduced error:
np.save('results/POD_SPE10_on_315_of_500_rel_L2_error_over_time.npy', L2_errors_over_time_relative)




