import numpy as np
import scipy
from common.euler_method import euler_method_sparse
from common.assemble import gram_schmidt_ortho

### This function constructs a reduced basis from the SVD of the transfer operator that maps
### arbitrary initial conditions to the corresponding high-fidelity FE solution evaluated in
### a certain time step.
### To address problems with a right hand side different from zero we subsequently solve the
### PDE once with zero inital condition and the correct right hand side.


def transfer_operator(mass,lhs_solves,rhs_matrix,T_start,T_finish,nt,nt_trans,tol_trans):
    # mapping from time point 0 - for general version see below!
    # nt_trans: number of time steps for time end point of transfer operator
    # tol_trans: for determining dimension of reduced model

    # Define auxiliary right hand side for construction of spectral modes:
    rhs_matrix_aux = np.zeros(rhs_matrix.shape)

    # Compute time end point for range of transfer operator:
    ht = (T_finish-T_start)/nt
    T_finish_trans = nt_trans*ht

    # Generate high-fidelity FE solutions:
    H = np.zeros(mass.shape)
    identity_aux = np.identity(H.shape[1])
    for i in range(H.shape[1]):
        print(f'Computing function {i+1} of {H.shape[1]}.')
        # Compute solutions of PDE via implicit Euler with ith unit vector as initial condition
        # and store the solution in the last time step in H:
        u0_aux = identity_aux[:,i]
        H[:,i] = euler_method_sparse(u0_aux, T_start, T_finish_trans, nt_trans, mass, lhs_solves, rhs_matrix_aux)[:, -1]

    # Assemble and solve generalized eigenvalue problem and sort eigenvalues and -vectors in descending order:
    matrix_left = H.T.dot(mass.dot(H))
    eigvals, coeff_eigfuncs = scipy.linalg.eigh(matrix_left, mass.todense())
    ordering = np.argsort(eigvals)[::-1]
    eigvals = eigvals[ordering]
    coeff_eigfuncs = coeff_eigfuncs[:, ordering]

    # Determine dimension of reduced space:
    singular_vals = np.sqrt(eigvals[np.where(eigvals>=0)])
    red_dim = np.where(singular_vals <= tol_trans)[0][0]

    # Use reiterated Gram Schmidt to make eigenvectors orthonormal:
    coeff_eigfuncs = coeff_eigfuncs[:,:red_dim]
    gram_schmidt_ortho(coeff_eigfuncs,mass)

    # Compute high-fidelity FE solution corresponding to correct right hand side and zero initial conditions:
    u_f = euler_method_sparse(np.zeros(H.shape[1]), T_start, T_finish_trans, nt_trans, mass, lhs_solves, rhs_matrix)[:, -1]

    # Construct reduced basis adding representation of right hand side and orthonormalize again:
    red_basis = np.hstack((H.dot(coeff_eigfuncs),u_f[:,np.newaxis]))
    gram_schmidt_ortho(red_basis,mass)

    return singular_vals, red_basis



def transfer_operator_arbitrary(mass, lhs_solves, rhs_matrix, grid_t, start_int, end_int, tol_trans):
    # tol_trans: for determining dimension of reduced model

    # Cut rhs_matrix and lhs_solves:
    rhs_matrix_temp = rhs_matrix[:, start_int:end_int + 1]
    lhs_solves_temp = lhs_solves[start_int:end_int]

    # Define auxiliary right hand side for construction of spectral modes:
    rhs_matrix_aux = np.zeros(rhs_matrix_temp.shape)

    # Other important variables:
    trans_start = grid_t[start_int]
    trans_finish = grid_t[end_int]
    nt_trans = end_int-start_int

    # Generate high-fidelity FE solutions:
    H = np.zeros(mass.shape)
    identity_aux = np.identity(H.shape[1])
    for i in range(H.shape[1]):
        print(f'Computing function {i+1} of {H.shape[1]}.')
        # Compute solutions of PDE via implicit Euler with ith unit vector as initial condition
        # and store the solution in the last time step in H:
        u0_aux = identity_aux[:,i]
        H[:,i] = euler_method_sparse(u0_aux, trans_start, trans_finish, nt_trans, mass, lhs_solves_temp, rhs_matrix_aux)[:, -1]

    # Assemble and solve generalized eigenvalue problem and sort eigenvalues and -vectors in descending order:
    matrix_left = H.T.dot(mass.dot(H))
    eigvals, coeff_eigfuncs = scipy.linalg.eigh(matrix_left, mass.todense())
    ordering = np.argsort(eigvals)[::-1]
    eigvals = eigvals[ordering]
    coeff_eigfuncs = coeff_eigfuncs[:, ordering]

    # Determine dimension of reduced space:
    singular_vals = np.sqrt(eigvals[np.where(eigvals>=0)])
    red_dim = np.where(singular_vals <= tol_trans)[0][0]

    # Use reiterated Gram Schmidt to make eigenvectors orthonormal:
    coeff_eigfuncs = coeff_eigfuncs[:,:red_dim]
    gram_schmidt_ortho(coeff_eigfuncs,mass)

    # Compute high-fidelity FE solution corresponding to correct right hand side and zero initial conditions:
    u_f = euler_method_sparse(np.zeros(H.shape[1]), trans_start, trans_finish, nt_trans, mass, lhs_solves_temp, rhs_matrix_temp)[:, -1]

    # Construct reduced basis adding representation of right hand side and orthonormalize again:
    red_basis = np.hstack((H.dot(coeff_eigfuncs),u_f[:,np.newaxis]))
    gram_schmidt_ortho(red_basis,mass)

    return singular_vals, red_basis


def transfer_operator_arbitrary_svd(mass, lhs_solves, rhs_matrix, grid_t, start_int, end_int, red_dim):
    # Cut rhs_matrix and lhs_solves:
    rhs_matrix_temp = rhs_matrix[:, start_int:end_int + 1]
    lhs_solves_temp = lhs_solves[start_int:end_int]

    # Define auxiliary right hand side for construction of spectral modes:
    rhs_matrix_aux = np.zeros(rhs_matrix_temp.shape)

    # Other important variables:
    trans_start = grid_t[start_int]
    trans_finish = grid_t[end_int]
    nt_trans = end_int-start_int

    # Generate high-fidelity FE solutions:
    H = np.zeros(mass.shape)
    identity_aux = np.identity(H.shape[1])
    for i in range(H.shape[1]):
        print(f'Computing function {i+1} of {H.shape[1]}.')
        # Compute solutions of PDE via implicit Euler with ith unit vector as initial condition
        # and store the solution in the last time step in H:
        u0_aux = identity_aux[:,i]
        H[:,i] = euler_method_sparse(u0_aux, trans_start, trans_finish, nt_trans, mass, lhs_solves_temp, rhs_matrix_aux)[:, -1]

    # Compute singular values and left singular vectors of transfer operator:
    left_singular_vecs, singular_vals, _ = scipy.linalg.svd(H)

    # Reduced basis:
    red_basis = left_singular_vecs[:,:red_dim]

    # Compute high-fidelity FE solution corresponding to correct right hand side and zero initial conditions:
    u_f = euler_method_sparse(np.zeros(H.shape[1]), trans_start, trans_finish, nt_trans, mass, lhs_solves_temp, rhs_matrix_temp)[:, -1]

    # Construct reduced basis adding representation of right hand side and orthonormalize again:
    red_basis = np.hstack((red_basis,u_f[:,np.newaxis]))
    #gram_schmidt_ortho(red_basis,mass) ?

    return singular_vals, red_basis