import scipy
import numpy as np
from common.assemble import gram_schmidt_ortho

### This function implements the standard POD. We start the simulation at t=0 and stop after nt_POD time steps.
### We construct the reduced model via an eigenvalue problem and decide on the dimension of the reduced space
### via the singular values and the tolerance tol_POD for the error in the Frobenius norm.
### (see POD script of Stefan Volkwein)


def standard_POD(mass, FE_solution, nt_POD, tol_POD):
    # Define snapshot matrix:
    snapshots = FE_solution[:,:nt_POD+1]

    # Assemble and solve eigenvalue problem and sort eigenvalues and -vectors in descending order:
    matrix = snapshots.T.dot(mass.dot(snapshots))
    matrix = 0.5*(matrix+matrix.T) # to address rounding problems
    eigvals, eigvecs = scipy.linalg.eigh(matrix)
    ordering = np.argsort(eigvals)[::-1]
    eigvals = eigvals[ordering]
    eigvecs = eigvecs[:, ordering]

    # Determine dimension of reduced space:
    singularvals = np.sqrt(eigvals[np.where(eigvals>=0)])
    for i in range(len(singularvals)):
        C = np.sqrt(np.sum(singularvals[i+1:]**2))/np.sqrt(np.sum(singularvals**2))
        if C <= tol_POD:
            red_dim = i+1
            break

    # Compute reduced basis and re-orthonormalize (to be sure):
    red_basis = snapshots.dot(eigvecs[:,:red_dim])
    for i in range(red_dim):
        red_basis[:,i] *= 1/singularvals[i]
    gram_schmidt_ortho(red_basis,mass)

    return singularvals, red_basis