import numpy as np

### This file contains functions to implement the implicit Euler method for a parabolic PDE.

def euler_method_sparse(u_0, T_start, T_finish, nt, mass, lhs_solves, rhs_matrix):
    # This function implements the implicit Euler method for a parabolic PDE
    # for sparse mass and stiffness matrices M and A_i, (M + ht A_i) given as factorized SuperLU in "lhs_solves"

    # Time step size:
    ht = (T_finish-T_start)/nt

    # Allocate memory for storing solution in every discrete time point and initialize with initial condition:
    solution = np.zeros((mass.shape[1],nt+1))
    solution[:,0] = u_0

    # Compute solution trajectory via implicit Euler:
    for i in range(*(1,nt+1)):
        # Right hand side in current time point:
        F_temp = rhs_matrix[:,i]
        rhs_temp = ht*F_temp + mass.dot(solution[:,i-1])
        # Solve and store solution:
        solution[:,i] = lhs_solves[i-1](rhs_temp)

    return solution


def euler_method_dense(u_0, T_start, T_finish, nt, mass, stiffs, rhs_matrix):
    # This function implements the implicit Euler method for a time-dependent PDE
    # for dense mass and stiffness(+advection) matrices

    # Time step size:
    ht = (T_finish-T_start)/nt

    # Allocate memory for storing solution in every discrete time point and initialize with initial condition:
    solution = np.zeros((mass.shape[1],nt+1))
    solution[:,0] = u_0

    # Compute solution trajectory via implicit Euler:
    for i in range(*(1,nt+1)):
        # Right hand side in current time point:
        F_temp = rhs_matrix[:,i]
        rhs_temp = ht*F_temp + mass.dot(solution[:,i-1])
        # Solve and store solution:
        solution[:,i] = np.linalg.solve(mass + ht*stiffs[i - 1], rhs_temp)

    return solution