! c computation of the   Compton scattered radiation 
! c incident photons are monoenergetic and monodirectional (beam)
! c scattered photons are at different scattering angles and energies 
! 
!   
! c function R(x,x1,mu) is computed in 
! c SUBROUTINE REDFUNC(X,X1,QMU,S,SI,SQ,SU,SV,SF)
! c here x,x1 are the photon energies 
! c QMU=1-mu 
! c S=R; SI=R_I, SQ=R_Q, SU=R_U, SV=R_V, SF=S_Q - S_U 
! c where R, R_I, R_Q, R_U, R_V are the elements of the matrix 
! c
! c     ( R   R_I  0  0 ) 
! c R = ( R_I R_Q  0  0 )
! c     ( 0   0   R_U 0 )
! c     ( 0   0   0  R_V)
! c see equation (55) in Nagirner & Poutanen 1993, A&A, 275, 325-336 
! c 

      IMPLICIT REAL*8(A-H,O-Z)
! c number of energy points  ,  number of cosine angle points , number of point in Gauss-Laguerre quadrature     
      PARAMETER(nx=51,NMU=16,NL=10,NF=2)
      COMMON /WYY/ YE,Y1,YR1,YR2,DY,YKY,DK02,DK2Y
      DATA EPS/1D-14/, NOUT/20/,NOUTPUT/24/,NOUTPUT/25/,NOUTPUTTEST/26/   
      DIMENSION XX(NX),XXLOG(NX)
      DIMENSION UMU(NMU),AMU(NMU),UMU_half(NMU),AMU_half(NMU)
! c planck spectrum 
      DIMENSION dinten(NX,NMU)
      real*8, allocatable :: SourI(:,:,:),SourQ(:,:,:),dI(:,:,:),dQ(:,:,:)
      real*8, allocatable :: dI_tot(:,:,:),dQ_tot(:,:,:),synch(:),sigma_cs(:)
      real*8, allocatable :: dI_tmp(:),dQ_tmp(:),dI_lab(:),dQ_lab(:),XX_lab(:)
      real*8, allocatable :: RE11(:,:,:,:), RE12(:,:,:,:),RE21(:,:,:,:),RE22(:,:,:,:)
      character(100), dimension(:) :: FF(13)
      character(100) :: filename


FF(1)='trunc_disc_IXPE.dat'
FF(2)='trunc_ssc_ixpe.dat'

! c#################################################
! c##### INPUT PARAMETERS #######
! c#################################################

      filename=FF(2)
      print *, 'filename=', filename

      TEkeV=1.0d2 !56.2d0!200.0d0 !56.2d0 ! c input electron temperature TE in keV 
! c define input spectrum and its angular distribution 	  
      TBBkeV=1.0d-1 !1.0d-1 ! c blackbody temperature in keV
      tau_T = 1.0d-0 !!! optical depth
      HR=1.0e5
      if (HR < 1.0e2) then 
        etacrit=HR/sqrt(HR**2+1.0) !1.0d0 !! cosine of maximal angle theta (\mu=\cos(pi/2-\theta))
        print *, 'etacrit = ', etacrit
      else
        etacrit=1.0
      endif  
      iinject=1    !!! injection of seed photons: 0-bottom, 1-middle layer (e.g. for truncated disc)
      itransm=0    !!! whether we account for photons transmitted through the medium: (0) no (1) yes 
      init_pol=0   !!! whether incident emission is non-polarized (init_pol=0) or Chandrasekhar-Sobolev profile (init_pol=1)
                   !!! note that polarization is currently only for incoming disc photons
      isc_min = 0  !!! start of the cycle from intensity (isc_min=1, for DC) or source function (isc_min=0, e.g. for SSC)
      iseed = 1    !!! seed photons from the disc (0) or synchrotron (1)
      ichan = 0    !!! isotropic injection (0) or Chandrasekhar-Sobolev e-scattering atm (1)
      isc_max = 20 !!! total number of scatterings


! c nodes and weight for angular integration [-1,1]
       CALL QDRGS(UMU_half,AMU_half,NMU/2)
       UMU_half(:) = (UMU_half(:)+1.0)/2.0
       AMU_half(:)=AMU_half(:)/2.0
       UMU(NMU/2+1:NMU)=UMU_half(1:NMU/2)
       AMU(NMU/2+1:NMU)=AMU_half(1:NMU/2)
       do ii=1,NMU/2
         UMU(ii)=-UMU_half(NMU/2+1-ii)
         AMU(ii)=AMU_half(NMU/2+1-ii)
       enddo
       do imu=1,NMU
         print *, 'mu=', UMU(imu), 'i=',acos(UMU(imu))*180/3.14159  !'delta tau/mu=' delta_tau/UMU(imu)
       enddo    

      Nt=int(tau_T/UMU(NMU/2+1))
      if(mod(Nt,2) == 0) Nt=Nt+1
      delta_tau = tau_T/(Nt-1)
      print*, 'tau_T', tau_T, 'UMU(NMU/2+1)',UMU(NMU/2+1),'Nt=', Nt
      print*, 'delta_tau=',delta_tau, 'Dtau/Dmu=', delta_tau/UMU(NMU/2+1)
      
      write(*,2010) TEkeV, tau_T
      write(*,2012) TBBkeV, etacrit
! c##### END PARAMETERS INPUT #######
! c#################################################
 
! c nodes and weight for energy integration 
       CALL AGUE(NL,EPS)
      TE=TEkeV/510.99d0      !!! changing units
      TBB=TBBkeV/510.99d0    !!!   to mc^2
! c electron temperature TE in unit mc^2 
      YE=1D0/TE
! c computation of auxiliary quantities (?common block WYY filling)
      CALL COMWY(EPS)
      

      allocate(sigma_cs(NX))
! c define vector of  photon energies in units me c**2 
! c minimum Log_10 of the energy grid 
      DLGXMIN=-6.5d0 
      DLGXMAX=0.3d0 
! c step in Log_10 in energy 
      DLGSTEP=(DLGXMAX-DLGXMIN)/DFLOAT(NX-1)    
      DLGX=DLGXMIN
      do ix=1,NX 
        XXLOG(ix)=DLGX 
        XX(ix)=10**DLGX
        DLGX=DLGX+DLGSTEP
        call CS(X,sigma_cs_var,EPS)
        sigma_cs(ix)=sigma_cs_var
      enddo


      
      if(iseed == 0) then   ! set planck spectrum
        do ix=1,NX 
          X=XX(ix)
          energy_input = PLANCK(X,TBB)    
          do imu=1, NMU
            eta=UMU(imu) 
            angle_input=angle_distrib(ichan,eta,etacrit)
            dinten(ix,imu)= energy_input*angle_input
          enddo
        enddo 
        print *, 'Input spectrum set (Planck)'
      elseif(iseed == 1) then    ! set synchrotron seed spectrum
        allocate(synch(NX))
        call synch_emiss(NX,XXLOG,synch)
        do ix=1,NX 
          X=XX(ix)
          energy_input = synch(ix)  
          do imu=1, NMU
            eta=UMU(imu) 
            angle_input=angle_distrib(ichan,eta,etacrit)
            dinten(ix,imu)= energy_input*angle_input
          enddo
        enddo 
        print *, 'Input spectrum set (synchrotron)'
        deallocate(synch)
      else
        print *, 'Undefined input seed spectrum, iseed should be 0 or 1'
        stop
      endif
!    
! c##################
! c##################
! c##################
! c##################

      allocate(SourI(Nt,NX,NMU),SourQ(Nt,NX,NMU),dI(Nt,NX,NMU),dQ(Nt,NX,NMU))
      allocate(dI_tot(Nt,NX,NMU),dQ_tot(Nt,NX,NMU))


! c######################################################
!!!########  Setting initial conditions in layers  ######
! c######################################################
      !!! setting intensity at the bottom as initial condition
      if ((isc_min == 1).and.(iinject == 0)) then 
        dI(:,:,:) = 0.0d0 ! Stokes I 
        dQ(:,:,:) = 0.0d0 ! Stokes Q
        SourI(:,:,:) = 0.0d0
        SourQ(:,:,:) = 0.0d0
        dI(1,:,NMU/2+1:NMU) = dinten(:,NMU/2+1:NMU)   ! lower boundary condition (Planck function from below)
        do it=2,Nt ! loop in layers of optical depth
          tau_down = delta_tau*(it-1)
          do ix=1,NX 
            X=XX(ix)
!             call CS(X,sigma_cs,EPS)
            do imu=NMU/2+1,NMU  !! mu>0
              eta=UMU(imu) 
              dI(it,ix,imu)=dI(1,ix,imu)*exp(-sigma_cs(ix)*tau_down/dabs(eta))
              if(init_pol==1) then
                dQ(it,ix,imu)=-0.1171d0*dI(it,ix,imu)*(1.0d0-eta)/(1.0+3.5*eta)
              endif
            enddo
          enddo
        enddo
        print *, 'Intensity initialised at the bottom, boundary conditions set'
      !!! setting intensity in the middle of the slab as initial condition
      elseif ((isc_min == 1).and.(iinject == 1)) then 
        dI(:,:,:) = 0.0d0 ! Stokes I 
        dQ(:,:,:) = 0.0d0 ! Stokes Q
        SourI(:,:,:) = 0.0d0
        SourQ(:,:,:) = 0.0d0
        itmid=(Nt+1)/2
        nmumid=NMU/2
        dI(itmid,:,:) = dinten(:,:)   ! lower boundary condition (Planck function from below)
        do it=1,(Nt-1)/2 ! loop in layers of optical depth, both up and down
          tau_layer = delta_tau*it
          do ix=1,NX 
            X=XX(ix)
!             call CS(X,sigma_cs,EPS)
            do imu=1,NMU/2  
              eta=UMU(imu) !! mu<0
              dI(itmid-it,ix,imu)=dI(itmid,ix,imu)*exp(-sigma_cs(ix)*tau_layer/dabs(eta))
              eta2=UMU(imu+nmumid) !! mu>0
              dI(itmid+it,ix,imu+nmumid)=dI(itmid,ix,imu+nmumid)*exp(-sigma_cs(ix)*tau_layer/dabs(eta2))
              if(init_pol==1) then
                dQ(itmid-it,ix,imu)=-0.1171d0*dI(itmid-it,ix,imu)*(1.0d0-eta)/(1.0+3.5*eta)
                dQ(itmid+it,ix,imu+nmumid)=-0.1171d0*dI(itmid+it,ix,imu+nmumid)*(1.0d0-eta2)/(1.0+3.5*eta2)
              endif
            enddo
          enddo
        enddo
        print *, 'Intensity initialised in the middle of the slab, boundary conditions set'
      !!! setting source function as initial condition
      else
        dI(:,:,:) = 0.0d0 ! Stokes I 
        dQ(:,:,:) = 0.0d0 ! Stokes Q
        SourI(:,:,:) = 0.0d0
        SourQ(:,:,:) = 0.0d0
        do it=2,Nt-1
          SourI(it,:,:) = dinten(:,:)
          if(init_pol==1) then
            SourQ(it,:,:)=0.1d0*SourI(it,:,:)  !!! Assuming source is 10% polarized
          endif
        enddo
        SourI(1,:,1:NMU/2) = dinten(:,1:NMU/2)   !!! setting source function at the outer boundaries (no external sources)
        SourI(Nt,:,NMU/2+1:NMU) = dinten(:,NMU/2+1:NMU)   !!! setting source function at the outer boundaries (no external sources)
        print *, 'Source function initialised, boundary conditions set'
      endif
!!!####################################


!!! Calculating scattering matrices
      allocate(RE11(NX,NMU,NX,NMU),RE12(NX,NMU,NX,NMU),RE21(NX,NMU,NX,NMU),RE22(NX,NMU,NX,NMU))
          do ix=1,NX 
            X=XX(ix)
            do imu=1,NMU 
              eta=UMU(imu)
              do jx=1,NX    
                X1=XX(jx)
                CALL COMX(X,X1)
                do jmu=1,NMU !!! integral over angles of incoming photons
                  eta1=UMU(jmu)
                  REsc11=0d0 
                  REsc12=0d0 
                  REsc21=0d0 
                  REsc22=0d0 
! c***********************************************************************
! C compute azimuthally-averaged matrix R11,R12,R21,R22 for I,Q Stokes parameters 
! c to compute source function S(x,eta) one should make a sum 
! c S(i,j)= hx*ln(10)* sum_ij R(i,j;i1,j1) * I(i1,j1) 
! c at use frequencies equally distant in LOGARITHM 
                  CALL MAXWRF(REsc11,REsc12,REsc21,REsc22,X,eta,X1,eta1) 
                  RE11(ix,imu,jx,jmu)=REsc11
                  RE12(ix,imu,jx,jmu)=REsc12
                  RE21(ix,imu,jx,jmu)=REsc21
                  RE22(ix,imu,jx,jmu)=REsc22
                enddo ! end loop dmu1
              enddo   ! end loop x1
            enddo     ! end look in dmu
          enddo       ! end look in x
!!!##################################


!!!########   Loop in Compton scattering orders  #####################
      if(itransm == 1) then       !!! this case corresponds to, e.g. slab geometry
        dI_tot(:,:,:)=dI(:,:,:)   !!! we account for all photons, also those transmitted through the slab at the zero scattering order
        dQ_tot(:,:,:)=dQ(:,:,:)
      else  
        dI_tot(:,:,:)=0.0    !!! for the cases imitating truncated disc 
        dQ_tot(:,:,:)=0.0    !!! we do not account for photons transmitted through the hot medium
      endif

      do isc=isc_min, isc_max 
        print *, 'scattering ', isc
        if (isc > 0) then   !!! calculating source function for scattering orders isc>0
          do ix=1,NX 
            X=XX(ix)
!             call CS(X,sigma_cs,EPS)
            do imu=1,NMU 
              eta=UMU(imu)
              do it=1,Nt ! loop in layers of optical depth
!#################   Update source function #######################      
! c integrating over incident photon energies and angles 
! c loop over incident photon energies 
                sumI=0d0 
                sumQ=0d0  
                do jx=1,NX    
                  X1=XX(jx)
                  do jmu=1,NMU !!! integral over angles of incoming photons
                    eta1=UMU(jmu)
! ! c***********************************************************************
! ! C compute azimuthally-averaged matrix R11,R12,R21,R22 for I,Q Stokes parameters 
! ! c to compute source function S(x,eta) one should make a sum 
! ! c S(i,j)= hx*ln(10)* sum_ij R(i,j;i1,j1) * I(i1,j1) 
! ! c at use frequencies equally distant in LOGARITHM 
! ! convolve with seed photon Stokes vector to get source vector Stokes (I,Q)
! ! (SI,SQ)=  \int d\ln x1 \int_-1^1 d\eta1 [x^2/x1 *(R11,R12/R21,R22)] * (I,Q=0)_incident (x1,eta1)
                     sumI=sumI + AMU(jmu)*(RE11(ix,imu,jx,jmu)*dI(it,jx,jmu) + &
                                           RE12(ix,imu,jx,jmu)*dQ(it,jx,jmu))    
                     sumQ=sumQ + AMU(jmu)*(RE21(ix,imu,jx,jmu)*dI(it,jx,jmu) + &
                                           RE22(ix,imu,jx,jmu)*dQ(it,jx,jmu))  
                  ENDDO ! end loop dmu1
                ENDDO   ! end loop x1
                SourI(it,ix,imu) = DLGSTEP*DLOG(1d1)*sumI
                SourQ(it,ix,imu) = DLGSTEP*DLOG(1d1)*sumQ
              ENDDO  ! end loop tau
            ENDDO !  end loop dmu 
          enddo ! end loop in x
        endif
!#################  End of source function update #######################      
           
         
!#################   Update intensity #######################
        dI(:,:,:)=0.0d0
        dQ(:,:,:)=0.0d0
        do ix=1,NX 
          X=XX(ix)
          do imu=1,NMU/2 !!! mu<0, i.e. sum over sources coming from layers above 
!!! Auxiliary quantities for intensity calculation (see Suleimanov et al. 2012, Appendix B)
            eta=UMU(imu)
            Dtau = sigma_cs(ix)*delta_tau/dabs(eta)
            exp_sigmataumu = exp(-Dtau)
            e0 = 1.0d0 - exp_sigmataumu
            e1 = Dtau - e0
            e2 = Dtau**2 - 2.0d0*e1
            alpha = e0 + (e2 - 3.0d0*Dtau*e1)/(2.0d0*Dtau**2)
            beta = (2.0d0*Dtau*e1 - e2)/Dtau**2
            gamma = (e2 - Dtau*e1)/(2.0d0*Dtau**2)
          
            do it=Nt-1,2,-1 ! loop in layers of optical depth
              dI(it,ix,imu)=dI(it+1,ix,imu)*exp_sigmataumu + alpha*SourI(it+1,ix,imu) + &
                            beta*SourI(it,ix,imu) + gamma*SourI(it-1,ix,imu)
              dQ(it,ix,imu)=dQ(it+1,ix,imu)*exp_sigmataumu + alpha*SourQ(it+1,ix,imu) + &
                            beta*SourQ(it,ix,imu) + gamma*SourQ(it-1,ix,imu)
            enddo  ! end loop tau
            beta=e1/Dtau
            alpha=e0-beta
            dI(1,ix,imu)=dI(2,ix,imu)*exp_sigmataumu + alpha*SourI(2,ix,imu) + beta*SourI(1,ix,imu)
            dQ(1,ix,imu)=dQ(2,ix,imu)*exp_sigmataumu + alpha*SourQ(2,ix,imu) + beta*SourQ(1,ix,imu)
          enddo   ! end loop mu
          
          do imu=NMU/2+1,NMU !!! mu>0
!!! Auxiliary quantities for intensity calculation (see Suleimanov et al. 2012, Appendix B)
            eta=UMU(imu)
            Dtau = sigma_cs(ix)*delta_tau/dabs(eta)
            exp_sigmataumu = exp(-Dtau)
            e0 = 1.0d0 - exp_sigmataumu
            e1 = Dtau - e0
            e2 = Dtau**2 - 2.0d0*e1
            alpha = (e2 - Dtau*e1)/(2.0d0*Dtau**2)
            beta = (2.0d0*Dtau*e1 - e2)/(Dtau**2)
            gamma = e0 + (e2 - 3.0d0*Dtau*e1)/(2.0d0*Dtau**2)
          
            do it=2,Nt-1,1 ! loop in layers of optical depth; can be done in vector form
              dI(it,ix,imu)=dI(it-1,ix,imu)*exp_sigmataumu + alpha*SourI(it+1,ix,imu) + &
                            beta*SourI(it,ix,imu) + gamma*SourI(it-1,ix,imu)
              dQ(it,ix,imu)=dQ(it-1,ix,imu)*exp_sigmataumu + alpha*SourQ(it+1,ix,imu) + &
                            beta*SourQ(it,ix,imu) + gamma*SourQ(it-1,ix,imu)
            enddo ! end loop tau
            beta=e1/Dtau
            gamma=e0-beta
            dI(Nt,ix,imu)=dI(Nt-1,ix,imu)*exp_sigmataumu + beta*SourI(Nt,ix,imu) + gamma*SourI(Nt-1,ix,imu) !!! Emerging intensity
            dQ(Nt,ix,imu)=dQ(Nt-1,ix,imu)*exp_sigmataumu + beta*SourQ(Nt,ix,imu) + gamma*SourQ(Nt-1,ix,imu) !!! Emerging intensity
!             if (dI(Nt,ix,imu) < 0.0) then
!               print *, ix, imu, dI(Nt,ix,imu)
!             endif  
          enddo  ! end loop mu
!#################   End of intensity update #######################                        
        ENDDO  ! end loop x
        dI_tot(:,:,:)=dI_tot(:,:,:) + dI(:,:,:)
        dQ_tot(:,:,:)=dQ_tot(:,:,:) + dQ(:,:,:)
      ENDDO   !  end loop in scattering orders

      deallocate(RE11,RE12,RE21,RE22)


! c########  Printouts  ##########
      OPEN(NOUTPUTINTPOL,file='compton_intpol_' // filename)            

 2010 FORMAT(' kTe = ',F7.2,' keV;  Thomson optical depth tau = ',F7.2,' Outflow \beta=',F7.2)
 1000 FORMAT(' Energy and angle bins')
 2012 FORMAT(' kTseed = ',E11.3,' keV;  maximal angle of seed injection etacrit=',E11.3)
      write(NOUTPUTINTPOL,2010) TEkeV, tau_T
      write(NOUTPUTINTPOL,2012) TBBkeV, etacrit
      write(NOUTPUTINTPOL,*) 'Non-polarized (0) or Cha-Sob pol profile (1) of incident emission', init_pol, &
                             'Isotropic (0) or e-scattering (1) angular distribution', ichan
      write(NOUTPUTINTPOL,*) 'Number of layers in slab', Nt
      write(NOUTPUTINTPOL,*) 'Number of scatterings', isc_max
      write(NOUTPUTINTPOL,*) 'Input as incoming intensity (1) or source (0)', isc_min
      write(NOUTPUTINTPOL,1000) 
      write(NOUTPUTINTPOL,*) NX,NMU 
      write(NOUTPUTINTPOL,*) 'cos(i)'
      write(NOUTPUTINTPOL,*) (UMU(imu), imu=NMU/2+1, NMU)



      imu_out=NMU/2+1!NMU/2+1 ! observer's inclination 
      eta=UMU(imu_out)
      print *, 'cosine of observers inclination = ', UMU(imu_out)
       write (NOUTPUTINTPOL,1038)
1038  FORMAT('Energy  Q(cos(i))     I(cos(i))    Planck')
      do ix=1,NX
        write(NOUTPUTINTPOL,*) XX(ix)*510.99d0, & 
        (dQ_tot(Nt,ix,jmu), jmu=NMU/2+1, NMU), &
        (dI_tot(Nt,ix,jmu), jmu=NMU/2+1, NMU), dinten(ix,imu_out)
      enddo
1036  FORMAT(8E11.3)
1037  FORMAT('# Ener    Planck     I_esc    Q_esc     Source_last_sca  I_direct')
      
      deallocate(SourI,SourQ,dI,dQ,dI_tot,dQ_tot)
      deallocate(sigma_cs)
 
      close(NOUTPUT); close(NOUTPUTTEST); close(NOUTPUTPOL)
      END 
 
! c***********************************************************************                              
! c black body radiation 
! c Planck formula with arbitrary normalization 
! c      
      FUNCTION PLANCK(X,TBB) 
      IMPLICIT REAL*8(A-H,O-Z)
      XT=X/TBB                                     
      EBB=0D0                                                                                   
      IF(XT.LT.5D1) EBB=DEXP(-XT)                       
      PLANCK=XT**3*EBB/(1D0-EBB)                            
                                     
      RETURN
      END 
  
! c***********************************************************************                              
! c synchrotron radiation 
! c from the pre-computed file
! c      
      Subroutine synch_emiss(nx,lnx,synch) 
      implicit real*8(a-h,o-z)
      character(LEN=10000) :: txt
      real*8, dimension(nx), intent(inout) :: synch
      real*8, dimension(nx), intent(in) :: lnx
      real*8, dimension(:), allocatable :: lnx_old, synch_old      
      real*8, dimension(:), allocatable :: lnx_h, lnx_old_h
      real*8 :: d_lnx_old, d_lnx, Csum, dd, du, lnxi
      integer*4, dimension(:), allocatable :: ind
!       integer*4 :: np, nn, in, nmax, nmin, nph, i

      eps=1e-35
      np = 1000
      nn = nx
      allocate(synch_old(np),lnx_old(np))


	  open(122,file='emissivity_cygx-1.dat',action='READ')
!       open(122,file='tot_input.dat',action='READ')
    
      do ip=0,3 
        read(122,*) txt; read(122,*) txt; read(122,*) txt
        read(122,*) comp,dlum,constlum 
        print *,'synchrotron: comp,dlum,constlum=',comp,dlum,constlum 
        do i=1,np
          read(122,*) x_old, emis, tau, synch_old_tmp
          synch_old(i)=synch_old_tmp
          lnx_old(i) = dlog10(x_old)
!           if (ip == 3) print*, lnx_old(i), synch_old(i)
        enddo
      enddo  
      close(122)
! stop
      d_lnx_old = (lnx_old(np) - lnx_old(1))/(np - 1)
      d_lnx = (lnx(nx) - lnx(1))/(nx - 1)

      if (d_lnx.lt.(2.0*d_lnx_old)) then
      print *,'ReBin: the resolution of the original grid is too small. d_lnx_old, d_lnx=', d_lnx_old, d_lnx
      end if

      if (size(synch_old).ne.np.or.size(synch).ne.nn) then
        print *,'ReBin: The sizes of the arrays do not match'
        print *,'size(lnx_old,synch_old,lnx,synch)', size(lnx_old), size(synch_old), size(lnx), size(synch)
        stop
      end if

      allocate(lnx_h(nn+1))
      lnx_h(1:nn) = lnx(1:nn) - 0.5*d_lnx
      lnx_h(nn+1) = lnx(nn) + 0.5*d_lnx
      allocate(lnx_old_h(np+1))
      lnx_old_h(1:np) = lnx_old(1:np) - 0.5*d_lnx_old
      lnx_old_h(np+1) = lnx_old(np) + 0.5*d_lnx_old

      allocate(ind(nn+1))
      ind = 0
      do in = 1,nn+1				!!! lnx_neu
        lnxi = lnx_h(in)
        if (lnxi.lt.lnx_old_h(1)) then
          ind(in) = 0
        else if (lnxi.gt.lnx_old_h(np)) then
          ind(in) = -1
        else if (lnxi.eq.lnx_old_h(1)) then
          ind(in) = 1
        else if (lnxi.eq.lnx_old_h(np+1)) then
          ind(in) = -1
        else
          nmax = np+1
          nmin = 0
          do while ((nmax-nmin).gt.1)
            nph = (nmin + nmax)/2
            if (lnxi.gt.lnx_old_h(nph)) then
              nmin = nph
            else if (lnxi.lt.lnx_old_h(nph)) then
	          nmax = nph
            else if (lnxi.eq.lnx_old_h(nph)) then
	          nmin = nph
	          exit
            end if
          end do
          ind(in) = nmin
        end if
      end do

      do in = 2,nn+1
        if ((ind(in).lt.ind(in-1)).and.(ind(in).ne.-1).and.(ind(in-1).ne.0)) then						!!! Modified on 01.04.08
          print *,'ReBin: error in determining indices. in, ind(in-1),ind(in)', in, ind(in-1), ind(in)
          stop
        end if
        Csum = 0.0			!!! For synch(in-1)
        if (ind(in).gt.0) then
          if (ind(in-1).ne.0) then
            if ((ind(in)-ind(in-1)).gt.1) then
              do i = ind(in-1)+1,ind(in)-1
                Csum = Csum + synch_old(i)*d_lnx_old
	          end do
            end if
	        du = lnx_h(in) - lnx_old_h(ind(in))
            dd = lnx_old_h(ind(in-1)+1) - lnx_h(in-1)
            Csum = Csum + synch_old(ind(in))*du + synch_old(ind(in-1))*dd
          else if (ind(in-1).eq.0) then
            if (ind(in).gt.1) then
	          do i = 1,ind(in)-1								!!! Implicitly assumes that synch = 0 outside the old grid
                Csum = Csum + synch_old(i)*d_lnx_old
              end do
            end if
	        du = lnx_h(in) - lnx_old_h(ind(in))
	        Csum = Csum + synch_old(ind(in))*du
          end if
        else if ((ind(in).eq.-1).and.(ind(in-1).gt.0)) then
          if (ind(in-1).lt.np) then
	        do i = ind(in-1)+1,np								!!! Implicitly assumes that synch = 0 outside the old grid
	          Csum = Csum + synch_old(i)*d_lnx_old
            end do
          end if
          dd = lnx_old_h(ind(in-1)+1) - lnx_h(in-1)
	      Csum = Csum + synch_old(ind(in-1))*dd
        else
          Csum = 0.0
        end if
        synch(in-1) = Csum/d_lnx
      end do

      deallocate(lnx_h,lnx_old_h,ind)
      deallocate(synch_old,lnx_old)
      
      synch=(synch+eps)!*constlum
     End Subroutine synch_emiss
  


! c angular distribution of incoming intensity     
! c it is 1 for cosine of zenith angle close to 0 (|eta|<etacrit)
! c and zero otherwise    
      FUNCTION angle_distrib(ichan,eta,etacrit) 
      real*8 angle_distrib,eta,etacrit
      integer :: ichan
 
      angle_distrib=0d0                                
      if(ichan == 1) then
        if(etacrit == 0.0d0) then
          print *, 'etacrit = 0'
          stop
        endif  
        if(dabs(eta) .lt. etacrit) angle_distrib=(1d0+2.06*eta)/(etacrit**2/2.0+2.06*etacrit**3/3.0)       
      else
        if(dabs(eta) .lt. etacrit) angle_distrib=2.0d0/(etacrit**2)             
      endif                                                                   
      RETURN
      END 


subroutine interp_linear ( m, data_num, t_data, p_data, interp_num, &
  t_interp, p_interp )
!*****************************************************************************80
!
!! INTERP_LINEAR: piecewise linear interpolation to a curve in M dimensions.
!
!  Discussion:
!
!    From a space of M dimensions, we are given a sequence of
!    DATA_NUM points, which are presumed to be successive samples
!    from a curve of points P.
!
!    We are also given a parameterization of this data, that is,
!    an associated sequence of DATA_NUM values of a variable T.
!    The values of T are assumed to be strictly increasing.
!
!    Thus, we have a sequence of values P(T), where T is a scalar,
!    and each value of P is of dimension M.
!
!    We are then given INTERP_NUM values of T, for which values P
!    are to be produced, by linear interpolation of the data we are given.
!
!    Note that the user may request extrapolation.  This occurs whenever
!    a T_INTERP value is less than the minimum T_DATA or greater than the
!    maximum T_DATA.  In that case, linear extrapolation is used.
!
!  Licensing:
!
!    This code is distributed under the GNU LGPL license.
!
!  Modified:
!
!    03 December 2007
!
!  Author:
!
!    John Burkardt
!
!  Parameters:
!
!    Input, integer M, the spatial dimension.
!
!    Input, integer DATA_NUM, the number of data points.
!
!    Input, real ( kind = rk ) T_DATA(DATA_NUM), the value of the
!    independent variable at the sample points.  The values of T_DATA
!    must be strictly increasing.
!
!    Input, real ( kind = rk ) P_DATA(M,DATA_NUM), the value of the
!    dependent variables at the sample points.
!
!    Input, integer INTERP_NUM, the number of points
!    at which interpolation is to be done.
!
!    Input, real ( kind = rk ) T_INTERP(INTERP_NUM), the value of the
!    independent variable at the interpolation points.
!
!    Output, real ( kind = rk ) P_INTERP(M,DATA_NUM), the interpolated
!    values of the dependent variables at the interpolation points.
!
  implicit none

  integer data_num
  integer m
  integer interp_num

  integer interp
  integer left
  real*8 :: p_data(m,data_num)
  real*8 :: p_interp(m,interp_num)
!   logical :: r8vec_ascends_strictly
  integer :: right
  real*8 :: t
  real*8 :: t_data(data_num)
  real*8 :: t_interp(interp_num)

  do interp = 1, interp_num

    t = t_interp(interp)
!
!  Find the interval [ TDATA(LEFT), TDATA(RIGHT) ] that contains, or is
!  nearest to, TVAL.
!
    call r8vec_bracket ( data_num, t_data, t, left, right )

    p_interp(1:m,interp) = &
      ( ( t_data(right) - t                ) * p_data(1:m,left)   &
      + (                 t - t_data(left) ) * p_data(1:m,right) ) &
      / ( t_data(right)     - t_data(left) )

  end do

  return
end


subroutine r8vec_bracket ( n, x, xval, left, right )

!*****************************************************************************80
!
!! R8VEC_BRACKET searches a sorted R8VEC for successive brackets of a value.
!
!  Discussion:
!
!    An R8VEC is an array of double precision real values.
!
!    If the values in the vector are thought of as defining intervals
!    on the real line, then this routine searches for the interval
!    nearest to or containing the given value.
!
!  Licensing:
!
!    This code is distributed under the GNU LGPL license.
!
!  Modified:
!
!    06 April 1999
!
!  Author:
!
!    John Burkardt
!
!  Parameters:
!
!    Input, integer N, length of input array.
!
!    Input, real ( kind = rk ) X(N), an array sorted into ascending order.
!
!    Input, real ( kind = rk ) XVAL, a value to be bracketed.
!
!    Output, integer LEFT, RIGHT, the results of the search.
!    Either:
!      XVAL < X(1), when LEFT = 1, RIGHT = 2;
!      X(N) < XVAL, when LEFT = N-1, RIGHT = N;
!    or
!      X(LEFT) <= XVAL <= X(RIGHT).
!
  implicit none


  integer :: n

  integer i
  integer left
  integer right
  real*8 :: x(n)
  real*8 :: xval

  do i = 2, n - 1

    if ( xval < x(i) ) then
      left = i - 1
      right = i
      return
    end if

   end do

  left = n - 1
  right = n

  return
end


! c***********************************************************************
! C CALCULATIONS OF ANGLE-DEPENDENT REDISTRIBUTION FUNCTIONS  for Maxwellian electrons 
! C averaged over azimuthal angle PHI in the range (0,2*PI) using NS number of points 
! c X1,ETA1 - input frequency and angle 
! c X,ETA   - output frequency and angle 
! c-------------------------------------------------------------------------
! c they are multiplied by FACTOR X^2 /X1 
! C IT IS ONLY (1/X') INSTEAD OF (1/X'**2) BECAUSE OF  dLOG_10 FREQUENCY INTEGRATION SCALE     
! c***********************************************************************
      SUBROUTINE MAXWRF(RE11,RE12,RE21,RE22,X,ETA,X1,ETA1) 
      PARAMETER(NSA=11)
      IMPLICIT REAL*8(A-H,O-Z)
!      COMMON /WYY/ YE,Y1,YR1,YR2,DY,YKY,DK02,DK2Y
      DATA PI/3.141592653589792384D0/,EPS/1D-14/ 
      DATA C13/0.33333333333333333D0/
!      DATA NOUTPUT/20/   
! c-------------------------------------------------------------------------
      NS=NSA
      IF(2*(NS/2).EQ.NS) NS=NS+1 
! c      WRITE(IDEVC,6600) NS   
! c 6600 FORMAT('NS=',I3,' IT IS EVEN: IT IS NOW CHANGED TO ODD NS+1') 
      DLN=DLOG(1D1)                                          
      NS1=NS-1
      DNS=DFLOAT(NS1) 
! c HPHI IS THE STEP IN THE AZIMUTHAL ANGLE INTEGRATION 
! c integral over phi from 0,2\pi => 2*integral_0^pi dphi 
      HPHI=PI/DNS 
      H3=HPHI*C13*2D0*X*X/X1 
! c------------------------------------------------------------------
 
! c             write(NOUTPUT,1000) ETA,ETA1 
! c 1000    format('ETA,ETA1=',6E11.3)   
         
              SUM11=0D0 
              SUM12=0D0 
              SUM21=0D0 
              SUM22=0D0 
! c azimuth INTEGRATION LOOP using Simpson quadrature (weights coded in NA)
                 PHI=0D0
                 NSIGN=-1
                 DO 50 ID=1,NS
                 NA=NSIGN+3 
                 IF(ID.eq.1.or.ID.eq.NS) NA=1 

! c             write(NOUTPUT,1010) ID,PHI
! c 1010    format('ID,PHI=',I5,E13.5)      
             
! C COSINES  OF THE ROTATION ANGLES   
       CALL PHICOSNEW(ETA,ETA1,PHI,QMU,PPCOS0,PPCOS1,PPCOSM,PPCOSP) 
! c        write(NOUTPUT,1022) PPCOS0,PPCOS1,PPCOSM,PPCOSP
! c 1022    format('PPCOS0,PPCOS1,PPCOSM,PPCOSP AFTER=',6E11.3)    
! c CALCULATION OF S,S_I,S_Q,S_U,S_V, SF=S_Q-S_U 
       CALL REDFUNC(X,X1,QMU,SP,SIP,SQP,SUP,SVP,SFP) 
! c             write(NOUTPUT,1030) X,X1,QMU
! c 1030    format('X,X1,QMU=',9E11.3)  
! c             write(NOUTPUT,1040) SP,SIP,SQP,SUP,SVP,SFP
! c 1040    format('SP,SIP,SQP,SUP,SVP,SFP=',9E11.3)     
! c eq A5 in PS96 
                 SUM11=SUM11+NA*SP
                 SUM12=SUM12+NA*SIP*PPCOS1
                 SUM21=SUM21+NA*SIP*PPCOS0
                 SUM22=SUM22+NA*((SQP+SUP)*PPCOSM+SFP*PPCOSP)*5D-1
                 NSIGN=-NSIGN
                 PHI=PHI+HPHI 
                           
   50 CONTINUE              
! c END  azimuth INTEGRATION LOOP 
! c--------------------------------------------------------------------------
      RE11=H3*SUM11
      RE12=H3*SUM12
      RE21=H3*SUM21
      RE22=H3*SUM22 

      RETURN
      END


! C
! C COSINES  OF THE ROTATION ANGLES 
! C INPUT: 
! C  1. ETA  - COSINE  OF THE ANGLE BETWEEN SCATTERED 
! C     PHOTON AND FIXED DIRECTION
! C  2. ETA1 - COSINE AND SINE OF THE ANGLE BETWEEN INITIAL  
! C     PHOTON AND FIXED DIRECTION;  SI12=SI1**2
! C  3. PHI - azimuthal angle phi_1-phi 
! 
! C OUTPUT: 
! C  QMUMI1= 1 - DMU; where DMU=MU=ETA*ETA1+SI*SI1*CFI - COSINE OF SCATTERING ANGLE; 
! C  1. CO:  COS 2 CHI;       
! C  2. CO1: COS 2 CHI_1; 
! C  3. COM: COS 2(CHI-CHI_1); 
! C  4. COP: COS 2 (CHI+CHI_1);  
! C  
      SUBROUTINE PHICOSNEW(ETA,ETA1,PHI,QMUMI1,CO,CO1,COM,COP)  
      IMPLICIT REAL*8(A-H,O-Z)   
      DATA PI/3.141592653589792384D0/,EPS/1D-5/ 
      DATA NOUTPUT/20/   
      
      
! c eta^2
            ETA2=ETA*ETA
! c sin^2(theta)
            SI2=1D0-ETA2
! c sin(theta)
            SI=DSQRT(SI2) 
! c eta_1^2
            ETA12=ETA1*ETA1
! c sin^2(theta_1)
            SI12=1D0-ETA12
! c sin(theta_1)
            SI1=DSQRT(SI12)  
! c sin(theta)*sin(theta_1)
            SIKL=SI*SI1
! c cos(phi) 
             CFI=DCOS(PHI)
! c cos^2(phi) 
             CF2=CFI*CFI       
! c  sin^2(phi)                                 
             SIF2=1D0-CF2

! c mu - cos of the scattering angle 
                 DMU=ETA*ETA1+SIKL*CFI
! c 1+mu
                 QMUPL1=1D0+DMU
! c 1-mu
                 QMUMI1=1D0-DMU
! c (1-mu)*(1+mu)  in PS96  sin^2 theta
                 QMPRO=QMUPL1*QMUMI1
! c        write(NOUTPUT,1022) ETA,ETA1,PHI,DPHI,DABS(PHI-PI)
! c 1022    format('ETA,ETA1,PHI,DPHI,DABS(PHI-PI)=',6E11.3)   
 
 
! c special cases of forward or backscattering 
! c phi=0 & eta=eta1 or eta=-eta1 & phi=pi
    IF((DABS(ETA-ETA1).lt.EPS.and.DABS(PHI).lt.EPS).or.(DABS(ETA+ETA1).lt.EPS.and.DABS(PHI-PI).lt.EPS)) THEN 
                   CO=-1D0 
                   CO1=-1D0 
                   COM=1D0
                   COP=1D0   

! c        write(NOUTPUT,1022) CO,CO1,COM,COP
! c 1022    format('CO,CO1,COM,COP 1=',6E11.3)   
 
         ELSE 

! C cos 2 CHI  = cos^2 chi - sin^2 chi   
      CO=((ETA1-ETA*DMU)**2-SI12*SI2*SIF2)/(SI2*QMPRO)  
! C cos 2 CHI_1
      CO1=((ETA-ETA1*DMU)**2-SI12*SI2*SIF2)/(SI12*QMPRO) 
! C cos 2(CHI-CHI_1)
      COM=(((1D0+ETA*ETA1)*CFI+SIKL)**2-(ETA+ETA1)**2*SIF2)/(QMUPL1*QMUPL1)
! C cos 2 (CHI+CHI_1) 
      COP=(((-1D0+ETA*ETA1)*CFI+SIKL)**2-(ETA-ETA1)**2*SIF2)/(QMUMI1*QMUMI1) 
  
! c           write(NOUTPUT,1024) CO,CO1,COM,COP
! c 1024    format('CO,CO1,COM,COP 2=',6E11.3)                       
          ENDIF
                 
      RETURN                                               
      END      
      
      SUBROUTINE COMWY(EPS) 
! C*** CALCULATES SOME FUNCTION OF ELECTRON TEMPERATURE. 
! C* * INPUT: 
! C* * 1. YE - INVERSE TEMPERATURE (m_c^2/kT_e) .
! C* * OUTPUT: 
! C* * 2. YR= 1/Y, YR1=2/Y, YR2=2/(Y*Y), DY=1./SQRT(Y)  
! C* * 3. YKY=EXP(-Y)/K_2(Y) 
! C* * 4. DK2Y=YKY/(2.*SQRT(Y))
! C* * 5. DK02=K_0(Y)/K_2(Y) 
! C* * 6. DK12=K_1(Y)/K_2(Y) 
! C* * 7. GAMMEAN=3D0/Y + K_1(Y)/K_2(Y) 
! C* * 8. GAMSQUA=4D0/Y*GAMMEAN +1/2*(1+ K_0(Y)/K_2(Y))
! C***
      IMPLICIT REAL*8 (A-H,O-Z)                                       
      COMMON /WYY/ YE,Y1,YR1,YR2,DY,YKY,DK02,DK2Y
      DIMENSION DK(20)                                              
      DATA PI/3.141592653589792384D0/,C13/0.33333333333333333D0/
! C    PIQR=1./SQRT(2.*PI)
      DATA PIQR/0.39894228040143267794D0/
      PI2=2D0/PI                
                                         
      CALL DBESK(YE,4,DK,8D0,6)                                      
      DK2=DK(3) 
      Y1=1D0/YE     
      G0=-Y1*DLOG(EPS)
      IF(YE.LT.1D0) G0=-Y1*DLOG(EPS*G0)
      DK02=DK(1)/DK2  
      DK12=DK(2)/DK2                              
      GAMMEAN=3D0*Y1 + DK12
      GAMSQUA=1D0+3D0/YE*DK(4)/DK2                                 
      YR1=2D0*Y1             
      YR2=YR1*Y1                   
      DY=DSQRT(Y1)           
      DK2R=1D0/DK2   
                       
      IF(YE.GT.8D0) THEN 
      YPI=DSQRT(PI2*YE)        
      YKY=YPI/DK2           
      DK2Y=PIQR*DK2R          
      ELSE 
      EDY=DEXP(-YE)            
      YKY=EDY/DK2      
      DK2Y=0.5D0*EDY*DY*DK2R  
      ENDIF 
      RETURN   
      END      


! C
! C AUXILARY QUANTITIES DEPENDING ON X,X1 AND MU ; 
! C NEEDED TO CALL FIRST  COMX(X,X1) SUBROUTINE 
! C 
      SUBROUTINE COMXM(QM)                                 
      IMPLICIT REAL*8(A-H,O-Z)
      COMMON /WX/ XX,XX1,SX,DX,T,TR,XM2,DX2,GM1,GA1,RB,RWB,NR
      COMMON /WXM/ GST,GST1,QO,QR,QOT,Q2,Q,QT,QMT,PM,RS,QS
      DATA EPS/1D-10/
      QO=T*QM
      Q2=DX2+2D0*QO
      Q=DSQRT(Q2)
      IF(QM.gt.EPS) THEN
      QR=1D0/QO                       
      QOT=2D0*QR                      
      QT=2D0/Q                        
      QMT=2D0/QM                      
      PM=2D0-QM                       
      RS=PM/QM                        
      QS=T*PM                         
      GST=(DX+Q*DSQRT(1D0+QOT))*5D-1
      GST1=GST-1D0
      ELSE
      GST=1D0
      GST1=0D0
      QT=0D0
      ENDIF
      RETURN                                          
      END                           


! C 
! C AUXILARY QUANTITIES DEPENDING ON GAMMA 
! C 
      SUBROUTINE COMWG            
      IMPLICIT REAL*8(A-H,O-Z)     
      COMMON /WZ/ Z,Z2,G,G2,GT,G1  
      G1=G-1D0                     
      G2=G*G                       
      GT=2D0*G                     
      Z2=G1*(G1+2D0)               
      Z=DSQRT(Z2)                  
      RETURN                       
      END                          


! C 
! C AUXILARY QUANTITIES DEPENDING ON  X AND X1
! C 
      SUBROUTINE COMX(X,X1)                                 
      IMPLICIT REAL*8(A-H,O-Z)                              
      COMMON /WX/XX,XX1,SX,DX,T,TR,XM2,DX2,GM1,GA1,RB,RWB,NR
      COMMON /WFT/TRT,TRF,TRR,TRS,TR2,XS,XS1,XA,XA1,XXA,XXA1
      COMMON /WFW/TRO,TT,TOO,TTT                            
      DATA C13/0.33333333333333D0/,C23/0.66666666666666D0/  
      XS=1D0+2D0*X                                          
      XA=X/XS                                               
      XS1=1D0+2D0*X1                                        
      XA1=X1/XS1                                            
      XX=2D0*XA*X     
      XX1=-2D0*XA1*X1 
      SX=X+X1         
      DX=X-X1         
      T=X*X1          
      TT=T*T          
      TR=1D0/T        
      XM2=SX*DX       
      DX2=DX*DX       
      TR2=TR*TR       
      TRT=-2D0*TR     
      TRF=4D0*TR      
      TRR=TR+1D0      
      TRS=5D-1*TR2                    
      TTT=DX2*TR2-TRF                 
      TOO=1D0-TRT                     
      XMAX=X                          
      IF(X.LT.X1) XMAX=X1             
      XMIN=X                          
      IF(X.GT.X1) XMIN=X1             
      RB=4D0/XMAX                     
      RWB=RB-4D0*C13*XMIN/(XMAX*XMAX) 
      GA=0.5D0*(DX+SX*DSQRT(TRR))     
      XXA=X-XA1                       
      XXA1=X1-XA                      
      XXX=(1D0-2D0*X)/XS              
      GA1=GA-1D0                      
      IF(XXA.GT.-0.1D0)      GA1=0.25D0*XS1*XS1*TR*XXA*XXA/(GA+1D0-DX)  
      IF(XXA1.GT.-0.1D0*XXX) GA1=0.25D0*XS*XS*TR*XXA1*XXA1/(GA+1D0)+DX  
      IF(XXA.LE.0D0) NR=1                 
      IF(XXA1.LE.0D0) NR=2                
      IF((XXA.GT.0D0).AND.(X.LE.X1)) NR=3 
      IF((XXA1.GT.0D0).AND.(X1.LT.X)) NR=4
      IF(NR-3) 10,20,30                   
   10 GM1=GA1                             
      GO TO 40                            
   20 CONTINUE                            
      GM1=0D0                             
      GO TO 40                           
   30 CONTINUE                             
      GM1=DX                             
   40 CONTINUE                           
      RETURN                             
! C     DEBUG INIT(GM1,GA1,T,TR)           
      END                                


! C 
! C REDISTRIBUTION MATRIX : S(X,X1,MU)
! c for Maxwellian electron distribution 
! C INPUT: 
! C 1. X, X1 - ENERGIES OF SCATTERED AND  INITIAL PHOTONS, RESPECTIVELY; 
! C 2. QM=1-MU , AND MU - THE COSINE OF THE SCATTERING ANGLE; 
! C OUTPUT: 
! C 3. S, SI, SQ, SU, SV, SA=SQ-SU  (X,X1,MU) 
! C 
      SUBROUTINE REDFUNC(X,X1,QM,S,SI,SQ,SU,SV,SA) 
      IMPLICIT REAL*8(A-H,O-Z)                                    
      COMMON /WXM/ GST,GST1,QO,QR,QOT,Q2,Q,QT,QMT,PM,RS,QS 
      COMMON /WX/ XX,XX1,SX,DX,T,TR,XM2,DX2,GM1,GA1,RB,RWB,NR
      COMMON /WYY/ YE,Y1,YR1,YR2,DY,YKY,DK02,DK2Y
      COMMON /WZ/ Z,Z2,G,G2,GT,G1   
! C PI332=3/(32*PI)      
      DATA PI332/0.2984155182973039D-01/,EPS/1D-10/  


      CALL COMXM(QM) 
            EXGST=0D0                                            
            YGST=YE*GST1                                    
            IF(YGST.LT.7D1) EXGST=DEXP(-YGST)     
            CONST=PI332*YKY*EXGST
            CALL MAXGALA(X,X1,QM,S,SI,SQ,SU,SV,SA)
            S=S*CONST    
            SI=SI*CONST        
            SQ=SQ*CONST    
            SU=SU*CONST    
            SV=SV*CONST
            SA=SA*CONST    
      RETURN                                               
      END                                                  


! C 
! C SCATTERING MATRIX FOR MAXWELLIAN ELECTRONS 
! C 
      SUBROUTINE MAXGALA(X,X1,QM,S,SI,SQ,SU,SV,SA) 
      IMPLICIT REAL*8(A-H,O-Z)                                    
      COMMON /W/ UL(40),AL(40),AU(40),AE(40),N0        
      COMMON /WYY/ YE,Y1,YR1,YR2,DY,YKY,DK02,DK2Y
      COMMON /WXM/ GST,GST1,QO,QR,QOT,Q2,Q,QT,QMT,PM,RS,QS
      COMMON /WZ/ Z,Z2,G,G2,GT,G1   
            S=0D0
            SI=0D0
            SQ=0D0
            SU=0D0
            SV=0D0                                            
            SA=0D0                                            
               DO 10 I=1,N0
               DGZ=UL(I)*Y1 
               G=DGZ+GST
               CALL COMWG
               CALL RDSTRB(X,X1,QM,R,RI,RQ,RU,RV,RA)                        
               S=S+R*AL(I)                                       
               SI=SI+RI*AL(I)                                    
               SQ=SQ+RQ*AL(I)                                       
               SU=SU+RU*AL(I)                                    
               SV=SV+RV*AL(I)                                    
               SA=SA+RA*AL(I)                                    
   10          CONTINUE                                          
      RETURN                                               
      END                                                  

! C SCATTERING MATRIX R(X,X1,MU,GAMMA) 
! C needed first to call subroutines: 
! C COMX(X,X1); COMXM(QM); COMWG 
! C  INPUT: 
! C  1. X,X1 - ENERGIES OF  SCATTERED AND INITIAL PHOTONS;  
! C  2. QM=1-MU (MU- COSINE OF SCATTERING ANGLE); 
! C  3.  GAMMA=G (in COMMON/WZ/) 
! C  OUTPUT: R,RI,RQ,RU,RV, RF=RQ-RU 
! C 
      SUBROUTINE RDSTRB(X,X1,QM,R,RI,RQ,RU,RV,RF)
      IMPLICIT REAL*8(A-H,O-Z)                               
      COMMON /WZ/ Z,Z2,G,G2,GT,G1                            
      COMMON /WX/ XX,XX1,SX,DX,T,TR,XM2,DX2,GM1,GA1,RB,RWB,NR
      COMMON /WXM/ GST,GST1,QO,QR,QOT,Q2,Q,QT,QMT,PM,RS,QS
      DATA PI /3.141592653589792384D0/,EPS/1D-10/ 
      IF(Z.LE.0D0)  GO TO 510         
      IF(QM.GT.2D0) GO TO 520         
      IF(X.LE.0D0)  GO TO 530         
      IF(X1.LE.0D0) GO TO 540         
      IF(QM.lt.EPS) THEN
       R=0D0
       RI=0D0
       RQ=0D0
       RU=0D0
       RV=0D0
       RF=0D0
       IF(DABS(X-X1).lt.EPS) R=4D0*DLOG(G+Z)
      RETURN
      ELSE
      A=DSQRT(Z2-(GT-X)*X+QMT)        
      A1=DSQRT(Z2+(GT+X1)*X1+QMT)     
      U=SX*(GT-DX)/(A+A1)             
      U2=U*U                                  
      V=A*A1                                  
      VR=1D0/V                                
      VQ=QR*VR                                
      UV=U*VR                                 
      UVQ=UV*QR                               
      QU=U+Q                                  
      UQ=U-Q                                   
      IF(UQ.LT.3D-1) GO TO 500               
      QSR=1D0/QS
      UQQ=U2-Q2                               
      RF=0.5D0*UV*VQ*VQ*(U2*UQQ+V*(5D0*U2-3D0*Q2)) 
      RG=QT+UV*(1D0-QOT)                      
      RH=UV*QR*(UQQ*QSR-1D0)                  
      R=RF+RG                                 
      RI=RF+RH-UVQ                            
      QQQ=UQ*QSR                              
      RU=QT+2D0*QQQ*(QQQ*(Q+QU)-4D0)+2D0*RH   
      RQ=RU+RF                                
      RV=RG-QO*RF                             
      RETURN                                  
  500 GX=G-DX                                 
      DG=G-GST
      C=2D0/(G*GX+RS+T-QO+V)                  
      D=2D0*(GX+GST)*DG                        
      CD=C*D                                  
      UQQ=QS*CD                               
      QQQ=CD/QU                               
      RFO=0.5D0*UV*VQ*VQ*UQQ*(U2+5D0*V)       
      RGO=QT+UV                                
      RHO=UVQ*CD                              
      DFG=UV*VQ*((1D0-QS*C)*D+RS*(2D0+QO))    
      RF=RFO+UV*QR*VQ*Q2                      
      RG=RGO-2D0*UVQ                          
      RH=RHO-UVQ                              
      R=RFO+RGO-DFG                           
      RI=RFO+RHO-DFG                          
      RU=QT+2D0*QQQ*(QQQ*(Q+QU)-4D0)+2D0*RH   
      RQ=RU+RF                                
      RV=RG-QO*RF                             
      ENDIF 
      RETURN                                 
  510 WRITE(4,1010)                          
      RETURN                                 
  520 WRITE(4,1020)                          
      RETURN                                 
  530 WRITE(4,1030)                          
      RETURN                                 
  540 WRITE(4,1040)                          
      RETURN                                                                
 1010 FORMAT(1X,' Z  <= 0')
 1020 FORMAT(1X,' MU IS OUT  OF  RANGE')
 1030 FORMAT(1X,' X  <= 0') 
 1040 FORMAT(1X,' X1 <= 0')
      END                                                              

        
  
! c################
      SUBROUTINE AGUE(NL,E)             
      IMPLICIT REAL*8(A-H,O-Z)          
      COMMON /W/ UL(40),AL(40),AU(40),AE(40),N0
      DATA NOUT/8/ 
! c      pause 1
      DLE=-DLOG(E)                           
      CALL FQLAG0(NL,E,UL,AL)               
! c      pause 2
      DO 200 L=1,NL                          
! C         IF(DABS(AL(L)).LT.E) RETURN        
         A=DEXP(-UL(L))                      
         AU(L)=A                             
         AE(L)=DEXP(-A*A)*2D0                
! C         N0=L                                
  200 CONTINUE                               
       N0=NL
! c      WRITE(NOUT,1000) (i,UL(I),i,AL(I),I=1,NL)
! c 1000 format('ul(',i2,')=',e11.4,' al(',I2,')=',E11.4)
      RETURN                                 
      END                                    

! C     SUBROUTINE FQLAG0(K,EPS,T,A) CALCULATE WEIGHTS AND POINTS 
! C     K-POINTS GAUSSIAN QUADRATURE  
! C     FROM INTERVAL  [0,INFINITY) WITH WEIGHT  EXP(-X).
! C     EPS- RELATIVE ERROR OF POINTS CALCULATION 
! C     T(1:K),A(1:K)-VECTORS OF POINTS AND WEIGHTS 
! C     LIT.: A.K.PONOMARENKO,    LENINGRAD UNIVERSITY,1974 
! C 
      SUBROUTINE FQLAG0 (K,EPS,T,A)    
      IMPLICIT REAL*8(A-H,O-Z)         
      DIMENSION T(K),A(K)              
      I=0                              
      DP=K                             
      X=3.0D0/(1.0D0 + 2.4D0 * DP)     
      GO TO 30                         
   10 X=T(I) + 6.0D0 / (0.4D0 + DP)    
      GO TO 30                         
   20 J=I-1                            
      DJ=J                             
      X=T(I) + (T(I) - T(J)) * (1.0D0 + 2.55D0 * DJ)/1.9D0/DJ
   30 E2=1.0D0                         
      E3=1.0D0 - X                         
      IF (K.EQ.1) GO TO 41                 
      DO 40 J=2,K                          
       E1 = E2                             
      E2 = E3                              
      DJ = J                               
      DJ1= DJ - 1.0D0                      
   40 E3 = ((DJ + DJ1 - X)*E2 - DJ1 * E1)/DJ
   41 E1=DP*(E3-E2)                         
      E2 = E3/E1                            
      X  = X*(1.0D0 - E2)                   
      IF(DABS(E2)-EPS) 50,30,30             
   50 I=I+1                                 
      T(I)=X                                
      A(I)= X/E1/E1                         
      IF (I-1) 55,10,60                     
   55 WRITE(6,56)                           
   56 FORMAT(1X,'FQLAG0:ERROR.')             
      RETURN                                
   60 IF (I-K) 20,70,55                     
   70 RETURN                                
      END                                   


! C SUBROUTINE  DBESK (X,N,BK,X1,NOUT) CALCULATE 
! C FUNCTION CONNECTED WITH K(N,X) -- BESSEL FUNCTIONS OF ORDER 
! C FROM  0 TO N, WHICH PUT TO THE FIRST N+1 ELEMENTS OF MASSIVE BK.
! C C   X1 - BOUNDARY OF DIFFERENT  TYPES OF CALCULATIONS
! C        (5<=X1<=8). 
! C IF  0<=X<=X1  IS CALCULATED  K(N,X), IF   X>X1
! C IS CALCULATED FUNCTION:
! C
! C    K(N,X)*EXP(X)*SQRT(2*X/PI), WHERE PI=3.14... 
! C
! C USE SUBROUTINES:
! C
! C    DBI0LE(X)      -8<=X<=8  
! C    DBI1LO(X)      -8<=X<=8      DBK0GS(X)    X>=5 
! C    DBK0LE(X)       0< X<=8      DBK1GS(X)    X>=5 
! C    DBK1LO(X)       0< X<=8  
! C
      SUBROUTINE DBESK(X,N,BK,X1,NOUT)        
      IMPLICIT REAL*8(A-H,O-Z)                
      DIMENSION BK(5)                         
      DATA EILER /0.57721566490153286061D0/
      DATA C1 /1D0/                           
! C CONTROL OF ORDER:
      IF (N.GE.0) GO TO 10                    
! C REACTION ON THE NEGATIVE ORDER:
                              WRITE(NOUT,1000) 
                              RETURN           
! C CONTROL OF NEGATIVE ARGUMENT (X):
   10 IF (X) 13,15,20              
! C REACTION ON THE NEGATIVE ARGUMENT 
   13                         WRITE(NOUT,1300) 
                              RETURN           
! C CALCULATION IF X=0
   15 BK(1)=1D0     
      DO 17 I=1,N   
   17   BK(I+1)=0D0 
      RETURN        
! C       CALCULATION OF K(0,X)
   20 IF (X.LE.X1) BK(1)=DBK0LE(X) 
      IF (X.LE.X1) BK(1)=BK(1)-(EILER+DLOG(X/2D0))*DBI0LE(X)
      IF (X.GT.X1) BK(1)=DBK0GS(X)                          
      IF (N.EQ.0) RETURN                                    
! C CALCULATION OF K(1,X)
      IF (X.LE.X1) BK(2)=DBK1LO(X) 
      IF (X.LE.X1) ED=EILER+DLOG(X/2D0) 
      IF (X.LE.X1) BK(2)=ED*DBI1LO(X)+C1/X-BK(2) 
      IF (X.GT.X1) BK(2)=DBK1GS(X)               
      IF (N.EQ.1) RETURN                         
! C CALCULATION OF K(N,X)
      DO 30 I=2,N      
        BK(I+1)=BK(I-1)+(2*(I-1))*BK(I)/X  
   30 CONTINUE                             
      RETURN                               
 1000 FORMAT(1X,'DBESK: ORDER (SECOND PARAMETER) < 0.')
 1300 FORMAT(1X,'DBESK: ARGUMENT (FIRST PARAMETER) < 0.')
      END                                        


! C SUBROUTINE   DCHEB0 (A,X,N,KEY,NOUT)  CALCULATE 
! C THE VALUES OF COEFFICIENTS OF EXPANSION OVER THE 
! C CHEBYSHEV POLYNOMS
! C IF ARE KNOWN FIRST  N+1 ELEMENTS OF VESTOR  A 
! C 
! C           F(N,X)=[K=0,N] A(K) T(L(KEY),X),     
! C                                                
! C IF        L        KEY                         
! C         ______________                         
! C                                                
! C           N         0                          
! C         2N+11                                   
! C          2N         2                           
! C           N (DISPL) 3,                          
! C                                                 
! C ALGORITHM KLENSHOW
! C NOUT - NUMBER OF INFORMATION CHANAL OF  DCHEB0. 
! C LITERATURE:Y.LUKE "SPECIAL MATHEMATICAL FUNCTIONS 
! C                   AND ITS APPROXIMATIONS"
! C                "MIR",MOSCOW  1980 608 P.
! C  P.511
      FUNCTION DCHEB0(A,X,N,KEY,NOUT)   
      IMPLICIT REAL*8 (A-H,O-Z)         
      DIMENSION A(1)                    
      DATA CMAX /1D76/                  
      DATA C0,C2,C4 /0D0,2D0,4D0/       
! C
! C
! C CONTROL OF NUMBER OF ITEMS
! C
      IER=0  
      IF (N) 3,5,7
! C        REACTION ON THE NEGATIVE NUMBER
    3                    WRITE(NOUT,1000)
                         DCHEB0=CMAX     
                         IER=1           
! C   CONTROL OF KEY
    5 IF ((KEY.GE.0).AND.(KEY.LE.3)) GO TO 7       
! C  REACTION ON THE NONPOSITIVE VALUE OF KEY
                           WRITE(NOUT,1001)
                           DCHEB0=1D76     
                           IER=1           
    7 IF (IER.NE.0) RETURN                 
      N1=N+1                               
      IF (KEY.EQ.0) Z=C2*X                 
      IF (KEY.EQ.1) Z=C4*X*X-C2            
      IF (KEY.EQ.2) Z=C4*X*X-C2            
      IF (KEY.EQ.3) Z=C4*X-C2              
      B0=C0                                
      B1=C0                                
      DO 10 I=1,N1                         
        K=N1-I                             
        B2=B1                              
        B1=B0                              
        B0=Z*B1-B2+A(K+1)                  
   10 CONTINUE                             
      IF (KEY.EQ.0) DCHEB0=B0-X*B1         
      IF (KEY.EQ.1) DCHEB0=X*(B0-B1)       
      IF (KEY.EQ.2) DCHEB0=B0-B1*Z/C2      
      IF (KEY.EQ.3) DCHEB0=B0-B1*Z/C2      
      RETURN                               
 1000 FORMAT(1X,'DCHEB0:THIRD PARAMETER NEGATIVE')
 1001 FORMAT(1X,'DCHEB0:NONPOSITIVE KEY VALUE ')
! C     DEBUG INIT                                
      END                                       



! C FUNCTION  DBK0LE(X) CALCULATE 
! C BESSEL FUNCTION   K(0,X) + I(0,X)*[ EILER+DLOG(X/2) ] 
! C IF  X IS FROM INTERVAL [0,+8]
! C  ( EILER=0.57721 56649 01532 86061 ).
! C THROUGH EXPANSIONS OVER THE CHEBYSHEV POLYNOMS OF EVEN ORDER
! C LITERATURE:Y.LUKE "SPECIAL MATHEMATICAL FUNCTIONS 
! C                   AND ITS APPROXIMATIONS"
! C                "MIR",MOSCOW  1980 608 P.
! C TABLE  9.5  ; P.350
! C COEFFICIENTS C
      FUNCTION DBK0LE(X)    
      IMPLICIT REAL*8 (A-H,O-Z)   
      DIMENSION B(19)             
      DATA NOUT /8/               
      DATA B / 240.27705964072038910102D0, &
              369.47407397286728263764D0, &
              169.97341169840114804378D0, &
               49.02046377726343939371D0, &
                9.38849732526844232756D0, &
                1.25947976366770358618D0, &
                0.12377696411492454118D0, &
                0.00924430986286690621D0, &
                0.00054062389649255807D0, &
                0.00002537379602808704D0, &
                0.00000097547830283898D0, &
                0.00000003124957177932D0, &
                0.00000000084643470610D0, &
                0.00000000001962888451D0, &
                0.00000000000039396098D0, &
                0.00000000000000690835D0, &
                0.10673D-15, 0.146D-17, 0.2D-19  /
! C                                                  
      W=X*0.125D0                                  
      DBK0LE=DCHEB0(B,W,18,2,NOUT)                 
      RETURN                                       
! C                                                  
      END                                          



! C FUNCTION  DBK1LO(X) CALCULATE 
! C BESSEL FUNCTION 
! C            1/X + I(1,X)*[EILER+LN(X/2)]- K(1,X)       
! C  IF X IS FROM INTERVAL [0,+8]
! C ( EILER=0.57721 56649 01532 86061 ).
! C THROUGH EXPANSIONS OVER THE CHEBYSHEV POLYNOMS OF ODD  ORDER
! C LITERATURE:Y.LUKE "SPECIAL MATHEMATICAL FUNCTIONS 
! C                   AND ITS APPROXIMATIONS"
! C                "MIR",MOSCOW  1980 608 P.
! C TABLE  9.6  ; P.352
! C COEFFICIENTS C 
      FUNCTION DBK1LO(X)                      
      IMPLICIT REAL*8 (A-H,O-Z)               
      DIMENSION C(18)                         
      DATA NOUT /8/                           
      DATA C / 418.88944616639689097522D0, &
              249.89554904286808038961D0, &
               91.18031933874178775763D0, &
               21.44499505396224043921D0, &
                3.43841539288046459793D0, &
                0.39484609294093823432D0, &
                0.03382874552688419281D0, &
                0.00223572033417088760D0, &
                0.00011713102246084561D0, &
                0.00000497542712213645D0, &
                0.00000017460493176984D0, &
                0.00000000514329411806D0, &
                0.00000000012890339664D0, &
                0.00000000000278094119D0, &
                0.00000000000005217097D0, &
                0.00000000000000085869D0, &
                0.00000000000000001250D0, &
                0.00000000000000000016D0 /
      W=X*0.125D0                             
      DBK1LO=DCHEB0(C,W,17,1,NOUT) 
      RETURN                                  
! C                                             
      END                                     



! C FUNCTION  DBK0GS(X) CALCULATE 
! C BESSEL FUNCTION K(0,X)*SQRT(2*X/PI)*EXP(X) IF X>=5
! C THROUGH EXPANSIONS OVER THE SHIFTED(DISPLACED) CHEBYSHEV POLYNOMS 
! C LITERATURE:Y.LUKE "SPECIAL MATHEMATICAL FUNCTIONS 
! C                   AND ITS APPROXIMATIONS"
! C                "MIR",MOSCOW  1980 608 P.
! C TABLE  9.5  ; P.351
! C COEFFICIENTS D
      FUNCTION DBK0GS(X)      
      IMPLICIT REAL*8 (A-H,O-Z)  
      DIMENSION D(21)            
      DATA NOUT /8/              
      DATA D / 0.98840817423082580035D0, &
             -0.01131050464692828069D0, &
              0.00026953261276272369D0, &
             -0.00001110668519666535D0, &
              0.00000063257510850049D0, &
             -0.00000004504733764110D0, &
              0.00000000379299645568D0, &
             -0.00000000036454717921D0, &
              0.00000000003904375576D0, &
             -0.00000000000457993622D0, &
              0.00000000000058081063D0, &
             -0.00000000000007883236D0, &
              0.00000000000001136042D0, &
             -0.00000000000000172697D0, &
       0.27545D-15,   -0.4589D-16,    0.796D-17,  -0.143D-17, &
       0.27D-18,      -0.5D-19,       0.1D-19  /             
! C
! C
      W=5D0/X                                                 
      DBK0GS=DCHEB0(D,W,20,3,NOUT)                           
      RETURN                                                 
! C
      END                                                    



! C FUNCTION  DBK1GS(X) CALCULATE 
! C BESSEL FUNCTION K(1,X)*SQRT(2*X/PI)*EXP(X) IF X>=5
! C THROUGH EXPANSIONS OVER THE SHIFTED(DISPLACED) CHEBYSHEV POLYNOMS 
! C LITERATURE:Y.LUKE "SPECIAL MATHEMATICAL FUNCTIONS 
! C                   AND ITS APPROXIMATIONS"
! C                "MIR",MOSCOW  1980 608 P.
! C TABLE  9.6  ; P.351
! C     COEFFICIENTS   E  
      FUNCTION DBK1GS(X)                      
      IMPLICIT REAL*8 (A-H,O-Z)               
      DIMENSION D(21)                         
      DATA NOUT /8/                           
      DATA D / 1.03595085877235833071D0,   &
              0.03546529124333111380D0,   &
             -0.00046847502816688856D0,   &
              0.00001618506381005343D0,   &
             -0.00000084517204812368D0,   &
              0.00000005713221810284D0,   &
             -0.00000000464555460661D0,   &
              0.00000000043541733857D0,   &
             -0.00000000004575729704D0,   &
              0.00000000000528813281D0,   &
             -0.00000000000066261293D0,   &
              0.00000000000008904792D0,   &
             -0.00000000000001272607D0,   &
              0.00000000000000192086D0,   &
      -0.30450D-15,    0.5046D-16,   -0.871D-17,   0.156D-17, &
      -0.29D-18,       0.6D-19,      -0.1D-19  / 
! C
! C
      W=5D0/X                                     
      DBK1GS=DCHEB0(D,W,20,3,NOUT)                
      RETURN                                      
! C                                                 
      END                                         



! C FUNCTION  DBI0LE(X) CALCULATE 
! C BESSEL FUNCTION I(0,X) IF X IS FROM INTERVAL [-8,+8]
! C THROUGH EXPANSIONS OVER THE CHEBYSHEV POLYNOMS OF EVEN ORDER
! C LITERATURE:Y.LUKE "SPECIAL MATHEMATICAL FUNCTIONS 
! C                   AND ITS APPROXIMATIONS"
! C                "MIR",MOSCOW  1980 608 P.
! C TABLE  9.5  ; P.350
      FUNCTION DBI0LE(X)                      
      IMPLICIT REAL*8 (A-H,O-Z)               
      DIMENSION A(19)                         
      DATA NOUT /8/                           
      DATA A / 127.73343981218108356301D0, &
              190.49432017274284419322D0, &
               82.48903274402409961321D0, &
               22.27481924246223087742D0, &
                4.01167376017934853351D0, &
                0.50949336543998287079D0, &
                0.04771874879817413524D0, &
                0.00341633176601234095D0, &
                0.00019246935968811366D0, &
                0.00000873831549662236D0, &
                0.00000032609105057896D0, &
                0.00000001016972672769D0, &
                0.00000000026882812895D0, &
                0.00000000000609689280D0, &
                0.00000000000011989083D0, &
                0.00000000000000206305D0, &
                0.3132D-16, 0.42D-18, 0.1D-19  /
! C                                                
      W=X*0.125D0                                
      DBI0LE=DCHEB0(A,W,18,2,NOUT)               
      RETURN                                     
! C                                                
      END                                        



! C FUNCTION  DBI1LO(X) CALCULATE 
! C BESSEL FUNCTION I(1,X) IF X IS FROM INTERVAL [-8,+8]
! C THROUGH EXPANSIONS OVER THE CHEBYSHEV POLYNOMS OF ODD  ORDER
! C LITERATURE:Y.LUKE "SPECIAL MATHEMATICAL FUNCTIONS 
! C                   AND ITS APPROXIMATIONS"
! C                "MIR",MOSCOW  1980 608 P.
! C TABLE  9.6  ; P.352
      FUNCTION DBI1LO(X)                      
      IMPLICIT REAL*8 (A-H,O-Z)               
      DIMENSION A(18)                         
      DATA NOUT /8/                           
      DATA A / 220.60142692352377856112D0, &
              125.35426683715235646451D0, &
               42.86523409312825685130D0, &
                9.45300522943491053517D0, &
                1.42965770907621346814D0, &
                0.15592429547625629116D0, &
                0.01276804908173388545D0, &
                0.00081088790069069214D0, &
                0.00004101046193823750D0, &
                0.00000168804220343687D0, &
                0.00000005758695054206D0, &
                0.00000000165345353976D0, &
                0.00000000004048476606D0, &
                0.00000000000085496289D0, &
                0.00000000000001572708D0, &
                0.00000000000000025419D0, &
                0.00000000000000000364D0, &
                0.00000000000000000005D0 /
      W=X*0.125D0                             
      DBI1LO=DCHEB0(A,W,17,1,NOUT) 
      RETURN                                  
! C                                             
      END                                     



      SUBROUTINE QDRGS(Z,W,NUM)
      IMPLICIT REAL*8(A-H,O-Z)
! c      PARAMETER(NMAX=20)
      DIMENSION Z(1),W(1) 
      DATA ITRMAX/10/
      N=NUM
      IF (N.GT.1) GO TO 20
      IF (N.EQ.1) GO TO 10
      N=5
      GO TO 20
   10 Z(1)=0.D0
      W(1)=2.D0
      RETURN
   20 IND=MOD(N,2)
      K=N/2
      IF(IND.EQ.0) GOTO 40
      HP=1.D0
      FJ=1.D0
      DO 30 J=3,N,2
      FJ=FJ+2.D0
   30 HP=(FJ-1.D0)*HP/FJ
      HP2=HP*HP
      KIND=K+IND
      Z(KIND)=0.D0
      W(KIND)=HP2+HP2
   40 FN=DFLOAT(N)
      NP=N+1
      DO 100 I=1,K
      M=NP-I
      IF(I.EQ.1) X=1.D0-3.D0/(FN+2.D0)/FN
      IF(I.EQ.2) X=(X-1.D0)*5.25D0+1.D0
      IF(I.EQ.3) X=(X-Z(N))*1.6D0+X
      IF(I.GT.3) X=(X-Z(M+2))*3.D0+Z(M+3)
      ITER=0
   50 Y=X-1.D0
      ITER=ITER+1
      TY=2.D0*Y
      FJ=1.D0
      DP=Y
      P=X
      Q=1.D0
      DQ=1.D-0
      DO 60 J=2,N
      DQ=TY*Q+DQ+P
      Q=DQ+Q
      FJ=FJ+1.D0
      DP=(2.D0*FJ-1.D0)*Y*P+DP
   60 P=DP/FJ+P
      DIV=P/Q
      X=X-DIV
      FR=DIV/X
      IF(ITER.GT.ITRMAX) GOTO 190
      IF(DABS(FR).GT.1.D-16) GOTO 50
  190 CONTINUE
      Z(M)=X
      W(M)=2.D0/(1.D0-X*X)/Q/Q
      Z(I)=-X
      W(I)=W(M)
  100 CONTINUE
      RETURN
      END

! c***********************************************************************
! C calculation of cross-section for   
! C Maxwellian electron distribution. 
! C THIS VERSION CALCULATES ONLY THE TOTAL  
! C CS CROSS-SECTION:   AB (in units of Thomson 
! C                                         cross-section);  
! C INPUT: 
! C 1. X - PHOTON ENERGY IN UNITS OF ELECTRON REST MASS m_ec^2 ; 
! C 2. E - RELATIVE ERROR; 
! C OUTPUT: 
! C 3. AB - TOTAL CS CROSS-SECTION (in units of Thomson 
! C         cross-section;  
      SUBROUTINE CS(X,AB,E)                              
      IMPLICIT REAL*8(A-H,O-Z)                                          
      COMMON /W/ UG(40),AL(40),AU(40),AE(40),N0 
	  COMMON /WYY/ YE,Y1,YR1,YR2,DY,YKY,DK02,DK2Y ! new, short
!	  COMMON /WYY/ YR,YR1,YR2,DY,YKY,DK02,DK12,DK32,DK42,DK2Y,GAMMEAN,GAMSQUA,G0 !old, long
! c      COMMON /WY/ YR,YR1,YR2,DY,DK2R,DK2Y,DK(20),DK0,DK1,DK2,EXY,DLY,EY 
      DATA C13 /0.3333333333333333333D0/,DE1/3.67879441171442D-01/   
      
            
      X2=X*X                                   
      AB=0D0                                   
	  XM=0D0                             
      QM=0D0                                   
      PM=0D0                                   
!       DO 600 L=1,N0                            
      do L=1,N0                            
         U=UG(L)+1D0                           
         A=AU(L)                               
         AEL=AE(L)                             
         AGL=AL(L)                             
         G1=1D0+A*A*Y1                         
         GY1=G1+Y1                             
         YG1=G1*(G1+YR1)+YR2                   
         DZ1=DSQRT(1D0+G1)                     
         DR1=1D0/DZ1                           
         Z1=A*DZ1*DY                           
         GP1=G1+Z1                            
         ZP1=GP1*GP1                          
         GM1=1D0/GP1                          
         ZM1=GM1*GM1                          
         CALL SN(X*GP1,2,SSP,SSP1,SSP2,SP1,SP2,SP3,SP4,SP5,SP6,SP7,E) 
         CALL SN(X*GM1,2,SSM,SSM1,SSM2,SM1,SM2,SM3,SM4,SM5,SM6,SM7,E) 
         AB1=(ZP1*SSP+ZM1*SSM)*DR1                                    
!          XM1=((GP1*ZP1*(GY1*SP1+X*SP2)+GM1*ZM1*(GY1*SM1+X*SM2)))*DR1  
!          QM1=(ZP1*(0.5D0*SP6-GP1*(GY1*SP5+GP1*(SP7-YG1*SP4)))+ &     
!              ZM1*(0.5D0*SM6-GM1*(GY1*SM5+GM1*(SM7-YG1*SM4))))*DR1    
!          PM1=(ZP1*ZP1*SP1+ZM1*ZM1*SM1)*DR1                       
         G2=1D0+U*Y1                                             
         GY2=G2+Y1                                               
         YG2=G2*(G2+YR1)+YR2                                     
         DZ2=DSQRT(U*(1D0+G2))                                   
         DR2=1D0/DZ2                                             
         Z2=DZ2*DY                                               
         GP2=G2+Z2                                               
         ZP2=GP2*GP2                                             
         GM2=1D0/GP2                                             
         ZM2=GM2*GM2                                             
         CALL SN(X*GP2,2,SSP,SSP1,SSP2,SP1,SP2,SP3,SP4,SP5,SP6,SP7,E)
         CALL SN(X*GM2,2,SSM,SSM1,SSM2,SM1,SM2,SM3,SM4,SM5,SM6,SM7,E)
         AB2=(ZP2*SSP+ZM2*SSM)*DR2                                   
!          XM2=(GP2*ZP2*(GY2*SP1+X*SP2)+GM2*ZM2*(GY2*SM1+X*SM2))*DR2   
!          QM2=(ZP2*(0.5D0*SP6-GP2*(GY2*SP5+GP2*(SP7-YG2*SP4)))+   &    
!             ZM2*(0.5D0*SM6-GM2*(GY2*SM5+GM2*(SM7-YG2*SM4))))*DR2   
!          PM2=(ZP2*ZP2*SP1+ZM2*ZM2*SM1)*DR2 
         AB=AB+AGL*(AEL*AB1+DE1*AB2) 
!          XM=XM+AGL*(AEL*XM1+DE1*XM2) 
!          QM=QM+AGL*(AEL*QM1+DE1*QM2) 
!   600    PM=PM+AGL*(AEL*PM1+DE1*PM2) 
      enddo
      AB=AB*DK2Y                     
!       XM=XM*DK2Y                     
!       QM=QM*DK2Y                     
!       PM=PM*DK2Y            
! c	  WRITE(*,*) 'AB=',AB,' XM=',XM,' QM=',QM,' PM=',PM,' DK2Y=',DK2Y,' N0=',N0         
      RETURN                         
      END                            


! c***********************************************************************
! C calculation of cross-section, mean energy of scattered photon, 
! C dispersion of energy of scattered photon, and radiative pressure for   
! C Maxwellian electron distribution. 
! C THIS VERSION CALCULATES ONLY THE TOTAL  
! C C S CROSS-SECTION:   AB (in units of Thomson 
! C                                         cross-section);  
! C INPUT: 
! C 1. X - PHOTON ENERGY IN UNITS OF ELECTRON REST MASS m_ec^2 ; 
! C 2. E - RELATIVE ERROR; 
! C OUTPUT: 
! C 3. AB - TOTAL CS CROSS-SECTION (in units of Thomson 
! C         cross-section;  
! C 4. XM - MEAN ENERGY OF SCATTERED PHOTON; 
! C 5. QM - DISPERSION OF ENERGY OF SCATTERED PHOTON; 
! C 6. PM - RADIATIVE PRESSURE;    <x1(1-mu)>
      SUBROUTINE MXWELL(X,AB,XM,QM,PM,E)                              
      IMPLICIT REAL*8(A-H,O-Z)                                          
      COMMON /W/ UG(40),AG(40),AU(40),AE(40),N0 
	  COMMON /WYY/ YE,Y1,YR1,YR2,DY,YKY,DK02,DK2Y ! new, short
!	  COMMON /WYY/ YR,YR1,YR2,DY,YKY,DK02,DK12,DK32,DK42,DK2Y,GAMMEAN,GAMSQUA,G0 !old, long
! c      COMMON /WY/ YR,YR1,YR2,DY,DK2R,DK2Y,DK(20),DK0,DK1,DK2,EXY,DLY,EY 
      DATA C13 /0.3333333333333333333D0/,DE1/3.67879441171442D-01/   
      
            
      X2=X*X                                   
      AB=0D0                                   
	  XM=0D0                             
      QM=0D0                                   
      PM=0D0                                   
      DO 600 L=1,N0                            
         U=UG(L)+1D0                           
         A=AU(L)                               
         AEL=AE(L)                             
         AGL=AG(L)                             
         G1=1D0+A*A*Y1                         
         GY1=G1+Y1                             
         YG1=G1*(G1+YR1)+YR2                   
         DZ1=DSQRT(1D0+G1)                     
         DR1=1D0/DZ1                           
         Z1=A*DZ1*DY                           
         GP1=G1+Z1                            
         ZP1=GP1*GP1                          
         GM1=1D0/GP1                          
         ZM1=GM1*GM1                          
         CALL SN(X*GP1,2,SSP,SSP1,SSP2,SP1,SP2,SP3,SP4,SP5,SP6,SP7,E) 
         CALL SN(X*GM1,2,SSM,SSM1,SSM2,SM1,SM2,SM3,SM4,SM5,SM6,SM7,E) 
         AB1=(ZP1*SSP+ZM1*SSM)*DR1                                    
         XM1=((GP1*ZP1*(GY1*SP1+X*SP2)+GM1*ZM1*(GY1*SM1+X*SM2)))*DR1  
         QM1=(ZP1*(0.5D0*SP6-GP1*(GY1*SP5+GP1*(SP7-YG1*SP4)))+ &     
             ZM1*(0.5D0*SM6-GM1*(GY1*SM5+GM1*(SM7-YG1*SM4))))*DR1    
         PM1=(ZP1*ZP1*SP1+ZM1*ZM1*SM1)*DR1                       
         G2=1D0+U*Y1                                             
         GY2=G2+Y1                                               
         YG2=G2*(G2+YR1)+YR2                                     
         DZ2=DSQRT(U*(1D0+G2))                                   
         DR2=1D0/DZ2                                             
         Z2=DZ2*DY                                               
         GP2=G2+Z2                                               
         ZP2=GP2*GP2                                             
         GM2=1D0/GP2                                             
         ZM2=GM2*GM2                                             
         CALL SN(X*GP2,2,SSP,SSP1,SSP2,SP1,SP2,SP3,SP4,SP5,SP6,SP7,E)
         CALL SN(X*GM2,2,SSM,SSM1,SSM2,SM1,SM2,SM3,SM4,SM5,SM6,SM7,E)
         AB2=(ZP2*SSP+ZM2*SSM)*DR2                                   
         XM2=(GP2*ZP2*(GY2*SP1+X*SP2)+GM2*ZM2*(GY2*SM1+X*SM2))*DR2   
         QM2=(ZP2*(0.5D0*SP6-GP2*(GY2*SP5+GP2*(SP7-YG2*SP4)))+   &    
            ZM2*(0.5D0*SM6-GM2*(GY2*SM5+GM2*(SM7-YG2*SM4))))*DR2   
         PM2=(ZP2*ZP2*SP1+ZM2*ZM2*SM1)*DR2 
         AB=AB+AGL*(AEL*AB1+DE1*AB2) 
         XM=XM+AGL*(AEL*XM1+DE1*XM2) 
         QM=QM+AGL*(AEL*QM1+DE1*QM2) 
  600    PM=PM+AGL*(AEL*PM1+DE1*PM2) 
      AB=AB*DK2Y                     
      XM=XM*DK2Y                     
      QM=QM*DK2Y                     
      PM=PM*DK2Y            
! c	  WRITE(*,*) 'AB=',AB,' XM=',XM,' QM=',QM,' PM=',PM,' DK2Y=',DK2Y,' N0=',N0         
      RETURN                         
      END                            

! C  S- functions. if K=0, then only  SS -- mean cross-section; 
! C if  K=1, then  SS1 &  S1 -- mean frequency;
! C if K=2, all functions -- and the square of frequency
! C x -- small
      SUBROUTINE SN(X,K,SS,SS1,SS2,S1,S2,S3,S4,S5,S6,S7,E) 
      IMPLICIT REAL*8(A-H,O-Z)                             
      DATA XT /4D-1/                                       
      DATA C13/0.333333333333333D0/                        
      IF (X.LT.XT) GO TO 110                               
      X0=1D0/(1D0+2D0*X)                                   
      X2=X*X                                               
      XR=1D0/X                                             
      DLX0=-DLOG(X0)                                       
      X02=X0*X0                                            
      X2R=XR*XR                                            
      SS=0.375D0*(4D0-(2D0*XR+2D0-X)*DLX0+2D0*X2*(1D0+X)*X02)*X2R    
      IF(K.EQ.0) RETURN                                              
      SS1=0.125D0*X2R*XR*(3D0*DLX0+4D0*X2-4.5D0*X-X*X0*(1.5D0+X*X02))
      S1=(SS-SS1)*XR                             
      S2=(SS1-S1)*XR                             
      IF(K.EQ.1) RETURN                          
            SS2=(1D0+X*(4D0+X*(7D0+4.5D0*X)))*X02*X02
      S3=(SS1-SS2)*XR                                
            S4=(S1-S3)*XR                            
      GO TO 160                                      
  110 A=1D0                                          
      B=-2D0*X                                       
      DN=0D0                                         
      SS=0D0                                         
      IF (K.EQ.0) GO TO 120                          
      SS1=0D0                                        
      S1=0D0                                         
      S2=0D0                                         
      IF (K.EQ.1) GO TO 120                          
      SS2=0D0                                        
      S3=0D0                                         
      S4=0D0                                         
  120 DN1=DN+1D0                                     
      DN2=DN+2D0                                     
      DN3=DN+3D0                                     
      D1=1D0/DN1                                     
      D2=1D0/DN2                                     
      D3=1D0/DN3                                     
      DN4=DN+4D0                                     
      DN23=DN2*DN3                                   
      DN234=0.25D0*DN23*DN4                          
      SS=SS+A*(DN2+2D0*D1+8D0*D2-16D0*D3)            
      IF (K.EQ.0) GO TO 140                          
            D4=1D0/DN4                               
      D5=1D0/(DN4+1D0)                               
            DN13=DN1*DN3                             
                  DN14=DN1*DN4                       
      SS1=SS1+A*(DN23-6D0+24D0*D3)                   
      S1=S1+A*(DN13-6D0-6D0*D2-24D0*D3+72D0*D4)      
      S2=S2+A*(DN14-6D0-12D0*D3-72D0*D4+144D0*D5)    
      IF (K.EQ.1) GO TO 140                          
      DN134=0.25D0*DN13*DN4                          
      DN124=0.25D0*DN2*DN14                          
      SS2=SS2+A*(DN234+2D0-DN)                       
      S3=S3+A*(DN134+7D0-DN-24D0*D4)                 
      S4=S4+A*(DN124+12D0-DN+6D0*D3+24D0*D4-96D0*D5) 
  140 IF (DABS(4D0*A)*DN234.LT.E) GO TO 150          
      DN=DN+1D0                                      
      A=A*B                                          
      GO TO 120                                      
  150 SS=SS*0.375D0                                  
      IF (K.EQ.0) RETURN                             
      SS1=SS1*0.125D0                                
      S1=S1*0.25D0                                   
      S2=S2*0.25D0                                   
      IF (K.EQ.1) RETURN                             
            SS2=SS2*0.125D0                          
      S3=S3*0.25D0                                   
      S4=S4*0.5D0                                    
  160 S5=3D0*S4-4D0*S3                               
      S7=S3-0.5D0*S4                                 
      S6=2D0*SS2-6D0*S7                              
! c      WRITE(6,1000) X,SS,SS1,SS2,S1,S2,S3,S4,S5,S6,S7
 1000 FORMAT(1X,6D17.9/5D17.9)                       
      RETURN                                         
      END                                            

      
