#Script used to produce the analysis in the article "Food waste between environmental education, peers, and family influence. Insights from primary school students in Northern Italy," Journal of Cleaner Production (Piras et al., 2023)

# load dataset
db <- read.table("Dataset_final.csv",header=T, sep=";", dec=".")

# load libraries
library(lme4)
library(ordinal)

# DATA MANAGEMENT ####

# characteristics of dataset
dim(db)
names(db)

# declaration of ordinal variables

db$FREQ_FW_CANT <- factor(db$FREQ_FW_CANT, ordered=T, levels=6:1)
db$FREQ_FW_HOME <- factor(db$FREQ_FW_HOME, ordered=T, levels=6:1)
db$SELF_FW_CANT <- factor(db$SELF_FW_CANT, ordered=T, levels=5:1)

# DESCRIPTIVE STATISTICS ####

# list of descriptive variables
var <- c("SEX","N_FAM_CL","FOREIGN","CARS","PGG","DICTATOR","TRUST","SCOLD","STRICTNESS","OP_PAR_ENV","OP_PAR_MAT","OP_PAR_ETH","FREQ_FW_CANT","FREQ_FW_CANT_NEARBY_CL","FREQ_FW_CANT_FRIENDS_CL","FREQ_FW_CANT_NONFRIENDS_CL","FREQ_FW_CANT_BREAK_CL","FREQ_FW_CANT_NICE_CL","FREQ_FW_CANT_POPULAR_CL","FREQ_FW_CANT_DESK_CL","FREQ_FW_CANT_DESIRED_CL","OP_CH_ENV","OP_CH_MAT","OP_CH_ETH")

# type of descriptive variables (N=Numeric, C=Categorical)
type_var <- c("C","C","C","C","N","N","N","C","C","C","C","C","C","C","C","C","C","C","C","C","C","C","C","C","C")  

# stratification variable 
strat_var <- "WAVE"

# row selection variable 
sel_var <- rep(T,1260) # all rows

# loop for calculating descriptive statistics

tab <- data.frame(matrix(NA,0,5+2*length(levels(droplevels(as.factor(db[sel_var,strat_var]))))))

for (i in 1:length(var)) {
  if (type_var[i]=="C") {
    row <- data.frame(matrix(NA,length(levels(droplevels(as.factor(db[sel_var,var[i]])))),4+2*length(levels(droplevels(as.factor(db[sel_var,strat_var]))))))
    row[,1] <- levels(droplevels(as.factor(db[sel_var,var[i]])))
    cols_index <- seq(2,length(levels(droplevels(as.factor(db[sel_var,strat_var]))))*2,2)
    row[,cols_index] <- table(droplevels(as.factor(db[sel_var,var[i]])), db[sel_var,strat_var])
    for (k in cols_index) { row[,k+1] <- row[,k] / sum(row[,k]) }
    row[,length(cols_index)*2+2] <- table(droplevels(as.factor(db[sel_var,var[i]])))
    row[,length(cols_index)*2+3] <- row[,length(cols_index)*2+2] / sum(row[,length(cols_index)*2+2])
    row[,length(cols_index)*2+4] <- fisher.test(table(droplevels(as.factor(db[sel_var,var[i]])), droplevels(as.factor(db[sel_var,strat_var]))), simulate.p.value=TRUE)$p.value
    row[,length(cols_index)*2+5] <- sum(is.na(db[sel_var,var[i]]))
    tab <- rbind(tab,row)
  }
  if (type_var[i]=="N") {
    row <- data.frame(matrix(NA,1,4+2*length(levels(droplevels(as.factor(db[sel_var,strat_var]))))))
    row[,1] <- var[i]
    cols_index <- seq(2,length(levels(droplevels(as.factor(db[sel_var,strat_var]))))*2,2)
    row[,cols_index] <- by(db[sel_var,var[i]], db[sel_var,strat_var], FUN=function(x) mean(x, na.rm=T))
    row[,cols_index+1] <- by(db[sel_var,var[i]], db[sel_var,strat_var], FUN=function(x) sd(x, na.rm=T))
    row[,length(cols_index)*2+2] <- mean(db[sel_var,var[i]], na.rm=T)
    row[,length(cols_index)*2+3] <- sd(db[sel_var,var[i]], na.rm=T)
    row[,length(cols_index)*2+4] <- kruskal.test(db[sel_var,var[i]] ~ db[sel_var,strat_var])$p.value
    row[,length(cols_index)*2+5] <- sum(is.na(db[sel_var,var[i]]))
    tab <- rbind(tab,row)
  }
}

# save results
write.table(tab,"DESCRIPTIVE.csv", sep=";", dec=".", row.names = F)

# REGRESSION MODELS ####

# FOOD WASTE AT SCHOOL CANTEEN

# Proportional Odds Model - POM
mod_FREQ_FW_CANT <- clmm(FREQ_FW_CANT~ SEX + CARS + N_FAM_CL + FOREIGN + PGG + OP_PAR_ENV + OP_PAR_MAT + OP_PAR_ETH + SCOLD + STRICTNESS + FREQ_FW_CANT_NEARBY_CL + as.factor(WAVE) * TREATED + (1|UNIQUEID) + (1|SCHOOL_CLASS), data=db)
summary(mod_FREQ_FW_CANT)

# Export results
tab_s <- matrix(NA,16,4)
tab_s[,1] <- exp(summary(mod_FREQ_FW_CANT)$coef[-1:-5,1]) # cOR
tab_s[,2:3] <- exp(confint.default(mod_FREQ_FW_CANT)[-1:-5,]) # 95% CI
tab_s[,4] <- summary(mod_FREQ_FW_CANT)$coef[-1:-5,4] # p-value
write.table(tab_s,"FREQ_FW_CANT_MULTI.csv", sep=";", row.names=F, dec=".")

# FOOD WASTE AT HOME

# Proportional Odds Model - POM
mod_FREQ_FW_HOME <- clmm(FREQ_FW_HOME ~ SEX + CARS + N_FAM_CL + FOREIGN + PGG + OP_PAR_ENV + OP_PAR_MAT + OP_PAR_ETH + SCOLD + STRICTNESS + FREQ_FW_CANT_NEARBY_CL + as.factor(WAVE) * TREATED + (1|UNIQUEID) + (1|SCHOOL_CLASS), data=db)
summary(mod_FREQ_FW_HOME)

# Export results
tab_s <- matrix(NA,16,4)
tab_s[,1] <- exp(summary(mod_FREQ_FW_HOME)$coef[-1:-5,1]) # cOR
tab_s[,2:3] <- exp(confint.default(mod_FREQ_FW_HOME)[-1:-5,]) # 95% CI
tab_s[,4] <- summary(mod_FREQ_FW_HOME)$coef[-1:-5,4] # p-value
write.table(tab_s,"FREQ_FW_HOME_MULTI.csv", sep=";", row.names=F, dec=".")

# SELF-ASSESSMENT OF FOOD WASTE AT SCHOOL CANTEEN

# Proportional Odds Model - POM
mod_SELF_FW_CANT <- clmm(SELF_FW_CANT ~ SEX + CARS + N_FAM_CL + FOREIGN + PGG + OP_PAR_ENV + OP_PAR_MAT + OP_PAR_ETH + SCOLD + STRICTNESS + FREQ_FW_CANT_NEARBY_CL + as.factor(WAVE) * TREATED + (1|UNIQUEID) + (1|SCHOOL_CLASS), data=db)
summary(mod_SELF_FW_CANT)

# Export results
tab_s <- matrix(NA,16,4)
tab_s[,1] <- exp(summary(mod_SELF_FW_CANT)$coef[-1:-4,1]) # cOR
tab_s[,2:3] <- exp(confint.default(mod_SELF_FW_CANT)[-1:-4,]) # 95% CI
tab_s[,4] <- summary(mod_SELF_FW_CANT)$coef[-1:-4,4] # p-value
write.table(tab_s,"SELF_FW_CANT_MULTI.csv", sep=";", row.names=F, dec=".")

# CHILDREN PERCEPTION OF FOOD WASTE

# environmental
mod_OP_CH_ENV <- glmer(OP_CH_ENV~ SEX + CARS + N_FAM_CL + FOREIGN + PGG + OP_PAR_ENV + OP_PAR_MAT + OP_PAR_ETH + SCOLD + STRICTNESS + FREQ_FW_CANT_NEARBY_CL + as.factor(WAVE) * TREATED + (1|UNIQUEID) + (1|SCHOOL_CLASS), family=binomial(link="logit"), data=db)
summary(mod_OP_CH_ENV)

# Export results
tab_s <- matrix(NA,16,4)
tab_s[,1] <- exp(summary(mod_OP_CH_ENV)$coef[-1,1]) # cOR
tab_s[,2:3] <- exp(confint.merMod(mod_OP_CH_ENV, method="Wald")[-1:-3,]) # 95% CI
tab_s[,4] <- summary(mod_OP_CH_ENV)$coef[-1,4] # p-value
write.table(tab_s,"OP_CH_ENV_MULTI.csv", sep=";", row.names=F, dec=".")

# ethical
mod_OP_CH_ETH <- glmer(OP_CH_ETH~ SEX + CARS + N_FAM_CL + FOREIGN + PGG + OP_PAR_ENV + OP_PAR_MAT + OP_PAR_ETH + SCOLD + STRICTNESS + FREQ_FW_CANT_NEARBY_CL + as.factor(WAVE) * TREATED + (1|UNIQUEID) + (1|SCHOOL_CLASS), family=binomial(link="logit"), data=db)
summary(mod_OP_CH_ETH)

# Export results
tab_s <- matrix(NA,16,4)
tab_s[,1] <- exp(summary(mod_OP_CH_ETH)$coef[-1,1]) # cOR
tab_s[,2:3] <- exp(confint.merMod(mod_OP_CH_ETH, method="Wald")[-1:-3,]) # 95% CI
tab_s[,4] <- summary(mod_OP_CH_ETH)$coef[-1,4] # p-value
write.table(tab_s,"OP_CH_ETH_MULTI.csv", sep=";", row.names=F, dec=".")

# material
mod_OP_CH_MAT <- glmer(OP_CH_MAT~ SEX + CARS + N_FAM_CL + FOREIGN + PGG + OP_PAR_ENV + OP_PAR_MAT + OP_PAR_ETH + SCOLD + STRICTNESS + FREQ_FW_CANT_NEARBY_CL + as.factor(WAVE) * TREATED + (1|UNIQUEID) + (1|SCHOOL_CLASS), family=binomial(link="logit"), data=db)
summary(mod_OP_CH_MAT)

# Export results
tab_s <- matrix(NA,16,4)
tab_s[,1] <- exp(summary(mod_OP_CH_MAT)$coef[-1,1]) # cOR
tab_s[,2:3] <- exp(confint.merMod(mod_OP_CH_MAT, method="Wald")[-1:-3,]) # 95% CI
tab_s[,4] <- summary(mod_OP_CH_MAT)$coef[-1,4] # p-value
write.table(tab_s,"OP_CH_MAT_MULTI.csv", sep=";", row.names=F, dec=".")

# OTHER NETWORK VARIABLES ####

# to replicate the models with alternative network variables, the "FREQ_FW_CANT_NEARBY_CL" variable should be replaced in models equations
# e.g., to include the Food Waste of friends, instead of that of peers seeting nearby, for the Food Waste at school canteen outcome:
# clmm(FREQ_FW_CANT~ SEX + CARS + N_FAM_CL + FOREIGN + PGG + OP_PAR_ENV + OP_PAR_MAT + OP_PAR_ETH + SCOLD + STRICTNESS + FREQ_FW_CANT_FRIENDS_CL + as.factor(WAVE) * TREATED + (1|UNIQUEID) + (1|SCHOOL_CLASS), data=db)

# OTHER GAMES VARIABLES ####

# to replicate the models with alternative games variables, the "PGG" variable should be replaced in models equations
# e.g., to include the Dictator game, instead of the Public Good Game, for the Food Waste at school canteen outcome:
# clmm(FREQ_FW_CANT~ SEX + CARS + N_FAM_CL + FOREIGN + DICTATOR + OP_PAR_ENV + OP_PAR_MAT + OP_PAR_ETH + SCOLD + STRICTNESS + FREQ_FW_CANT_FRIENDS_CL + as.factor(WAVE) * TREATED + (1|UNIQUEID) + (1|SCHOOL_CLASS), data=db)

# FIGURES ####

# FOOD WASTE per WAVE

# export in tiff format
nome <- "Fig_A.tiff"
tiff(file=nome, width = 18, height = 10, units = 'cm', res = 600)
counts <-apply( cbind(table(db$FREQ_FW_CANT>=3,  db$WAVE),rbind(NA,NA),
            table(db$SELF_FW_CANT>=2,  db$WAVE),rbind(NA,NA),
            table(db$FREQ_FW_HOME>=3,  db$WAVE)), 2, sum)
values <- c(prop.table(table(db$FREQ_FW_CANT>=3,  db$WAVE),2)[2,],NA,
            prop.table(table(db$SELF_FW_CANT>=2,  db$WAVE),2)[2,],NA,
            prop.table(table(db$FREQ_FW_HOME>=3,  db$WAVE),2)[2,])
barplot(values, xlim=c(0,13), ylim=c(0,1), ylab="Share of students", col=rep(c("darkorange","firebrick","purple",NA),3),names.arg=NA)

mtext( "That waste at school \n >50% of the times", side=1, at=1.9, line=2, cex=0.75)
mtext( "That assess their waste at school \n as often or too often", side=1, at=6.7, line=2, cex=0.75)
mtext( "That waste at home \n >50% of the times", side=1, at=11.5, line=2, cex=0.75)

legend(x=0.7, y=1, horiz=T, c("Wave 1","Wave 2","Wave 3"), fill=c("darkorange","firebrick","purple"), box.col="white")

lower <- values - 1.96 * sqrt((values*(1-values))/counts)
upper <- values + 1.96 * sqrt((values*(1-values))/counts)
for (i in 1:11) {
  segments(x0=-0.5+1.2*i, x1=-0.5+1.2*i, y0=lower[i], y1=upper[i])
}

dev.off()

# export in .eps format
nome <- "Fig_A.eps"
cairo_ps(file = nome, onefile = T, width = 18/2.54, height = 10/2.54, fallback_resolution = 600)
counts <-apply( cbind(table(db$FREQ_FW_CANT>=3,  db$WAVE),rbind(NA,NA),
                      table(db$SELF_FW_CANT>=2,  db$WAVE),rbind(NA,NA),
                      table(db$FREQ_FW_HOME>=3,  db$WAVE)), 2, sum)
values <- c(prop.table(table(db$FREQ_FW_CANT>=3,  db$WAVE),2)[2,],NA,
            prop.table(table(db$SELF_FW_CANT>=2,  db$WAVE),2)[2,],NA,
            prop.table(table(db$FREQ_FW_HOME>=3,  db$WAVE),2)[2,])
barplot(values, xlim=c(0,13), ylim=c(0,1), ylab="Share of students", col=rep(c("darkorange","firebrick","purple",NA),3),names.arg=NA)

mtext( "That waste at school \n >50% of the times", side=1, at=1.9, line=2, cex=0.75)
mtext( "That assess their waste at school \n as often or too often", side=1, at=6.7, line=2, cex=0.75)
mtext( "That waste at home \n >50% of the times", side=1, at=11.5, line=2, cex=0.75)

legend(x=0.7, y=1, horiz=T, c("Wave 1","Wave 2","Wave 3"), fill=c("darkorange","firebrick","purple"), box.col="white")

lower <- values - 1.96 * sqrt((values*(1-values))/counts)
upper <- values + 1.96 * sqrt((values*(1-values))/counts)
for (i in 1:11) {
  segments(x0=-0.5+1.2*i, x1=-0.5+1.2*i, y0=lower[i], y1=upper[i])
}

dev.off()

# CHILDREN OPINION per WAVE

nome <- "Fig_B.tiff"
tiff(file=nome, width = 18, height = 10, units = 'cm', res = 600)
counts <-apply( cbind(table(db$OP_CH_ENV==1,  db$WAVE),rbind(NA,NA),
                      table(db$OP_CH_ETH==1,  db$WAVE),rbind(NA,NA),
                      table(db$OP_CH_MAT==1,  db$WAVE)), 2, sum)
values <- c(prop.table(table(db$OP_CH_ENV==1,  db$WAVE),2)[2,],NA,
            prop.table(table(db$OP_CH_ETH==1,  db$WAVE),2)[2,],NA,
            prop.table(table(db$OP_CH_MAT==1,  db$WAVE),2)[2,])
barplot(values, xlim=c(0,13), ylim=c(0,1), col=rep(c("darkorange","firebrick","purple",NA),3),ylab="Share of student expressing the opinion",
        names.arg=c(NA,"Environmental opinion",NA,NA,NA,"Ethical opinion",NA,NA,NA,"Material opinion",NA))
legend(x=0.7, y=1, horiz=T, c("Wave 1","Wave 2","Wave 3"), fill=c("darkorange","firebrick","purple"), box.col="white")

lower <- values - 1.96 * sqrt((values*(1-values))/counts)
upper <- values + 1.96 * sqrt((values*(1-values))/counts)
for (i in 1:11) {
  segments(x0=-0.5+1.2*i, x1=-0.5+1.2*i, y0=lower[i], y1=upper[i])
}

dev.off()

nome <- "Fig_B.eps"
cairo_ps(file = nome, onefile = T, width = 18/2.54, height = 10/2.54, fallback_resolution = 600)
counts <-apply( cbind(table(db$OP_CH_ENV==1,  db$WAVE),rbind(NA,NA),
                      table(db$OP_CH_ETH==1,  db$WAVE),rbind(NA,NA),
                      table(db$OP_CH_MAT==1,  db$WAVE)), 2, sum)
values <- c(prop.table(table(db$OP_CH_ENV==1,  db$WAVE),2)[2,],NA,
            prop.table(table(db$OP_CH_ETH==1,  db$WAVE),2)[2,],NA,
            prop.table(table(db$OP_CH_MAT==1,  db$WAVE),2)[2,])
barplot(values, xlim=c(0,13), ylim=c(0,1), col=rep(c("darkorange","firebrick","purple",NA),3),ylab="Share of student expressing the opinion",
        names.arg=c(NA,"Environmental opinion",NA,NA,NA,"Ethical opinion",NA,NA,NA,"Material opinion",NA))
legend(x=0.7, y=1, horiz=T, c("Wave 1","Wave 2","Wave 3"), fill=c("darkorange","firebrick","purple"), box.col="white")

lower <- values - 1.96 * sqrt((values*(1-values))/counts)
upper <- values + 1.96 * sqrt((values*(1-values))/counts)
for (i in 1:11) {
  segments(x0=-0.5+1.2*i, x1=-0.5+1.2*i, y0=lower[i], y1=upper[i])
}

dev.off()

# FOOD WASTE per EDUCATION

# export in tiff format
nome <- "Fig_C.tiff"
tiff(file=nome, width = 12, height = 10, units = 'cm', res = 600)
counts <-apply( cbind(table(db$FREQ_FW_CANT[db$WAVE==1]>=3, db$TREATED[db$WAVE==1]),rbind(NA,NA),
                      table(db$FREQ_FW_CANT[db$WAVE==2]>=3, db$TREATED[db$WAVE==2]),rbind(NA,NA),
                      table(db$FREQ_FW_CANT[db$WAVE==3]>=3, db$TREATED[db$WAVE==3])), 2, sum)
values <- c(prop.table(table(db$FREQ_FW_CANT[db$WAVE==1]>=3, db$TREATED[db$WAVE==1]),2)[2,],NA,
            prop.table(table(db$FREQ_FW_CANT[db$WAVE==2]>=3, db$TREATED[db$WAVE==2]),2)[2,],NA,
            prop.table(table(db$FREQ_FW_CANT[db$WAVE==3]>=3, db$TREATED[db$WAVE==3]),2)[2,])
barplot(values, xlim=c(0,9), ylim=c(0,0.8), col=rep(c("red","blue",NA),3), names.arg=F)
legend(x=0.7, y=0.8,  horiz=F, c("Lesson on energy waste","Lesson on food waste"), fill=c("red","blue"), box.col="white")

lower <- values - 1.96 * sqrt((values*(1-values))/counts)
upper <- values + 1.96 * sqrt((values*(1-values))/counts)
for (i in 1:8) {
  segments(x0=-0.5+1.2*i, x1=-0.5+1.2*i, y0=lower[i], y1=upper[i])
}
mtext("Wave 1",side = 1, at=1.3, line=1.5)
mtext("Wave 2",side = 1, at=4.9, line=1.5)
mtext("Wave 3",side = 1, at=8.5, line=1.5)
mtext("Share of students that waste at school \n >50% of the times", side=2, at=0.4, line=2.2)
dev.off()

# export in .eps format
nome <- "Fig_C.eps"
cairo_ps(file = nome, onefile = T, width = 12/2.54, height = 10/2.54, fallback_resolution = 600)
counts <-apply( cbind(table(db$FREQ_FW_CANT[db$WAVE==1]>=3, db$TREATED[db$WAVE==1]),rbind(NA,NA),
                      table(db$FREQ_FW_CANT[db$WAVE==2]>=3, db$TREATED[db$WAVE==2]),rbind(NA,NA),
                      table(db$FREQ_FW_CANT[db$WAVE==3]>=3, db$TREATED[db$WAVE==3])), 2, sum)
values <- c(prop.table(table(db$FREQ_FW_CANT[db$WAVE==1]>=3, db$TREATED[db$WAVE==1]),2)[2,],NA,
            prop.table(table(db$FREQ_FW_CANT[db$WAVE==2]>=3, db$TREATED[db$WAVE==2]),2)[2,],NA,
            prop.table(table(db$FREQ_FW_CANT[db$WAVE==3]>=3, db$TREATED[db$WAVE==3]),2)[2,])
barplot(values, xlim=c(0,9), ylim=c(0,0.8), col=rep(c("red","blue",NA),3), names.arg=F)
legend(x=0.7, y=0.8,  horiz=F, c("Lesson on energy waste","Lesson on food waste"), fill=c("red","blue"), box.col="white")
lower <- values - 1.96 * sqrt((values*(1-values))/counts)
upper <- values + 1.96 * sqrt((values*(1-values))/counts)
for (i in 1:8) {
  segments(x0=-0.5+1.2*i, x1=-0.5+1.2*i, y0=lower[i], y1=upper[i])
}
mtext("Wave 1",side = 1, at=1.3, line=1.5)
mtext("Wave 2",side = 1, at=4.9, line=1.5)
mtext("Wave 3",side = 1, at=8.5, line=1.5)
mtext("Share of students that waste at school \n >50% of the times", side=2, at=0.4, line=2.2)

dev.off()

# ENVIRONMENTAL OPINION per EDUCATION

# export in tiff format
nome <- "Fig_D.tiff"
tiff(file=nome, width = 12, height = 10, units = 'cm', res = 600)
counts <-apply( cbind(table(db$OP_CH_ENV[db$WAVE==1]==1, db$TREATED[db$WAVE==1]),rbind(NA,NA),
                      table(db$OP_CH_ENV[db$WAVE==2]==1, db$TREATED[db$WAVE==2]),rbind(NA,NA),
                      table(db$OP_CH_ENV[db$WAVE==3]==1, db$TREATED[db$WAVE==3])), 2, sum)
values <- c(prop.table(table(db$OP_CH_ENV[db$WAVE==1]==1, db$TREATED[db$WAVE==1]),2)[2,],NA,
            prop.table(table(db$OP_CH_ENV[db$WAVE==2]==1, db$TREATED[db$WAVE==2]),2)[2,],NA,
            prop.table(table(db$OP_CH_ENV[db$WAVE==3]==1, db$TREATED[db$WAVE==3]),2)[2,])
barplot(values, xlim=c(0,9), ylim=c(0,1), col=rep(c("red","blue",NA),3), names.arg=F)
legend(x=0.7, y=1,  horiz=F, c("Lesson on energy waste","Lesson on food waste"), fill=c("red","blue"), box.col="white")

lower <- values - 1.96 * sqrt((values*(1-values))/counts)
upper <- values + 1.96 * sqrt((values*(1-values))/counts)
for (i in 1:8) {
  segments(x0=-0.5+1.2*i, x1=-0.5+1.2*i, y0=lower[i], y1=upper[i])
}
mtext("Wave 1",side = 1, at=1.3, line=1.5)
mtext("Wave 2",side = 1, at=4.9, line=1.5)
mtext("Wave 3",side = 1, at=8.5, line=1.5)
mtext("Share of students \n expressing environmental opinions", side=2, at=0.5, line=2.2)

dev.off()

# export in .eps format
nome <- "Fig_D.eps"
cairo_ps(file = nome, onefile = T, width = 12/2.54, height = 10/2.54, fallback_resolution = 600)
counts <-apply( cbind(table(db$OP_CH_ENV[db$WAVE==1]==1, db$TREATED[db$WAVE==1]),rbind(NA,NA),
                      table(db$OP_CH_ENV[db$WAVE==2]==1, db$TREATED[db$WAVE==2]),rbind(NA,NA),
                      table(db$OP_CH_ENV[db$WAVE==3]==1, db$TREATED[db$WAVE==3])), 2, sum)
values <- c(prop.table(table(db$OP_CH_ENV[db$WAVE==1]==1, db$TREATED[db$WAVE==1]),2)[2,],NA,
            prop.table(table(db$OP_CH_ENV[db$WAVE==2]==1, db$TREATED[db$WAVE==2]),2)[2,],NA,
            prop.table(table(db$OP_CH_ENV[db$WAVE==3]==1, db$TREATED[db$WAVE==3]),2)[2,])
barplot(values, xlim=c(0,9), ylim=c(0,1), col=rep(c("red","blue",NA),3), names.arg=F)
legend(x=0.7, y=1,  horiz=F, c("Lesson on energy waste","Lesson on food waste"), fill=c("red","blue"), box.col="white")

lower <- values - 1.96 * sqrt((values*(1-values))/counts)
upper <- values + 1.96 * sqrt((values*(1-values))/counts)
for (i in 1:8) {
  segments(x0=-0.5+1.2*i, x1=-0.5+1.2*i, y0=lower[i], y1=upper[i])
}
mtext("Wave 1",side = 1, at=1.3, line=1.5)
mtext("Wave 2",side = 1, at=4.9, line=1.5)
mtext("Wave 3",side = 1, at=8.5, line=1.5)
mtext("Share of students \n expressing environmental opinions", side=2, at=0.5, line=2.2)
dev.off()

