# A WorldDynamics tutorial

`WorldDynamics` allows the user to *play* with the World3 model introduced in the book *Dynamics of Growth in a Finite World* (1974). Informally speaking, this model is formed by five systems, each containg one or more subsystems. The following picture shows the structure of the model and the connections between the subsystems which share a common variable.

![The World3 model](img/world3.png)

As it can be seen, the five systems are `Pop4` (which is the population system with four age levels), `Agriculture`, `Capital`, `Non-renewable` (resources), and `Pollution`. The `Pop4` system is formed by the three subsystems `pop` (population), `br` (birth rate), and `dr` (death rate). For instance, the subsystem `br` uses the variable `pop` which originates from the subsystem `pop`, while the subsystem `pop` uses the variable `le` which originates from the subsystem `dr`. Of course, there are variables which connect subsystem of different systems. For example, the subsystem `pp` of the system `Pollution` uses the variable `aiph` which originates from the subsystem `ai` of the system `Agriculture` (for an entire list of variables and of subsystems using them see the [World 3 equations, variables, and parameters](@ref eqs_vars_pars) page).

In `WorldDynamics` each system is a Julia module and each subsystem correspond to a Julia function of this module (or of a module which is included in this module), which defines the ODE system corresponding to the subsystem itself. All the ODE systems corresponding to the subsystems of the World3 model have to be composed (see the function `compose` in the `solvesystems.jl` code file). This will produce the entire ODE system of the World3 model, which can then be solved by using the function `solve` in the `solvesystems.jl` code file.

Let us now see how we can replicate the runs described in Chapter 7 of the above mentioned book. 
## Replicating historical runs

We first have to solve the World3 ODE system, which is constructed in the `world3_historicalrun` function, included in the `world3_historicalrun.jl` code file.

```
using WorldDynamics
include("examples/scenarios/world3_historicalrun.jl")
system = world3_historicalrun()
sol = WorldDynamics.solve(system, (1900, 2100))
```

We then have to define the variables that we want to plot. For example, Figure 7-2 of the above book shows the plot of eleven variables in the population system of the model. In order to easily access to these variables, we first create shortcuts to the subsystems in which they are introduced.

```
@named pop = WorldDynamics.World3.Pop4.population()
@named br = WorldDynamics.World3.Pop4.birth_rate()
@named dr = WorldDynamics.World3.Pop4.death_rate()
```

The eleven variables are then defined as follows.

```
fig_7_2_variables = [
    (pop.pop, 0, 4e9, "pop"),
    (br.cbr, 0, 50, "cbr"),
    (dr.cdr, 0, 50, "cdr"),
    (dr.le, 0, 60, "le"),
    (dr.lmf, 0.75, 1.75, "lmf"),
    (dr.lmp, 0.75, 1.75, "lmp"),
    (dr.lmhs, 0.75, 1.75, "lmhs"),
    (dr.lmc, 0.75, 1.75, "lmc"),
    (br.tf, 0, 8, "tf"),
    (br.dtf, 0, 8, "dtf"),
]
@variables t
```

For each variable of the model, the above vector includes a quadruple, containing the Julia variable, its range, and its symbolic name to be shown in the plot (the range and the symbolic name are optional). The time variable `t` has also to be declared.

Finally, we can plot the evolution of the model variables according to the previously computed solution.

```
plotvariables(sol, (t, 1900, 1970), fig_7_2_variables, name="Fig. 7-2", showlegend=true, colored=true)
```
## Replicating the reference behaviour

To replicate the figures in Section 7.3 of the above book, we can operate in a similar way by declaring the varibales to be plot and by changing the time range. For example the following code reproduce the plot of Figure 7-7.

```
@named pop = WorldDynamics.World3.Pop4.population()
@named br = WorldDynamics.World3.Pop4.birth_rate()
@named dr = WorldDynamics.World3.Pop4.death_rate()
@named is = WorldDynamics.World3.Capital.industrial_subsector()
@named ld = WorldDynamics.World3.Agriculture.land_development()
@named nr = WorldDynamics.World3.NonRenewable.non_renewable()
@named pp = WorldDynamics.World3.Pollution.persistent_pollution()
reference_variables = [
    (nr.nrfr, 0, 1, "nrfr"),
    (is.iopc, 0, 1000, "iopc"),
    (ld.fpc, 0, 1000, "fpc"),
    (pop.pop, 0, 16e9, "pop"),
    (pp.ppolx, 0, 32, "ppolx"),
    (br.cbr, 0, 50, "cbr"),
    (dr.cdr, 0, 50, "cdr"),
]
@variables t
plotvariables(sol, (t, 1900, 2100), reference_variables, name="Fig. 7-7", showlegend=true, colored=true)
```

## Replicating the sensitivity tests

In this case, we have to mdoify the parameter or the interpolation table of the varaible with respect to which we want to perform the sensistivity test, we have to solve agian the ODE system, and we can finally plot the resulting evolution of the system.

### Modifying a parameter

In order to reproduce Figure 7-10, in which the nonrenewable resources initial value (that is, the value of the `NRI` parameter) is doubled, we can modify the value of this parameter by getting the parameter set of the nonrenewable resources sector, and by changing the value of `NRI`.

```
nonrenewable_parameters_7_10 = WorldDynamics.World3.NonRenewable.getparameters()
nonrenewable_parameters_7_10[:nri] = 2.0 * nonrenewable_parameters_7_10[:nri]
```

We then have to solve again the ODE system, by specifying which set of paramer values has to be used for the nonrenewable resources sector.

```
system = world3_historicalrun(nonrenewable_params=nonrenewable_parameters_7_10)
sol = WorldDynamics.solve(system, (1900, 2100))
```

Finally, we can plot the seven variables of Figure 7-10.

```
plotvariables(sol, (t, 1900, 2100), reference_variables, name="Fig. 7-10", showlegend=true, colored=true)
```

### Modifying an interpolation table

In order to reproduce Figure 7-13, in which the slope of the fraction of industrial output allocated to agriculture is increased, we can modify the two tables `FIOAA1` and `FIOAA2` by getting the table set of the agriculture sector, and by changing the value of these two tables.

```
agriculture_tables_7_13 = WorldDynamics.World3.Agriculture.gettables()
agriculture_tables_7_13[:fioaa1] = (0.5, 0.3, 0.1, 0.0, 0.0, 0.0)
agriculture_tables_7_13[:fioaa2] = (0.5, 0.3, 0.1, 0.0, 0.0, 0.0)
```

We then have to solve again the ODE system, by specifying which set of tables has to be used for the agriculture sector.

```
system = world3_historicalrun(agriculture_tables=agriculture_tables_7_13)
sol = WorldDynamics.solve(system, (1900, 2100))
```

Finally, we can plot the seven variables of Figure 7-10.

```
plotvariables(sol, (t, 1900, 2100), reference_variables, name="Fig. 7-13", showlegend=true, colored=true)
```
