import matplotlib.pyplot as plt
import numpy as np
import pandas as pd
import os
from os import listdir
from os.path import isfile, join
from scipy.interpolate import *
import json

plt.rcParams.update({'font.size': 14})
plt.close('all')

def getAverageResponse(srcDir, prefix, N):
	"""
	Args:
		srcDir contains all files
		prefix a string in filename to load
		N number of sampling point
	Returns:
		average strain and stress
	"""
	onlyfiles = [f for f in listdir(srcDir) if isfile(join(srcDir, f)) and prefix == f[:len(prefix)]]
	if len(onlyfiles) ==0:
		print(f"three is no data in {srcDir} with prefix {prefix}")
		return None, None
	data = pd.read_csv(join(srcDir, onlyfiles[0]))	
	xdata = data['EngStrain'].values
	ydata = data['EngStress'].values
	XX = xdata[np.logical_not(np.isnan(ydata))]
	YY = ydata[np.logical_not(np.isnan(ydata))]
	fct = interp1d(XX, YY)
	epsData = np.linspace(XX[0],XX[-1],N)
	sigData = fct(epsData)
	numTest = 1
	for f in onlyfiles[1:]:
		numTest = numTest+1
		data = pd.read_csv(join(srcDir, f))	
		xdata = data['EngStrain'].values
		ydata = data['EngStress'].values
		XX = xdata[np.logical_not(np.isnan(ydata))]
		YY = ydata[np.logical_not(np.isnan(ydata))]
		fct = interp1d(XX, YY, bounds_error=False)
		sigData += fct(epsData)
	sigData *=(1./numTest)
	return epsData, sigData

""" J2-plasticity comparison
"""
plt.figure()
#1L
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-1L","exp",30)
plt.plot(epsEng, sigEng,"ro",label='Experiment, 1L')
L0 = 25.
S0 = 0.95*3
uZ = pd.read_csv('../simulationsData/1L/NodalDisplacementPhysical21Num3comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/1L/force24comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot(eps,sig,'r-',label='GTN-Th-Sh, 1L', linewidth=2)
#2PL
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-2PL","exp",30)
plt.plot(epsEng, sigEng,"gv",label='Experiment, 2PL')
L0 = 30.
S0 = 0.95*2
uZ = pd.read_csv('../simulationsData/2PL/NodalDisplacementPhysical21Num2comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/2PL/force24comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot(eps,sig,'g--',label='GTN-Th-Sh, 2PL',linewidth=2)
#3SHEAR
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-3SHEAR","exp",30)
plt.plot(epsEng, sigEng,"bs",label='Experiment, 3SHEAR')
L0 = 50.
S0 = 0.5*0.95*4 # half thickness
uZ = pd.read_csv('../simulationsData/3SHEAR/NodalDisplacementPhysical21Num10comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/3SHEAR/force24comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot([0]+list(eps),[0]+list(sig),'b-.',label='GTN-Th-Sh, 3SHEAR',linewidth =2.)

plt.xlabel('Engineering strain')
plt.ylabel('Engineering stress (MPa)')
plt.legend(loc='center',fontsize=12, bbox_to_anchor=(0.65,0.25),)
plt.savefig('l40EngineeringCurves1L2PL.eps',bbox_inches="tight")
os.system('epstopdf l40EngineeringCurves1L2PL.eps')
plt.show()

plt.figure()
#2NS1
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-2NS1","exp",30)
plt.plot(epsEng, sigEng,'ro',label='Experiment, 2NS1')
L0 = 11.
S0 = 0.95*2
uZ = pd.read_csv('../simulationsData/2NS1/NodalDisplacementPhysical21Num2comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/2NS1/force24comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot([0]+eps.tolist(), [0]+sig.tolist(),'r-',label='GTN-Th-Sh, 2NS1',linewidth=2)
#2NS2
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-2NS2","exp",30)
plt.plot(epsEng, sigEng,'gv',label='Experiment, 2NS2')
L0 = 11.
S0 = 0.95*2
uZ = pd.read_csv('../simulationsData/2NS2/NodalDisplacementPhysical21Num2comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/2NS2/force24comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot([0]+eps.tolist(), [0]+sig.tolist(),'g--',label='GTN-Th-Sh, 2NS2',linewidth=2)
#2NS4
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-2NS4","exp",30)
plt.plot(epsEng, sigEng,'bs',label='Experiment, 2NS4')
L0 = 11.
S0 = 0.95*2
uZ = pd.read_csv('../simulationsData/2NS4/NodalDisplacementPhysical21Num2comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/2NS4/force24comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot([0]+eps.tolist(), [0]+sig.tolist(),'b:',label='GTN-Th-Sh, 2NS4',linewidth=2)

plt.xlabel('Engineering strain')
plt.ylabel('Engineering stress (MPa)')
plt.legend(fontsize=12)
plt.savefig('l40EngineeringCurves2NS.eps',bbox_inches="tight")
os.system('epstopdf l40EngineeringCurves2NS.eps')
plt.show()

plt.figure()
#4SR
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-4SR","exp",30)
plt.plot(epsEng, sigEng,'rs',label='Experiment, 4SR')
R0 = 3.
L0 = 25.
S0 = np.pi*R0*R0
uZ = pd.read_csv('../simulationsData/4SR/NodalDisplacementPhysical4Num4comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/4SR/force105comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot(eps,sig,'r-',label='GTN-Th-Sh, 4SR',linewidth=2)
#5NR4
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-5NR4","exp",30)
plt.plot(epsEng, sigEng,'gv',label='Experiment, 5NR4')
R0 = 2.5
L0 = 25.
S0 = np.pi*R0*R0
uZ = pd.read_csv('../simulationsData/5NR4/NodalDisplacementPhysical4Num7comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/5NR4/force105comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot([0]+list(eps),[0]+list(sig),'g--',label='GTN-Th-Sh, 5NR4',linewidth=2)
plt.legend()
plt.xlabel('Engineering strain')
plt.ylabel('Engineering stress (MPa)')

plt.legend(fontsize=12)
plt.savefig('l40EngineeringCurvesBar.eps',bbox_inches="tight")
os.system('epstopdf l40EngineeringCurvesBar.eps')
plt.show()


plt.figure()
#7GR1
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-7GR1","exp_7G1_V_01",30)
plt.plot(epsEng, sigEng,'ro',label='Experiment, 7GR1')
L0 = 20.
S0 = 2.
uZ = pd.read_csv('../simulationsData/7GR1/NodalDisplacementPhysical4Num7comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/7GR1/force105comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot(eps,sig,'r-',label='GTN-Th-Sh, 7GR1',linewidth=2)
#7GR3
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-7GR3","exp",30)
plt.plot(epsEng, sigEng,'gv',label='Experiment, 7GR3')
L0 = 20.
S0 = 2.
uZ = pd.read_csv('../simulationsData/7GR3/NodalDisplacementPhysical4Num7comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/7GR3/force105comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot(eps,sig,'g--',label='GTN-Th-Sh, 7GR3')
#7GR6
epsEng, sigEng = getAverageResponse("../experimentalCurves/exp-7GR6","exp",30)
plt.plot(epsEng, sigEng,'bs',label='Experiment, 7GR6',linewidth=2)
L0 = 20.
S0 = 2.
uZ = pd.read_csv('../simulationsData/7GR6/NodalDisplacementPhysical4Num7comp2.csv',sep=';',header=None) 
f = pd.read_csv('../simulationsData/7GR6/force105comp2.csv',sep=';',header=None)
eps = uZ.values[:,1]/L0
sig = -f.values[:,1]/S0
plt.plot(eps,sig,'b:',linewidth=2,label='GTN-Th-Sh, 7GR6')
plt.legend(ncol=1, loc ='center', bbox_to_anchor=(0.33,0.25), fontsize=12)
plt.xlabel('Engineering strain')
plt.ylabel('Engineering stress (MPa)')

plt.savefig('l40EngineeringCurves7GR.eps',bbox_inches="tight")
os.system('epstopdf l40EngineeringCurves7GR.eps')
plt.show()

