function plotSettings = plot_properties(plotSettings)

%Small ui where the user can change the plot settings of TrackIt.
%
%plotSettings = plot_properties(plotSettings)
%
%Input and output:
%   plotSettings: Struct containing the following entries
%     'invertColMap',         Invert pixel colormap
%     'bgColor',              LuT pixel color for 0 value
%     'trackLinewidth',       Plotted track linewidth
%     'trackMarkerSize',      Size of the plotted spot points in a track
%     'initialPosMarker',     Marker for initial position
%     'roiColor',             Color of the tracking region
%     'roiLinewidth',         Tracking region linewidth
%     'spotColor',            Color of the plotted spots
%     'spotMarkerSize',       Size of the plotted spots
%     'spotMarker',           Marker of the plotted spots
%     'singleColor',          Color in which non-linked spots are plotted
%     'singleMarkerSize',     Size in which events < min. track length are plotted
%     'singleMarker',         Marker of events < min. track length
%     'frameRate',            Rate at which frames are shown if the "Play" button is pressed
%     'sbDim',                Scalebar dimension in pixels
%     'sbPos',                Position of the lower left corner of the scalebar in pixels
%     'sbTextString',         Scalebar text
%     'sbTextPos',            Position of the scalebar text
%     'sbTextSize',           Size of the scalebar text
%     'scalingFactor',        Scaling factor at which "Detection mapping" or "Jump distance mapping" is plotted
%     'nFramesTrackIsVisible',Number of frames a track is still visible after it disappeared
%     'ITM',                  Wether to use an interlaced timelapse scheme to distinguish long and short bound molecules
%     'subRoiColors',         Color order of the sub-regions
%     'curTrackSel',          Track Id of the currently selected track
%     'tracksInFrame',        Matrix of logicals indicating which track has to be shown in each frame
%     'trackColors',          List of the colors for each track

S = CreatePlotPropertiesUI();
read_curPlot()

uiwait(S.f);

    function S = CreatePlotPropertiesUI()
        
        textPosHor = .05;
        textHeight = .03;
        textWidth = .5;
        editWidth = .15;
        editPosHor = 0.6;
        editHeight = textHeight;
        vertDist = 0.062;
        
        S.f = figure('Units','normalized',...
            'Position',[0.1 0.2 .2 .55],...
            'MenuBar','None',...
            'Name','Advanced plot properties',...
            'NumberTitle','off',...
            'WindowKeyPressFcn',@KeyPressFcnCB,...
            'CloseRequestFcn',@CloseRequestCB);
        
        textInvertCmap = uicontrol('String','Invert colormap',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist  textWidth  textHeight]); 
        
        S.checkBoxInvertCmap = uicontrol('String','',...
            'Style','checkbox',...
            'Units','normalized',...
            'Tag','lineWidth',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist editWidth editHeight]);  
        
        textBgColor = uicontrol('String','Zero value pixel color',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*2  textWidth  textHeight]); 
        
        S.popBgColor = uicontrol('String',{'black','white','colormap standard'},...
            'Style','popupmenu',...
            'Units','normalized',...
            'Tag','lineWidth',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*2 editWidth+.15 editHeight]);  
        
        textLinewidth = uicontrol('String','Track linewidth',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*3  textWidth  textHeight]); 
        
        S.editLinewidth = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...
            'Tag','lineWidth',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*3 editWidth editHeight]);  
        
        textTrackMarkerSize = uicontrol('String','Track markersize',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*4  textWidth  textHeight]); 
        
        S.editTrackMarkerSize = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...            
            'Tag','trackMarkerSize',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*4 editWidth editHeight]);  
        
        textSpotMarker = uicontrol('String','Spot marker (+, o, *, ., x, square,...)',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*5  textWidth  textHeight]); 
        
        S.editSpotMarker = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...
            'Tag','spotMarker',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*5 editWidth editHeight]); 
        
        textSpotMarkerSize = uicontrol('String','Spot markersize',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*6  textWidth  textHeight]); 
        
        S.editSpotMarkerSize = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...
            'Tag','spotMarkerSize',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*6 editWidth editHeight]); 
        
        textSpotColor = uicontrol('String','Spot color (r, g, b, c, m, y, k, w)',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*7  textWidth  textHeight]); 
        
        S.editSpotColor = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...
            'Tag','spotColor',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*7 editWidth editHeight]); 
        
        textRoiColor = uicontrol('String','Tracking ROI color',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*8  textWidth  textHeight]); 
        
        S.editRoiColor = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...
            'Tag','roiColor',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*8 editWidth editHeight]); 
        
        textRoiLinewidth = uicontrol('String','Tracking ROI linewidth',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*9  textWidth  textHeight]); 
        
        S.editRoiLinewidth = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...
            'Tag','roiLinewidth',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*9 editWidth editHeight]); 
                
        textSingleMarker = uicontrol('String','Non-linked spots marker',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*10 textWidth  textHeight]); 
        
        S.editSingleMarker = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...
            'Tag','singleMarker',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*10 editWidth editHeight]); 
                
        textSingleMarkerSize = uicontrol('String','Non-linked spots markersize',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*11  textWidth  textHeight]); 
        
        S.editSingleMarkerSize = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...
            'Tag','singleMarkerSize',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*11 editWidth editHeight]);
        
        textSingleColor = uicontrol('String','Non-linked spots color',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*12 textWidth  textHeight]); 
        
        S.editSingleColor = uicontrol('String','',...
            'Style','Edit',...
            'Units','normalized',...
            'Tag','singleColor',...
            'Callback',@editCB,...
            'Position',[editPosHor 1-vertDist*12 editWidth editHeight]);
                
        textITM = uicontrol('String','Interlaced illumination (ITM)',...
            'Style','Text',...
            'Units','normalized',...
            'HorizontalAlignment','Left',...
            'Position',[textPosHor  1-vertDist*13  textWidth  textHeight]); 
        
        S.cboxITM = uicontrol('String','',...
            'Style','checkbox',...
            'Units','normalized',...
            'Tag','lineWidth',...
            'Position',[editPosHor 1-vertDist*13 editWidth editHeight]);  
        
                        
        okButton = uicontrol('String','OK',...
            'Units','normalized',...
            'Position',[.25 0.02 textWidth textHeight*1.5],...
            'Callback',@OkCB);
        
    end


    function read_curPlot()
        
        S.checkBoxInvertCmap.Value = plotSettings.invertColMap;

        if strcmp(plotSettings.bgColor, 'black')
            S.popBgColor.Value = 1;    
        elseif strcmp(plotSettings.bgColor, 'white')
            S.popBgColor.Value = 2;      
        elseif strcmp(plotSettings.bgColor, 'colormap standard')
            S.popBgColor.Value = 3; 
        end
        
        S.editRoiColor.String = plotSettings.roiColor;
        S.editRoiLinewidth.String = plotSettings.roiLinewidth;
        S.editLinewidth.String = num2str(plotSettings.trackLinewidth);
        S.editTrackMarkerSize.String = num2str(plotSettings.trackMarkerSize);
        S.editSpotMarker.String = plotSettings.spotMarker;
        S.editSpotMarkerSize.String = num2str(plotSettings.spotMarkerSize);
        S.editSpotColor.String = plotSettings.spotColor;        
        S.editSingleMarker.String = plotSettings.singleMarker;
        S.editSingleMarkerSize.String = num2str(plotSettings.singleMarkerSize);
        S.editSingleColor.String = plotSettings.singleColor;        
        S.cboxITM.Value = plotSettings.ITM;
    end

    function CloseRequestCB(~,~)
        plotSettings = [];
        delete(gcf)
    end

    function KeyPressFcnCB(~,event)
        if strcmp(event.Key, 'escape')
            CloseRequestCB()
        end
    end

    function editCB(src,~)

        switch src.Tag   
            case 'roiColor' %spot Color
                if ~any(strcmp(S.editRoiColor.String,{'r','g','b','c','m','y','k','w'}))
                    src.String = plotSettings.roiColor;
                end
            case 'roiLinewidth' %track Linewidth
                if ~ge(str2double(S.editRoiLinewidth.String),0.01)
                    src.String = num2str(plotSettings.roiLinewidth);
                end
            case 'lineWidth' %track Linewidth
                if ~ge(str2double(S.editLinewidth.String),0.01)
                    src.String = num2str(plotSettings.trackLinewidth);
                end
            case 'trackMarkerSize' %track MarkerSize
                if ~ge(str2double(S.editTrackMarkerSize.String),0.01)
                    src.String = '0.01';
                end
            case 'spotMarker' %spot Marker
                if ~any(strcmp(S.editSpotMarker.String,{'+','o','*','.','x','square','diamond','v','^','>','<'}))
                    src.String = plotSettings.spotMarker;
                end
            case 'spotMarkerSize' %spot MarkerSize
                if ~ge(str2double(S.editSpotMarkerSize.String),0.01)
                    src.String = num2str(plotSettings.spotMarkerSize);
                end
            case 'spotColor' %spot Color
                if ~any(strcmp(S.editSpotColor.String,{'r','g','b','c','m','y','k','w'}))
                    src.String = plotSettings.spotColor;
                end
            case 'singleMarker' %spot Marker
                if ~any(strcmp(S.editSingleMarker.String,{'+','o','*','.','x','square','diamond','v','^','>','<'}))
                    src.String = plotSettings.spotMarker;
                end
            case 'singleMarkerSize' %spot MarkerSize
                if ~ge(str2double(S.editSingleMarkerSize.String),0.01)
                    src.String = num2str(plotSettings.spotMarkerSize);
                end
            case 'singleColor' %spot Color
                if ~any(strcmp(S.editSingleColor.String,{'r','g','b','c','m','y','k','w'}))
                    src.String = plotSettings.spotColor;
                end
        end
    end

    function OkCB(~,~)
        
        plotSettings.roiColor = S.editRoiColor.String;
        plotSettings.roiLinewidth = str2double(S.editRoiLinewidth.String);
        plotSettings.bgColor = S.popBgColor.String{S.popBgColor.Value};
        plotSettings.invertColMap = S.checkBoxInvertCmap.Value;
        plotSettings.trackLinewidth = str2double(S.editLinewidth.String);
        plotSettings.trackMarkerSize = str2double(S.editTrackMarkerSize.String);
        plotSettings.spotMarker = S.editSpotMarker.String;
        plotSettings.spotMarkerSize = str2double(S.editSpotMarkerSize.String);
        plotSettings.spotColor = S.editSpotColor.String;
        
        
        plotSettings.singleMarker = S.editSingleMarker.String;
        plotSettings.singleMarkerSize = str2double(S.editSingleMarkerSize.String);
        plotSettings.singleColor = S.editSingleColor.String;

        plotSettings.ITM = S.cboxITM.Value;
        delete(gcf)
    end

 
end
