function rom = ml_projtrunc_improper(sys, Vinf, Winf)
%ML_PROJTRUNC_IMPROPER Compute reduced improper model via projection.
%
% SYNATX:
%   rom = ML_PROJTRUNC_IMPROPER(sys, Vinf, Winf)
%
% DESCRIPTION:
%   This function computes reduced models of differential-algebraic
%   constraints using the projection approach and the basis matrices given
%   in Vinf and Winf such that
%
%       Ar{k} = Winf{k}' * Ainf * Vinf{k}, Br{k} = Winf{k}' * Binf,
%       Cr{k} = Cinf * Vinf{k}, Er{k} = Winf{k}' * Einf * Vinf{k}.
%
% INPUTS:
%   sys - structure, containing the improper descriptor system in the form:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Einf            | matrix with dimensions ninf x ninf                |
%   +-----------------+---------------------------------------------------+
%   | Ainf            | matrix with dimensions ninf x ninf                |
%   +-----------------+---------------------------------------------------+
%   | Binf            | matrix with dimensions ninf x m                   |
%   +-----------------+---------------------------------------------------+
%   | Cinf            | matrix with dimensions p x ninf                   |
%   +-----------------+---------------------------------------------------+
%   Vinf - matrix of dimensions ninf x r, basis of right projection space
%   {!}
%   Winf - matrix of dimensions ninf x r, basis of left projection space
%   {!}
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom - struct, containing the reduced-order matrices as:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | E               | matrix with dimensions r x r                      |
%   +-----------------+---------------------------------------------------+
%   | A               | matrix with dimensions r x r                      |
%   +-----------------+---------------------------------------------------+
%   | B               | matrix with dimensions r x m                      |
%   +-----------------+---------------------------------------------------+
%   | C               | matrix with dimensions p x r                      |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_projtrunc_proper, ml_balproj_improper.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(3, 3);

ml_assert_infdescsys(sys);

if isa(Vinf, 'cell')
    numRoms = length(Vinf);
else
    Vinf    = {Vinf};
    numRoms = 1;
end

if not(isa(Winf, 'cell')), Winf = {Winf}; end

assert(length(Winf) == numRoms, ...
    'MORLAB:data', ...
    'There must be as many Winf as Vinf matrices.');

ninf = size(sys.Ainf, 1);

for k = 1:numRoms
    assert(size(Vinf{k}, 1) == ninf, ...
        'MORLAB:data', ...
        'The matrix Vinf must have the same number of rows as sys.Ainf!');

    assert(size(Winf{k}, 1) == ninf, ...
        'MORLAB:data', ...
        'The matrix Winf must have the same number of rows as sys.Ainf!');

    assert(size(Vinf{k}, 2) == size(Winf{k}, 2), ...
        'MORLAB:data', ...
        ['The matrices Vinf and Winf must have the same number of' ...
        ' columns!']);

    if issparse(Vinf{k}), Vinf{k} = full(Vinf{k}); end
    if issparse(Winf{k}), Winf{k} = full(Winf{k}); end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF REDUCED-ORDER MODELS.                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if numRoms == 1
    rom = struct( ...
        'A', Winf{k}' * (sys.Ainf * Vinf{k}), ...
        'B', Winf{k}' * sys.Binf, ...
        'C', sys.Cinf * Vinf{k}, ...
        'E', Winf{k}' * (sys.Einf * Vinf{k}));
else
    rom = cell(1, numRoms);
    for k = 1:numRoms
        rom{k} = struct( ...
            'A', Winf{k}' * (sys.Ainf * Vinf{k}), ...
            'B', Winf{k}' * sys.Binf, ...
            'C', sys.Cinf * Vinf{k}, ...
            'E', Winf{k}' * (sys.Einf * Vinf{k}));
    end
end
