function sys = ml_prepare_system_data(sys)
%ML_PREPARE_SYSTEM_DATA Densification and implicit projection of matrices.
%
% SYNTAX:
%   sys = ML_PREPARE_SYSTEM_DATA(sys)
%
% DESCRIPTION:
%   This function is used in classical model reduction routines to prepare
%   the system data such that the right-hand sides of matrix equations are
%   dense matrices and, in the case of sparse DAE systems, the input and
%   output matrices of the hidden manifold are pre-computed and stored.
%   The function ml_decide_system_type must be run before.
%
% INPUTS:
%   sys - struct, containing the original system matrices
%
% OUTPUTS:
%   sys - struct, containing the edited system matrices, in the case of
%         sparse systems, the edited (projected) versions of the matrices
%         are stored as 'p', e.g., sys.pB is the edited version of sys.B.
%         The flag sys.PreparedSystemData is used to determine if the
%         function has already been run.
%
%
% See also ml_decide_system_type, ml_format_output.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 1);

assert(isa(sys, 'struct'), ...
    'MORLAB:data', ...
    'The input argument must be a struct.');

assert(ml_field_set_to_value(sys, 'SystemType') ...
    && ml_field_set_to_value(sys, 'DecidedSystemType') ...
    && sys.DecidedSystemType, ...
    'MORLAB:data', ...
    'Run first ml_decide_system_type.');

if ml_field_set_to_value(sys, 'PreparedSystemData') ...
        && sys.PreparedSystemData
    return;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EDIT SYSTEM MATRICES.                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

switch lower(sys.SystemType)
    case {'ct_d_ss', 'dt_d_ss'}
        sys.A = full(sys.A);
        sys.B = full(sys.B);
        sys.C = full(sys.C);
        sys.D = full(sys.D);

    case {'ct_d_dss', 'dt_d_dss'}
        sys.A = full(sys.A);
        sys.B = full(sys.B);
        sys.C = full(sys.C);
        sys.D = full(sys.D);
        sys.E = full(sys.E);

    case {'ct_d_soss', 'dt_d_soss'}
        sys.M  = full(sys.M);
        sys.E  = full(sys.E);
        sys.K  = full(sys.K);
        sys.Bu = full(sys.Bu);
        sys.Cp = full(sys.Cp);
        sys.Cv = full(sys.Cv);
        sys.D  = full(sys.D);

        sys.pB = [zeros(size(sys.Bu)); sys.Bu];

        if isempty(sys.Cp)
            sys.pC = [zeros(size(sys.Cv)), sys.Cv];
        elseif isempty(sys.Cv)
            sys.pC = [sys.Cp, zeros(size(sys.Cp))];
        else
            sys.pC = [sys.Cp, sys.Cv];
        end

        sys.pD = sys.D;

    case {'ct_s_ss_default', 'ct_s_dss_default', ...
            'dt_s_ss_default', 'dt_s_dss_default'}
        sys.pB = full(sys.B);
        sys.pC = full(sys.C);
        sys.pD = full(sys.D);

    case {'ct_s_dss_dae_1', 'dt_s_dss_dae_1'}
        one = sys.nE;
        two = sys.zE;

        tmpB = full(sys.B(one, :) - sys.A(one, two) * (sys.A(two, two) ...
            \ sys.B(two, :)));
        tmpC = full(sys.C(:,one) - (sys.C(:, two) / sys.A(two, two)) ...
            * sys.A(two, one));

        if ml_field_set_to_value(sys, 'D')
            tmpD = full(sys.D - sys.C(:, two) * (sys.A(two, two) ...
                \ sys.B(two, :)));
        else
            tmpD = -full(sys.C(:, two) * (sys.A(two, two) ...
                \ sys.B(two, :)));
        end

        sys.pB = tmpB;
        sys.pC = tmpC;
        sys.pD = tmpD;

    case {'ct_s_dss_dae_2', 'dt_s_dss_dae_2'}
        one = sys.nE;
        two = sys.zE;

        tmpB = full(sys.B(one, :));
        tmpC = full(sys.C(:, one));
        tmpD = [];

        if any(any(sys.B(two, :)))
            tmpEA = sys.A(two, one) * (sys.E(one, one) \ sys.A(one, two));
            tmpB  = tmpB - sys.A(one, one) * (sys.E(one, one) ...
                \ (sys.A(one, two) * (tmpEA \ sys.B(two, :))));
            tmpD  = -sys.C(:, one) * (sys.E(one, one) ...
                \ (sys.A(one, two) * (tmpEA \ sys.B(two, :))));
        elseif any(any(sys.C(:, two)))
            tmpEA = sys.A(two, one) * (sys.E(one, one) \ sys.A(one, two));
            tmpC  = tmpC - (((sys.C(:, two) / tmpEA) * sys.A(two, one)) ...
                / sys.E(one, one)) * sys.A(one, one);
            tmpD  = -((sys.C(:, two) / tmpEA) * sys.A(two, one)) ...
                * (sys.E(one, one) \ sys.B(one, :));
        end

        if ml_field_set_to_value(sys, 'D')
            tmpD = sys.D + tmpD;
        end

        sys.pB = full(tmpB);
        sys.pC = full(tmpC);
        sys.pD = full(tmpD);

    case 'ct_s_soss_so_1'
        sys.pBu = full(sys.Bu);
        sys.pCp = full(sys.Cp);
        sys.pCv = full(sys.Cv);
        sys.pD  = full(sys.D);

        sys.pB = [zeros(size(sys.pBu)); sys.pBu];

        if isempty(sys.pCp)
            sys.pC = [zeros(size(sys.pCv)), sys.pCv];
        elseif isempty(sys.pCv)
            sys.pC = [sys.pCp, zeros(size(sys.pCp))];
        else
            sys.pC = [sys.pCp, sys.pCv];
        end

    case 'ct_s_soss_dae_1_so'
        one = sys.nM;
        two = sys.zM;

        if ml_field_set_to_value(sys, 'Cp')
            tmpCp = full(sys.Cp(:, one) - (sys.Cp(:, two) ...
                / sys.K(two, two)) * sys.K(two, one));

            if ml_field_set_to_value(sys, 'D')
                tmpD = full(sys.D + (sys.Cp(:, two) / sys.K(two, two)) ...
                    * sys.Bu(two, :));
            else
                tmpD = full((sys.Cp(:, two) / sys.K(two, two)) ...
                    * sys.Bu(two, :));
            end
        else
            tmpCp = full(sys.Cp);
            tmpD  = full(sys.D);
        end

        if ml_field_set_to_value(sys, 'Cv')
            tmpCv = full(sys.Cv(:, one) - (sys.Cv(:, two) ...
                / sys.K(two, two)) * sys.K(two, one));
        else
            tmpCv = full(sys.Cv);
        end

        tmpBu = full(sys.Bu(one, :) - sys.K(one, two) ...
            * (sys.K(two, two) \ sys.Bu(two, :)));

        sys.pBu = tmpBu;
        sys.pCp = tmpCp;
        sys.pCv = tmpCv;

        sys.pB = [tmpBu; zeros(size(tmpBu))];
        sys.pD = tmpD;

        if isempty(tmpCp)
            sys.pC = [zeros(size(tmpCv)), sys.pCv];
        elseif isempty(tmpCv)
            sys.pC = [tmpCp, zeros(size(tmpCp))];
        else
            sys.pC = [tmpCp, tmpCv];
        end

    case {'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        one = sys.nM;
        two = sys.zM;

        sys.pM = sys.M(one, one);
        sys.pE = -sys.E(one, one);
        sys.pG = sys.K(one, two)';
        sys.pK = -sys.K(one, one);

        sys.pBu = full(sys.Bu(one, :));

        if ml_field_set_to_value(sys, 'Cp')
            sys.pCp = full(sys.Cp(:, one));
        else
            sys.pCp = full(sys.Cp);
        end

        if ml_field_set_to_value(sys, 'Cv')
            sys.pCv = full(sys.Cv(:, one));
        else
            sys.pCv = full(sys.Cv);
        end

        sys.pD = full(sys.D);

        sys.pB = [zeros(size(sys.pBu)); sys.pBu];

        if isempty(sys.pCp)
            sys.pC = [zeros(size(sys.pCv)), sys.pCv];
        elseif isempty(sys.pCv)
            sys.pC = [sys.pCp, zeros(size(sys.pCp))];
        else
            sys.pC = [sys.pCp, sys.pCv];
        end

    otherwise
        error('MORLAB:data', ...
            'The given system type is not implemented.');
end

sys.PreparedSystemData = true;
