function [opts] = ml_morlabopts(fname)
%ML_MORLABOPTS Option struct generator for MORLAB functions.
%
% SYNTAX:
%   ML_MORLABOPTS(fname)
%   opts = ML_MORLABOPTS(fname)
%
% DESCRIPTION:
%   This function can be used to generate an option structure for the
%   MORLAB function specified by fname. The structure is filled with empty
%   arrays such that the default values of the MORLAB function are called.
%   For the choice of the default values, call this function without an
%   output argument or see the documentation of the corresponding function.
%
% INPUTS:
%   fname - character array, the name of the function for which the option
%           struct is requested
%
% OUTPUTS:
%   opts - struct, the desired option structure for a MORLAB function
%
% See also morlab.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUT.                                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 1);

assert(ischar(fname), ...
    'MORLAB:data', ...
    'The given function name has to be a character array.');

% Check if hyperlinks can be used.
hasHyperlink = usejava('desktop');

% Simplify case checks by restricting to lower case letters.
fname = lower(fname);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHOOSE FUNCTION.                                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign option structs.
switch fname
    case {'ml_assert_boolean', ...
            'ml_assert_char', ...
            'ml_assert_descsys', ...
            'ml_assert_infdescsys', ...
            'ml_assert_integer', ...
            'ml_assert_nonnegint', ...
            'ml_assert_nonnegintinf', ...
            'ml_assert_nonnegscalar', ...
            'ml_assert_nonnegvector', ...
            'ml_assert_posinteger', ...
            'ml_assert_posscalar', ...
            'ml_assert_scalar', ...
            'ml_assert_sosys', ...
            'ml_assert_stdsys', ...
            'ml_assert_vector', ...
            'ml_check_cell_param', ...
            'ml_check_cell_vector', ...
            'ml_check_krylovopts', ...
            'ml_check_messopts', ...
            'ml_field_set_to_value', ...
            'ml_citation', ...
            'ml_version'}
        if nargout == 0
            fprintf(1, [' The requested function doesn''t use an' ...
                ' option struct!\n']);
        else
            opts = struct([]);
        end

    case  {'morlab_data_desc_br', ...
            'morlab_data_desc_dstab', ...
            'morlab_data_desc_infdunstab', ...
            'morlab_data_desc_infunstab', ...
            'morlab_data_desc_pr', ...
            'morlab_data_so_stab', ...
            'morlab_data_std_br', ...
            'morlab_data_std_dstab', ...
            'morlab_data_std_dunstab', ...
            'morlab_data_std_pr', ...
            'morlab_data_std_unstab', ...
            'morlab_demo_additive_decomposition', ...
            'morlab_demo_equation_solvers', ...
            'morlab_demo_frequency_evaluation', ...
            'morlab_demo_mor_default', ...
            'morlab_demo_mor_secondorder', ...
            'morlab_demo_morlabopts', ...
            'morlab_demo_partial_stabilization', ...
            'morlab_demo_time_evaluation'}
        if nargout == 0
            fprintf(1, ' Scripts cannot use option structs!\n');
        else
            opts = struct([]);
        end

    case 'ml_cabe_sgn'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_caredl_sgn'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' EqnType: [ ''primal'' | ''dual'' | {''both''} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'EqnType', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_caredl_sgn_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {1.0e-02*sqrt(n*eps)} ]\n');
            fprintf(1, ' EqnType: [ ''primal'' | ''dual'' | {''both''} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...ml_morlabopts
                'AbsTol', [], ...
                'CompTol', [], ...
                'EqnType', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_care_nwt_fac'
        if nargout == 0
            fprintf(1, '   AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '     Info: [ 1 | {0} ]\n');
            fprintf(1, '       K0: [ matrix with dimensions m x n | {zeros(m, n)} ]\n');
            if hasHyperlink
                fprintf(1, ' lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyap_sgn_fac">ml_morlabopts(''ml_lyap_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, ' lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyap_sgn_fac'')} ]\n');
            end
            fprintf(1, '  MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '   RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol' , [], ...
                'Info'   , [], ...
                'K0'     , [], ...
                'lyapopts', ml_morlabopts('ml_lyap_sgn_fac'), ...
                'MaxIter', [], ...
                'RelTol' , []);
        end

    case 'ml_care_nwt_ldl'
        if nargout == 0
            fprintf(1, '   AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '     Info: [ 1 | {0} ]\n');
            fprintf(1, '       K0: [ matrix with dimensions m x n | {zeros(m, n)} ]\n');
            if hasHyperlink
                fprintf(1, ' lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyap_sgn_ldl">ml_morlabopts(''ml_lyap_sgn_ldl'')</a>} ]\n');
            else
                fprintf(1, ' lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyap_sgn_ldl'')} ]\n');
            end
            fprintf(1, '  MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '   RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol' , [], ...
                'Info'   , [], ...
                'K0'     , [], ...
                'lyapopts', ml_morlabopts('ml_lyap_sgn_ldl'), ...
                'MaxIter', [], ...
                'RelTol' , []);
        end

    case 'ml_daredl_sda'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' EqnType: [ ''primal'' | ''dual'' | {''both''} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'EqnType', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_daredl_sda_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {1.0e-02*sqrt(n*eps)} ]\n');
            fprintf(1, ' EqnType: [ ''primal'' | ''dual'' | {''both''} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'EqnType', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_dare_nwt_fac'
        if nargout == 0
            fprintf(1, '    AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '      Info: [ 1 | {0} ]\n');
            fprintf(1, '        K0: [ matrix with dimensions m x n | {zeros(m, n)} ]\n');
            if hasHyperlink
                fprintf(1, ' dlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dlyap_smith_fac">ml_morlabopts(''ml_dlyap_smith_fac'')</a>} ]\n');
            else
                fprintf(1, ' dlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_dlyap_smith_fac'')} ]\n');
            end
            fprintf(1, '   MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '    RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol'  , [], ...
                'Info'    , [], ...
                'K0'      , [], ...
                'dlyapopts', ml_morlabopts('ml_dlyap_smith_fac'), ...
                'MaxIter' , [], ...
                'RelTol'  , []);
        end

    case 'ml_dlyapdl_smith'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_dlyapdl_smith_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {1.0e-02*sqrt(n*eps)} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_dlyapdl_smith_ldl'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {sqrt(n)*eps} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_dlyap_smith'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_dlyap_smith_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {1.0e-02*sqrt(n*eps)} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_dlyap_smith_ldl'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {sqrt(n)*eps} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_dsylv_smith'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_dsylv_smith_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {sqrt(n)*eps} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_gdlyapdl_smith_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '   Index: [ nonnegative integer | {Inf} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Index' , [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_gdlyap_smith_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '   Index: [ nonnegative integer | {Inf} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Index' , [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_icare_ric_fac'
        if nargout == 0
            fprintf(1, '   AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '  CompTol: [ nonnegative scalar | {1.0e-02*sqrt(n*eps)} ]\n');
            fprintf(1, '     Info: [ 1 | {0} ]\n');
            if hasHyperlink
                fprintf(1, ' careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, ' careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac'')} ]\n');
            end
            fprintf(1, '  MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '   RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
            fprintf(1, ' StoreLQG: [ 1 | {0} ]\n');
            fprintf(1, '       Z0: [ matrix of dimensions n x k | {[]} ]\n');
        else
            opts = struct( ...
                'AbsTol' , [], ...
                'CompTol', [], ...
                'Info'   , [], ...
                'careopts', ml_morlabopts('ml_caredl_sgn_fac'), ...
                'MaxIter', [], ...
                'RelTol' , [], ...
                'StoreLQG', [], ...
                'Z0'     , []);
        end

    case 'ml_lyapdl_sgn'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_lyapdl_sgn_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {1.0e-02*sqrt(n*eps)} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_lyapdl_sgn_ldl'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {sqrt(n)*eps} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_lyap_sgn'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_lyap_sgn_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {1.0e-02*sqrt(n*eps)} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_lyap_sgn_ldl'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {sqrt(n)*eps} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_pcare_nwt_fac'
        if nargout == 0
            fprintf(1, '   AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '  CompTol: [ nonnegative scalar | {1.0e-02*sqrt(n*eps)} ]\n');
            fprintf(1, '     Info: [ 1 | {0} ]\n');
            if hasHyperlink
                fprintf(1, ' lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyap_sgn_fac">ml_morlabopts(''ml_lyap_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, ' lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyap_sgn_fac'')} ]\n');
            end
            fprintf(1, '  MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '   RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol' , [], ...
                'CompTol', [], ...
                'Info'   , [], ...
                'lyapopts', ml_morlabopts('ml_lyap_sgn_fac'), ...
                'MaxIter', [], ...
                'RelTol' , []);
        end

    case 'ml_sylv_sgn'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_sylv_sgn_fac'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, ' CompTol: [ nonnegative scalar | {sqrt(n)*eps} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+01*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'CompTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_bodemag'
        if nargout == 0
            fprintf(1, ' AccuracyTol: [ nonnegative scalar | {0.1} ]\n');
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '   FreqRange: [ vector of length 2 | {[-4, 4]} ]\n');
            fprintf(1, '  FreqSample: [ ''equal'' | [numeric] | {''adapt''} ]\n');
            fprintf(1, '    FreqUnit: [ ''Hz'' | {''rad/sec''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '  InitPoints: [ positive integer >= 3 or vector of length >= 3 | {3} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    MagScale: [ ''linear'' | {''log''} ]\n');
            fprintf(1, '     MagUnit: [ ''dB'' | {''abs''} ]\n');
            fprintf(1, '   MaxPoints: [ positive integer | {500} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
        else
            opts = struct( ...
                'AccuracyTol', [], ...
                'DiffMode'  , [], ...
                'FreqRange' , [], ...
                'FreqSample', [], ...
                'FreqUnit'  , [], ...
                'Info'      , [], ...
                'InitPoints', [], ...
                'LineSpecs' , [], ...
                'MagScale'  , [], ...
                'MagUnit'   , [], ...
                'MaxPoints' , [], ...
                'ShowPlot'  , []);
        end

    case 'ml_ct_dss_simulate_ie'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '  TimePoints: [ positive integer | {500} ]\n');
            fprintf(1, '   TimeRange: [ vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '  TimeSample: [ ''equalstep'' | [numeric] | {''equalpts''} ]\n');
            fprintf(1, '    TimeStep: [ scalar | {(tf - t0) / 500} ]\n');
            fprintf(1, '          X0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'TimePoints', [], ...
                'TimeRange' , [], ...
                'TimeSample', [], ...
                'TimeStep'  , [], ...
                'X0'        , []);
        end

    case 'ml_ct_dss_simulate_ss11'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '       Theta: [ nonnegative scalar | {0.5} ]\n');
            fprintf(1, '  TimePoints: [ positive integer | {500} ]\n');
            fprintf(1, '   TimeRange: [ vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '  TimeSample: [ ''equalstep'' | [numeric] | {''equalpts''} ]\n');
            fprintf(1, '    TimeStep: [ scalar | {(tf - t0) / 500} ]\n');
            fprintf(1, '          X0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'Theta'     , [], ...
                'TimePoints', [], ...
                'TimeRange' , [], ...
                'TimeSample', [], ...
                'TimeStep'  , [], ...
                'X0'        , []);
        end

    case 'ml_ct_dss_simulate_ss21'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '      Theta1: [ nonnegative scalar | {0.5} ]\n');
            fprintf(1, '      Theta2: [ nonnegative scalar | {0.5} ]\n');
            fprintf(1, '  TimePoints: [ positive integer | {500} ]\n');
            fprintf(1, '   TimeRange: [ vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '  TimeSample: [ ''equalstep'' | [numeric] | {''equalpts''} ]\n');
            fprintf(1, '    TimeStep: [ scalar | {(tf - t0) / 500} ]\n');
            fprintf(1, '          X0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'Theta1'    , [], ...
                'Theta2'    , [], ...
                'TimePoints', [], ...
                'TimeRange' , [], ...
                'TimeSample', [], ...
                'TimeStep'  , [], ...
                'X0'        , []);
        end

    case 'ml_ct_soss_simulate_ie'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '  TimePoints: [ positive integer | {500} ]\n');
            fprintf(1, '   TimeRange: [ vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '  TimeSample: [ ''equalstep'' | [numeric] | {''equalpts''} ]\n');
            fprintf(1, '    TimeStep: [ scalar | {(tf - t0) / 500} ]\n');
            fprintf(1, '         Xp0: [ vector | {zeros(n, 1)} ]\n');
            fprintf(1, '         Xv0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'TimePoints', [], ...
                'TimeRange' , [], ...
                'TimeSample', [], ...
                'TimeStep'  , [], ...
                'Xp0'       , [], ...
                'Xv0'       , []);
        end

    case 'ml_ct_soss_simulate_ss22'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '      Theta1: [ nonnegative scalar | {0.5} ]\n');
            fprintf(1, '      Theta2: [ nonnegative scalar | {0.5} ]\n');
            fprintf(1, '  TimePoints: [ positive integer | {500} ]\n');
            fprintf(1, '   TimeRange: [ vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '  TimeSample: [ ''equalstep'' | [numeric] | {''equalpts''} ]\n');
            fprintf(1, '    TimeStep: [ scalar | {(tf - t0) / 500} ]\n');
            fprintf(1, '         Xp0: [ vector | {zeros(n, 1)} ]\n');
            fprintf(1, '         Xv0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'Theta1'    , [], ...
                'Theta2'    , [], ...
                'TimePoints', [], ...
                'TimeRange' , [], ...
                'TimeSample', [], ...
                'TimeStep'  , [], ...
                'Xp0'       , [], ...
                'Xv0'       , []);
        end

    case 'ml_ct_ss_simulate_ie'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '  TimePoints: [ positive integer | {500} ]\n');
            fprintf(1, '   TimeRange: [ vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '  TimeSample: [ ''equalstep'' | [numeric] | {''equalpts''} ]\n');
            fprintf(1, '    TimeStep: [ scalar | {(tf - t0) / 500} ]\n');
            fprintf(1, '          X0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'TimePoints', [], ...
                'TimeRange' , [], ...
                'TimeSample', [], ...
                'TimeStep'  , [], ...
                'X0'        , []);
        end

    case 'ml_ct_ss_simulate_ss11'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '       Theta: [ nonnegative scalar | {0.5} ]\n');
            fprintf(1, '  TimePoints: [ positive integer | {500} ]\n');
            fprintf(1, '   TimeRange: [ vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '  TimeSample: [ ''equalstep'' | [numeric] | {''equalpts''} ]\n');
            fprintf(1, '    TimeStep: [ scalar | {(tf - t0) / 500} ]\n');
            fprintf(1, '          X0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'Theta'     , [], ...
                'TimePoints', [], ...
                'TimeRange' , [], ...
                'TimeSample', [], ...
                'TimeStep'  , [], ...
                'X0'        , []);
        end

    case 'ml_ct_ss_simulate_ss21'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '      Theta1: [ nonnegative scalar | {0.5} ]\n');
            fprintf(1, '      Theta2: [ nonnegative scalar | {0.5} ]\n');
            fprintf(1, '  TimePoints: [ positive integer | {500} ]\n');
            fprintf(1, '   TimeRange: [ vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '  TimeSample: [ ''equalstep'' | [numeric] | {''equalpts''} ]\n');
            fprintf(1, '    TimeStep: [ scalar | {(tf - t0) / 500} ]\n');
            fprintf(1, '          X0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'Theta1'    , [], ...
                'Theta2'    , [], ...
                'TimePoints', [], ...
                'TimeRange' , [], ...
                'TimeSample', [], ...
                'TimeStep'  , [], ...
                'X0'        , []);
        end

    case 'ml_dt_dss_simulate'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '   TimeRange: [ nonnegative vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '          X0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'TimeRange' , [], ...
                'X0'        , []);
        end

    case 'ml_dt_ss_simulate'
        if nargout == 0
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '    InputFcn: [ function handle | {ones(m, 1)} ]\n');
            fprintf(1, '  InputRange: [ vector of length <= 2 | {opts.TimeRange} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
            fprintf(1, ' StoreStates: [ 1 | {0} ]\n');
            fprintf(1, '   TimeRange: [ nonnegative vector of length <= 2 | {[0 10]} ]\n');
            fprintf(1, '          X0: [ vector | {zeros(n, 1)} ]\n');
        else
            opts = struct( ...
                'DiffMode'  , [], ...
                'Info'      , [], ...
                'InputFcn'  , [], ...
                'InputRange', [], ...
                'LineSpecs' , [], ...
                'ShowPlot'  , [], ...
                'StoreStates', [], ...
                'TimeRange' , [], ...
                'X0'        , []);
        end

    case 'ml_frobeniusplot'
        if nargout == 0
            fprintf(1, ' AccuracyTol: [ nonnegative scalar | {0.1} ]\n');
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '   FreqRange: [ vector of length 2 | {[-4, 4]} ]\n');
            fprintf(1, '  FreqSample: [ ''equal'' | [numeric] | {''adapt''} ]\n');
            fprintf(1, '    FreqUnit: [ ''Hz'' | {''rad/sec''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '  InitPoints: [ positive integer >= 3 or vector of length >= 3 | {3} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    MagScale: [ ''linear'' | {''log''} ]\n');
            fprintf(1, '     MagUnit: [ ''dB'' | {''abs''} ]\n');
            fprintf(1, '   MaxPoints: [ positive integer | {500} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
        else
            opts = struct( ...
                'AccuracyTol', [], ...
                'DiffMode'  , [], ...
                'FreqRange' , [], ...
                'FreqSample', [], ...
                'FreqUnit'  , [], ...
                'Info'      , [], ...
                'InitPoints', [], ...
                'LineSpecs' , [], ...
                'MagScale'  , [], ...
                'MagUnit'   , [], ...
                'MaxPoints' , [], ...
                'ShowPlot'  , []);
        end

    case 'ml_sigmaplot'
        if nargout == 0
            fprintf(1, ' AccuracyTol: [ nonnegative scalar | {0.1} ]\n');
            fprintf(1, '    DiffMode: [ ''abs'' | ''rel'' | ''all'' | {''off''} ]\n');
            fprintf(1, '   FreqRange: [ vector of length 2 | {[-4, 4]} ]\n');
            fprintf(1, '  FreqSample: [ ''equal'' | [numeric] | {''adapt''} ]\n');
            fprintf(1, '    FreqUnit: [ ''Hz'' | {''rad/sec''} ]\n');
            fprintf(1, '        Info: [ 1 | {0} ]\n');
            fprintf(1, '  InitPoints: [ positive integer >= 3 or vector of length >= 3 | {3} ]\n');
            fprintf(1, '   LineSpecs: [ character array | {''-''} ]\n');
            fprintf(1, '    MagScale: [ ''linear'' | {''log''} ]\n');
            fprintf(1, '     MagUnit: [ ''dB'' | {''abs''} ]\n');
            fprintf(1, '   MaxPoints: [ positive integer | {500} ]\n');
            fprintf(1, '    ShowPlot: [ 0 | {1} ]\n');
        else
            opts = struct( ...
                'AccuracyTol', [], ...
                'DiffMode'  , [], ...
                'FreqRange' , [], ...
                'FreqSample', [], ...
                'FreqUnit'  , [], ...
                'Info'      , [], ...
                'InitPoints', [], ...
                'LineSpecs' , [], ...
                'MagScale'  , [], ...
                'MagUnit'   , [], ...
                'MaxPoints' , [], ...
                'ShowPlot'  , []);
        end

    case 'ml_ct_d_dss_brbt'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_dss_bst'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyap_sgn_fac">ml_morlabopts(''ml_lyap_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyap_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'Epsilon'        , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'lyapopts'       , ml_morlabopts('ml_lyap_sgn_fac'), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_dss_bt'
        if nargout == 0
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_fac'), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_dss_flbt'
        if nargout == 0
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '        FreqRange: [ nonnegative vector | {[0, 1.0e+03]} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_ldl">ml_morlabopts(''ml_lyapdl_sgn_ldl'')</a> or <a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_ldl'') or ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '       ModGramian: [ 1 | {0} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''tolerance'' | {''sum''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'FreqRange'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_ldl'), ...
                'Method'         , [], ...
                'ModGramian'     , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_dss_hinfbt'
        if nargout == 0
            fprintf(1, '             Beta: [ nonnegative scalar | {0.1} ]\n');
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Gamma: [ positive scalar | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '       StabMethod: [ ''lyap'' | {''cabe''} ]\n');
            if hasHyperlink
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_cabe_sgn">ml_morlabopts(''ml_cabe_sgn'')</a> or <a href="matlab:ml_morlabopts ml_lyap_sgn">ml_morlabopts(''ml_lyap_sgn'')</a>} ]\n');
            else
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_cabe_sgn'') or ml_morlabopts(''ml_lyap_sgn'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'Beta'           , [], ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'Gamma'          , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StabMethod'     , [], ...
                'stabmethodopts' , ml_morlabopts('ml_cabe_sgn'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_dss_hna'
        if nargout == 0
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '    hankeldecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '    hankeldecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '        MinRelTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'hankeldecopts'  , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_fac'), ...
                'MinRelTol'      , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StoreGramians'  , [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_dss_krylov'
        if nargout == 0
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(SVs))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_ct_d_dss_lqgbt'
        if nargout == 0
            fprintf(1, '             Beta: [ nonnegative scalar | {0.1} ]\n');
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '       StabMethod: [ ''lyap'' | {''cabe''} ]\n');
            if hasHyperlink
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_cabe_sgn">ml_morlabopts(''ml_cabe_sgn'')</a> or <a href="matlab:ml_morlabopts ml_lyap_sgn">ml_morlabopts(''ml_lyap_sgn'')</a>} ]\n');
            else
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_cabe_sgn'') or ml_morlabopts(''ml_lyap_sgn'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'Beta'           , [], ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StabMethod'     , [], ...
                'stabmethodopts' , ml_morlabopts('ml_cabe_sgn'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_dss_mt'
        if nargout == 0
            fprintf(1, '           Alpha: [ real scalar | {-1.0} ]\n');
            fprintf(1, '       DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '       DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '      gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '      gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '       IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '       IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '   ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '           Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       signmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '       signmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '      infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '         RankTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '   StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'Alpha'         , [], ...
                'DecompEig'     , [], ...
                'DecompTol'     , [], ...
                'gdlyapopts'    , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'IGramFacC'     , [], ...
                'IGramFacO'     , [], ...
                'ImproperTrunc' , [], ...
                'Index'         , [], ...
                'signmopts'     , ml_morlabopts('ml_signm'), ...
                'infdecopts'    , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'RankTol'       , [], ...
                'StoreGramians' , [], ...
                'StoreProjection', []);
        end

    case 'ml_ct_d_dss_prbt'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'Epsilon'        , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_dss_tlbt'
        if nargout == 0
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_ldl">ml_morlabopts(''ml_lyapdl_sgn_ldl'')</a> or <a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_ldl'') or ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '       ModGramian: [ 1 | {0} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''tolerance'' | {''sum''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        TimeRange: [ nonnegative vector | {[0, 10]} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_ldl'), ...
                'Method'         , [], ...
                'ModGramian'     , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'TimeRange'      , [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_soss_brbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_soss_bst'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyap_sgn_fac">ml_morlabopts(''ml_lyap_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyap_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'Epsilon'        , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapopts'       , ml_morlabopts('ml_lyap_sgn_fac'), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_soss_bt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_fac'), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_soss_flbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_ldl">ml_morlabopts(''ml_lyapdl_sgn_ldl'')</a> or <a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_ldl'') or ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '       ModGramian: [ 1 | {0} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_ldl'), ...
                'Method'         , [], ...
                'ModGramian'     , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_soss_hinfbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '             Beta: [ nonnegative scalar | {0.1} ]\n');
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '            Gamma: [ positive scalar | {Inf} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '       StabMethod: [ ''lyap'' | {''cabe''} ]\n');
            if hasHyperlink
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_cabe_sgn">ml_morlabopts(''ml_cabe_sgn'')</a> or <a href="matlab:ml_morlabopts ml_lyap_sgn">ml_morlabopts(''ml_lyap_sgn'')</a>} ]\n');
            else
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_cabe_sgn'') or ml_morlabopts(''ml_lyap_sgn'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'Beta'           , [], ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'Gamma'          , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StabMethod'     , [], ...
                'stabmethodopts' , ml_morlabopts('ml_cabe_sgn'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_soss_hna'
        if nargout == 0
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '    hankeldecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '    hankeldecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '        MinRelTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'hankeldecopts'  , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_fac'), ...
                'MinRelTol'      , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StoreGramians'  , [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_soss_krylov'
        if nargout == 0
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(SVs))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_ct_d_soss_lqgbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '             Beta: [ nonnegative scalar | {0.1} ]\n');
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '       StabMethod: [ ''lyap'' | {''cabe''} ]\n');
            if hasHyperlink
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_cabe_sgn">ml_morlabopts(''ml_cabe_sgn'')</a> or <a href="matlab:ml_morlabopts ml_lyap_sgn">ml_morlabopts(''ml_lyap_sgn'')</a>} ]\n');
            else
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_cabe_sgn'') or ml_morlabopts(''ml_lyap_sgn'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'Beta'           , [], ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StabMethod'     , [], ...
                'stabmethodopts' , ml_morlabopts('ml_cabe_sgn'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_soss_mt'
        if nargout == 0
            fprintf(1, '           Alpha: [ real scalar | {-1.0} ]\n');
            fprintf(1, '       DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '       DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '      gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '      gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '       IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '       IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '   ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '           Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       signmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '       signmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '      infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '     OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '         RankTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '   StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'Alpha'         , [], ...
                'DecompEig'     , [], ...
                'DecompTol'     , [], ...
                'gdlyapopts'    , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'IGramFacC'     , [], ...
                'IGramFacO'     , [], ...
                'ImproperTrunc' , [], ...
                'Index'         , [], ...
                'signmopts'     , ml_morlabopts('ml_signm'), ...
                'infdecopts'    , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'OutputModel'   , [], ...
                'RankTol'       , [], ...
                'StoreGramians' , [], ...
                'StoreProjection', []);
        end

    case 'ml_ct_d_soss_prbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'Epsilon'        , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_soss_tlbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_ldl">ml_morlabopts(''ml_lyapdl_sgn_ldl'')</a> or <a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_ldl'') or ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '       ModGramian: [ 1 | {0} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        TimeRange: [ nonnegative vector | {[0, 10]} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_ldl'), ...
                'Method'         , [], ...
                'ModGramian'     , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'TimeRange'      , [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_ss_brbt'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_ss_bst'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyap_sgn_fac">ml_morlabopts(''ml_lyap_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyap_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            if hasHyperlink
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_sylv_sgn">ml_morlabopts(''ml_sylv_sgn'')</a>} ]\n');
            else
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_sylv_sgn'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '        UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'Epsilon'        , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapopts'       , ml_morlabopts('ml_lyap_sgn_fac'), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'stabsignmopts'  , ml_morlabopts('ml_signm'), ...
                'stabsylvopts'   , ml_morlabopts('ml_sylv_sgn'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'UnstabDim'      , []);
        end

    case 'ml_ct_d_ss_bt'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            if hasHyperlink
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_sylv_sgn">ml_morlabopts(''ml_sylv_sgn'')</a>} ]\n');
            else
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_sylv_sgn'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '        UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_fac'), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabsignmopts'  , ml_morlabopts('ml_signm'), ...
                'stabsylvopts'   , ml_morlabopts('ml_sylv_sgn'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'UnstabDim'      , []);
        end

    case 'ml_ct_d_ss_flbt'
        if nargout == 0
            fprintf(1, '        FreqRange: [ nonnegative vector | {[0, 1.0e+03]} ]\n');
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_ldl">ml_morlabopts(''ml_lyapdl_sgn_ldl'')</a> or <a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_ldl'') or ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '       ModGramian: [ 1 | {0} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            if hasHyperlink
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_sylv_sgn">ml_morlabopts(''ml_sylv_sgn'')</a>} ]\n');
            else
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_sylv_sgn'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '        UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'FreqRange'      , [], ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_ldl'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'ModGramian'     , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabsignmopts'  , ml_morlabopts('ml_signm'), ...
                'stabsylvopts'   , ml_morlabopts('ml_sylv_sgn'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'UnstabDim'      , []);
        end

    case 'ml_ct_d_ss_hinfbt'
        if nargout == 0
            fprintf(1, '             Beta: [ nonnegative scalar | {0.1} ]\n');
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '            Gamma: [ positive scalar | {Inf} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            fprintf(1, '       StabMethod: [ ''lyap'' | {''cabe''} ]\n');
            if hasHyperlink
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_cabe_sgn">ml_morlabopts(''ml_cabe_sgn'')</a> or <a href="matlab:ml_morlabopts ml_lyap_sgn">ml_morlabopts(''ml_lyap_sgn'')</a>} ]\n');
            else
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_cabe_sgn'') or ml_morlabopts(''ml_lyap_sgn'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '        UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'Beta'           , [], ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'Gamma'          , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StabMethod'     , [], ...
                'stabmethodopts' , ml_morlabopts('ml_cabe_sgn'), ...
                'stabsignmopts'  , ml_morlabopts('ml_signm'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'UnstabDim'      , []);
        end

    case 'ml_ct_d_ss_hna'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '  hankelsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '  hankelsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '   hankelsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_sylv_sgn">ml_morlabopts(''ml_sylv_sgn'')</a>} ]\n');
            else
                fprintf(1, '   hankelsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_sylv_sgn'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '        MinRelTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            if hasHyperlink
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_sylv_sgn">ml_morlabopts(''ml_sylv_sgn'')</a>} ]\n');
            else
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_sylv_sgn'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '        UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'hankelsignmopts', ml_morlabopts('ml_signm'), ...
                'hankelsylvopts' , ml_morlabopts('ml_sylv_sgn'), ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_fac'), ...
                'MinRelTol'      , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabsignmopts'  , ml_morlabopts('ml_signm'), ...
                'stabsylvopts'   , ml_morlabopts('ml_sylv_sgn'), ...
                'StoreGramians'  , [], ...
                'Tolerance'      , [], ...
                'UnstabDim'      , []);
        end

    case 'ml_ct_d_ss_krylov'
        if nargout == 0
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(SVs))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_ct_d_ss_lqgbt'
        if nargout == 0
            fprintf(1, '             Beta: [ nonnegative scalar | {0.1} ]\n');
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            fprintf(1, '       StabMethod: [ ''lyap'' | {''cabe''} ]\n');
            if hasHyperlink
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_cabe_sgn">ml_morlabopts(''ml_cabe_sgn'')</a> or <a href="matlab:ml_morlabopts ml_lyap_sgn">ml_morlabopts(''ml_lyap_sgn'')</a>} ]\n');
            else
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_cabe_sgn'') or ml_morlabopts(''ml_lyap_sgn'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '        UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'Beta'           , [], ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StabMethod'     , [], ...
                'stabmethodopts' , ml_morlabopts('ml_cabe_sgn'), ...
                'stabsignmopts'  , ml_morlabopts('ml_signm'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'UnstabDim'      , []);
        end

    case 'ml_ct_d_ss_mt'
        if nargout == 0
            fprintf(1, '           Alpha: [ real scalar | {-1.0} ]\n');
            if hasHyperlink
                fprintf(1, '       signmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '       signmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
            if hasHyperlink
                fprintf(1, '        sylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_sylv_sgn">ml_morlabopts(''ml_sylv_sgn'')</a>} ]\n');
            else
                fprintf(1, '        sylvopts: [ 1x1 struct | {ml_morlabopts(''ml_sylv_sgn'')} ]\n');
            end
        else
            opts = struct( ...
                'Alpha'         , [], ...
                'signmopts'     , ml_morlabopts('ml_signm'), ...
                'StoreProjection', [], ...
                'sylvopts'      , ml_morlabopts('ml_sylv_sgn'));
        end

    case 'ml_ct_d_ss_prbt'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '         careopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_caredl_sgn_fac">ml_morlabopts(''ml_caredl_sgn_fac'')</a>, <a href="matlab:ml_morlabopts ml_care_nwt_fac">ml_morlabopts(''ml_care_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         careopts: [ 1x1 struct | {ml_morlabopts(''ml_caredl_sgn_fac''), ml_morlabopts(''ml_care_nwt_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sign''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'careopts'       , merge_structs(ml_morlabopts('ml_caredl_sgn_fac'), ml_morlabopts('ml_care_nwt_fac')), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Epsilon'        , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_d_ss_tlbt'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '         lyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_lyapdl_sgn_ldl">ml_morlabopts(''ml_lyapdl_sgn_ldl'')</a> or <a href="matlab:ml_morlabopts ml_lyapdl_sgn_fac">ml_morlabopts(''ml_lyapdl_sgn_fac'')</a>} ]\n');
            else
                fprintf(1, '         lyapopts: [ 1x1 struct | {ml_morlabopts(''ml_lyapdl_sgn_ldl'') or ml_morlabopts(''ml_lyapdl_sgn_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '       ModGramian: [ 1 | {0} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            if hasHyperlink
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_sylv_sgn">ml_morlabopts(''ml_sylv_sgn'')</a>} ]\n');
            else
                fprintf(1, '     stabsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_sylv_sgn'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        TimeRange: [ nonnegative vector | {[0, 10]} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '        UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'lyapopts'       , ml_morlabopts('ml_lyapdl_sgn_ldl'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'ModGramian'     , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabsignmopts'  , ml_morlabopts('ml_signm'), ...
                'stabsylvopts'   , ml_morlabopts('ml_sylv_sgn'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'TimeRange'      , [], ...
                'Tolerance'      , [], ...
                'UnstabDim'      , []);
        end

    case 'ml_ct_brbt'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_bst'
        if nargout == 0
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'Epsilon'        , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_bt'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_flbt'
        if nargout == 0
            fprintf(1, '        FreqRange: [ nonnegative vector | {[0, 1.0e+03]} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '       ModGramian: [ 1 | {0} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'FreqRange'      , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'ModGramian'     , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_hinfbt'
        if nargout == 0
            fprintf(1, '            Gamma: [ positive scalar | {Inf} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'Gamma'          , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_hna'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        MinRelTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'MinRelTol'      , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreGramians'  , [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_krylov'
        if nargout == 0
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_ct_lqgbt'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_mt'
        if nargout == 0
            fprintf(1, '           Alpha: [ real scalar | {-1.0} ]\n');
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'Alpha'         , [], ...
                'StoreProjection', []);
        end

    case 'ml_ct_prbt'
        if nargout == 0
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'Epsilon'        , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_tlbt'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '       ModGramian: [ 1 | {0} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        TimeRange: [ nonnegative vector | {[0, 10]} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'ModGramian'     , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'TimeRange'      , [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_twostep_mor'
        if nargout == 0
            fprintf(1, '     krylovopts: [ struct | {struct()} ]\n');
            fprintf(1, '      KrylovROM: [ struct | {struct()} ]\n');
            fprintf(1, '      MORMethod: [ ''brbt'' | ''bst'' | ''flbt'' | ''hinfbt'' | ''hna'' | ''lqgbt'' | ''mt'' | ''prbt'' | ''tlbt'' | {''bt''} ]\n');
            fprintf(1, '  mormethodopts: [ struct | {struct()} ]\n');
            fprintf(1, ' StoreKrylovROM: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'krylovopts'   , [], ...
                'KrylovROM'    , [], ...
                'MORMethod'    , [], ...
                'mormethodopts', [], ...
                'StoreKrylovROM', []);
        end

    case 'ml_ct_s_foss_brbt'
        if nargout == 0
            fprintf(1, '         carecopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '         careoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '          GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '          GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '            Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '             Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, '  OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '     RiccatiSolver: [ ''ri'' | {''ri''} ]\n');
            fprintf(1, '     StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '   StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '         Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'carecopts'       , struct(), ...
                'careoopts'       , struct(), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_foss_bst'
        if nargout == 0
            fprintf(1, '         careopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '         lyapopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''ri'' | {''ri''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'careopts'       , struct(), ...
                'Epsilon'        , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapopts'       , struct(), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_foss_bt'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '         lyapcopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '         lyapoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapcopts'      , struct(), ...
                'lyapoopts'      , struct(), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_foss_hinfbt'
        if nargout == 0
            fprintf(1, '        carecopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        careoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '            Gamma: [ positive scalar | {Inf} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''radi''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'carecopts'      , struct(), ...
                'careoopts'      , struct(), ...
                'Gamma'          , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_foss_hna'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '    hankeldecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '    hankeldecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '  hankelsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '  hankelsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '   hankelsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_sylv_sgn">ml_morlabopts(''ml_sylv_sgn'')</a>} ]\n');
            else
                fprintf(1, '   hankelsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_sylv_sgn'')} ]\n');
            end
            fprintf(1, '        lyapcopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        lyapoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        MinRelTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'hankeldecopts'  , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'hankelsignmopts', ml_morlabopts('ml_signm'), ...
                'hankelsylvopts' , ml_morlabopts('ml_sylv_sgn'), ...
                'lyapcopts'      , struct(), ...
                'lyapoopts'      , struct(), ...
                'MinRelTol'      , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreGramians'  , [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_foss_krylov'
        if nargout == 0
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(SVs))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_ct_s_foss_lqgbt'
        if nargout == 0
            fprintf(1, '        carecopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        careoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''radi''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'carecopts'      , struct(), ...
                'careoopts'      , struct(), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_foss_prbt'
        if nargout == 0
            fprintf(1, '        carecopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        careoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''ri'' | {''ri''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'carecopts'      , struct(), ...
                'careoopts'      , struct(), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Epsilon'        , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_foss_twostep_mor'
        if nargout == 0
            fprintf(1, '     krylovopts: [ struct | {struct()} ]\n');
            fprintf(1, '      KrylovROM: [ struct | {struct()} ]\n');
            fprintf(1, '      MORMethod: [ ''brbt'' | ''bst'' | ''flbt'' | ''hinfbt'' | ''hna'' | ''lqgbt'' | ''mt'' | ''prbt'' | ''tlbt'' | {''bt''} ]\n');
            fprintf(1, '  mormethodopts: [ struct | {struct()} ]\n');
            fprintf(1, ' StoreKrylovROM: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'krylovopts'   , [], ...
                'KrylovROM'    , [], ...
                'MORMethod'    , [], ...
                'mormethodopts', [], ...
                'StoreKrylovROM', []);
        end

    case 'ml_ct_s_soss_brbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '        carecopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        careoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''ri'' | {''ri''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'carecopts'      , struct(), ...
                'careoopts'      , struct(), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_soss_bst'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '         careopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '         lyapopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''ri'' | {''ri''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'careopts'       , struct(), ...
                'Epsilon'        , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapopts'       , struct(), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_soss_bt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        lyapcopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        lyapoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'lyapcopts'      , struct(), ...
                'lyapoopts'      , struct(), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_soss_hinfbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '        carecopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        careoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '            Gamma: [ positive scalar | {Inf} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''radi''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'carecopts'      , struct(), ...
                'careoopts'      , struct(), ...
                'Gamma'          , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_soss_hna'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            if hasHyperlink
                fprintf(1, '    hankeldecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '    hankeldecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '        lyapcopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        lyapoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        MinRelTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'hankeldecopts'  , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'lyapcopts'      , struct(), ...
                'lyapoopts'      , struct(), ...
                'MinRelTol'      , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreGramians'  , [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_soss_krylov'
        if nargout == 0
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(SVs))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_ct_s_soss_lqgbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '        carecopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        careoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''radi''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'carecopts'      , struct(), ...
                'careoopts'      , struct(), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_soss_prbt'
        if nargout == 0
            fprintf(1, '      BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '        carecopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '        careoopts: [ 1x1 struct | {struct()} ]\n');
            fprintf(1, '          Epsilon: [ positive scalar | {1.0e-03} ]\n');
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '      OutputModel: [ ''so'' | {''fo''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''ri'' | {''ri''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType'    , [], ...
                'carecopts'      , struct(), ...
                'careoopts'      , struct(), ...
                'Epsilon'        , [], ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'OutputModel'    , [], ...
                'RiccatiSolver'  , [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_ct_s_soss_twostep_mor'
        if nargout == 0
            fprintf(1, '     krylovopts: [ struct | {struct()} ]\n');
            fprintf(1, '      KrylovROM: [ struct | {struct()} ]\n');
            fprintf(1, '      MORMethod: [ ''brbt'' | ''bst'' | ''flbt'' | ''hinfbt'' | ''hna'' | ''lqgbt'' | ''mt'' | ''prbt'' | ''tlbt'' | {''bt''} ]\n');
            fprintf(1, '  mormethodopts: [ struct | {struct()} ]\n');
            fprintf(1, ' StoreKrylovROM: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'krylovopts'   , [], ...
                'KrylovROM'    , [], ...
                'MORMethod'    , [], ...
                'mormethodopts', [], ...
                'StoreKrylovROM', []);
        end

    case 'ml_dt_d_dss_bt'
        if nargout == 0
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '        dlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dlyapdl_smith_fac">ml_morlabopts(''ml_dlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '        dlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_dlyapdl_smith_fac'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'dlyapopts'      , ml_morlabopts('ml_dlyapdl_smith_fac'), ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_dt_d_dss_krylov'
        if nargout == 0
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(SVs))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_dt_d_dss_lqgbt'
        if nargout == 0
            fprintf(1, '             Beta: [ nonnegative scalar | {0.1} ]\n');
            if hasHyperlink
                fprintf(1, '         dareopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_daredl_sda_fac">ml_morlabopts(''ml_daredl_sda_fac'')</a>, <a href="matlab:ml_morlabopts ml_dare_nwt_fac">ml_morlabopts(''ml_dare_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         dareopts: [ 1x1 struct | {ml_morlabopts(''ml_daredl_sda_fac''), ml_morlabopts(''ml_dare_nwt_fac'')} ]\n');
            end
            fprintf(1, '        DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '        DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '       gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '        IGramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '    ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '            Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '       infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsvp)) + Nu + Ni} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sda''} ]\n');
            if hasHyperlink
                fprintf(1, '      stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dlyap_smith">ml_morlabopts(''ml_dlyap_smith'')</a>} ]\n');
            else
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_dlyap_smith'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'Beta'           , [], ...
                'dareopts'       , merge_structs(ml_morlabopts('ml_daredl_sda_fac'), ml_morlabopts('ml_dare_nwt_fac')), ...
                'DecompEig'      , [], ...
                'DecompTol'      , [], ...
                'gdlyapopts'     , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'IGramFacC'      , [], ...
                'IGramFacO'      , [], ...
                'ImproperTrunc'  , [], ...
                'Index'          , [], ...
                'infdecopts'     , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'stabdecopts'    , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'stabmethodopts' , ml_morlabopts('ml_dlyap_smith'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_dt_d_dss_mt'
        if nargout == 0
            fprintf(1, '           Alpha: [ real scalar | {0.5} ]\n');
            fprintf(1, '       DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '       DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '      gdlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_gdlyapdl_smith_fac">ml_morlabopts(''ml_gdlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '      gdlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_gdlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '   ImproperTrunc: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '           Index: [ nonnegative integer | {Inf} ]\n');
            if hasHyperlink
                fprintf(1, '       signmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '       signmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '      infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '         RankTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            fprintf(1, '   StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'Alpha'         , [], ...
                'DecompEig'     , [], ...
                'DecompTol'     , [], ...
                'gdlyapopts'    , ml_morlabopts('ml_gdlyapdl_smith_fac'), ...
                'ImproperTrunc' , [], ...
                'Index'         , [], ...
                'signmopts'     , ml_morlabopts('ml_signm'), ...
                'infdecopts'    , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'RankTol'       , [], ...
                'StoreGramians' , [], ...
                'StoreProjection', []);
        end

    case 'ml_dt_d_ss_bt'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '        dlyapopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dlyapdl_smith_fac">ml_morlabopts(''ml_dlyapdl_smith_fac'')</a>} ]\n');
            else
                fprintf(1, '        dlyapopts: [ 1x1 struct | {ml_morlabopts(''ml_dlyapdl_smith_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            if hasHyperlink
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '    stabdsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dsylv_smith">ml_morlabopts(''ml_dsylv_smith'')</a>} ]\n');
            else
                fprintf(1, '    stabdsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_dsylv_smith'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '        UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'dlyapopts'      , ml_morlabopts('ml_dlyapdl_smith_fac'), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'stabsignmopts'  , ml_morlabopts('ml_signm'), ...
                'stabdsylvopts'  , ml_morlabopts('ml_dsylv_smith'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'UnstabDim'      , []);
        end

    case 'ml_dt_d_ss_krylov'
        if nargout == 0
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(SVs))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_dt_d_ss_lqgbt'
        if nargout == 0
            fprintf(1, '             Beta: [ nonnegative scalar | {0.1} ]\n');
            if hasHyperlink
                fprintf(1, '         dareopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_daredl_sda_fac">ml_morlabopts(''ml_daredl_sda_fac'')</a>, <a href="matlab:ml_morlabopts ml_dare_nwt_fac">ml_morlabopts(''ml_dare_nwt_fac'')</a>} ]\n');
            else
                fprintf(1, '         dareopts: [ 1x1 struct | {ml_morlabopts(''ml_daredl_sda_fac''), ml_morlabopts(''ml_dare_nwt_fac'')} ]\n');
            end
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv)) + Nu} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    RiccatiSolver: [ ''newton'' | {''sda''} ]\n');
            if hasHyperlink
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dlyap_smith">ml_morlabopts(''ml_dlyap_smith'')</a>} ]\n');
            else
                fprintf(1, '   stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_dlyap_smith'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '    stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '        UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'Beta'           , [], ...
                'dareopts'       , merge_structs(ml_morlabopts('ml_daredl_sda_fac'), ml_morlabopts('ml_dare_nwt_fac')), ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'RiccatiSolver'  , [], ...
                'stabmethodopts' , ml_morlabopts('ml_dlyap_smith'), ...
                'stabsignmopts'  , ml_morlabopts('ml_signm'), ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'UnstabDim'      , []);
        end

    case 'ml_dt_d_ss_mt'
        if nargout == 0
            fprintf(1, '           Alpha: [ real scalar | {0.5} ]\n');
            if hasHyperlink
                fprintf(1, '       dsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dsylv_smith">ml_morlabopts(''ml_dsylv_smith'')</a>} ]\n');
            else
                fprintf(1, '       dsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_dsylv_smith'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '       signmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '       signmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'Alpha'         , [], ...
                'dsylvopts'     , ml_morlabopts('ml_dsylv_smith'), ...
                'signmopts'     , ml_morlabopts('ml_signm'), ...
                'StoreProjection', []);
        end

    case 'ml_dt_bt'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_dt_krylov'
        if nargout == 0
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_dt_lqgbt'
        if nargout == 0
            fprintf(1, '         GramFacC: [ matrix | {[]} ]\n');
            fprintf(1, '         GramFacO: [ matrix | {[]} ]\n');
            fprintf(1, '           Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(Hsv))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | {''tolerance''} ]\n');
            fprintf(1, '    StoreGramians: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'GramFacC'       , [], ...
                'GramFacO'       , [], ...
                'Method'         , [], ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreGramians'  , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , []);
        end

    case 'ml_dt_mt'
        if nargout == 0
            fprintf(1, '           Alpha: [ real scalar | {0.5} ]\n');
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'Alpha'         , [], ...
                'StoreProjection', []);
        end

    case 'ml_dt_twostep_mor'
        if nargout == 0
            fprintf(1, '     krylovopts: [ struct | {struct()} ]\n');
            fprintf(1, '      KrylovROM: [ struct | {struct()} ]\n');
            fprintf(1, '      MORMethod: [ ''lqgbt'' | ''mt'' | {''bt''} ]\n');
            fprintf(1, '  mormethodopts: [ struct | {struct()} ]\n');
            fprintf(1, ' StoreKrylovROM: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'krylovopts'   , [], ...
                'KrylovROM'    , [], ...
                'MORMethod'    , [], ...
                'mormethodopts', [], ...
                'StoreKrylovROM', []);
        end

    case 'ml_dt_s_foss_krylov'
        if nargout == 0
            fprintf(1, '      krylovVopts: [ struct | {struct()} ]\n');
            fprintf(1, '      krylovWopts: [ struct | {struct()} ]\n');
            fprintf(1, '            Order: [ positive integer | {min(10,length(SVs))} ]\n');
            fprintf(1, ' OrderComputation: [ ''order'' | ''sum'' | {''tolerance''} ]\n');
            fprintf(1, '       StoreBases: [ 1 | {0} ]\n');
            fprintf(1, '  StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '        Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, '     TwoSidedProj: [ 1 | {0} ]\n');
            fprintf(1, '           VBasis: [ matrix | {[]} ]\n');
            fprintf(1, '           WBasis: [ matrix | {[]} ]\n');
        else
            opts = struct( ...
                'krylovVopts'    , local_krylovopts(), ...
                'krylovWopts'    , local_krylovopts(), ...
                'Order'          , [], ...
                'OrderComputation', [], ...
                'StoreBases'     , [], ...
                'StoreProjection', [], ...
                'Tolerance'      , [], ...
                'TwoSidedProj'   , [], ...
                'VBasis'         , [], ...
                'WBasis'         , []);
        end

    case 'ml_dt_s_foss_twostep_mor'
        if nargout == 0
            fprintf(1, '     krylovopts: [ struct | {struct()} ]\n');
            fprintf(1, '      KrylovROM: [ struct | {struct()} ]\n');
            fprintf(1, '      MORMethod: [ ''lqgbt'' | ''mt'' | {''bt''} ]\n');
            fprintf(1, '  mormethodopts: [ struct | {struct()} ]\n');
            fprintf(1, ' StoreKrylovROM: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'krylovopts'   , [], ...
                'KrylovROM'    , [], ...
                'MORMethod'    , [], ...
                'mormethodopts', [], ...
                'StoreKrylovROM', []);
        end

    case 'ml_balproj_improper'
        if nargout == 0
            fprintf(1, ' ImproperTrunc: [ nonnegative scalar | {log(ninf)*eps} ]\n');
            fprintf(1, '         Index: [ nonnegative integer | {Inf} ]\n');
            fprintf(1, '        Method: [ ''bfsr'' | {''sr''} ]\n');
        else
            opts = struct( ...
                'ImproperTrunc', [], ...
                'Index'       , [], ...
                'Method'      , []);
        end

    case 'ml_balproj_proper'
        if nargout == 0
            fprintf(1, '     Gamma: [ positive scalar | {Inf} ]\n');
            fprintf(1, '    Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '     Order: [ positive integer | {min(10,length(hsvp)) + nu + ni} ]\n');
            fprintf(1, ' Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'Gamma'   , [], ...
                'Method'  , [], ...
                'Order'   , [], ...
                'Tolerance', []);
        end

    case 'ml_balproj_soss'
        if nargout == 0
            fprintf(1, ' BalanceType: [ ''p'' | ''pm'' | ''pv'' | ''vp'' | ''vpm'' | ''v'' | ''fv'' | {''so''} ]\n');
            fprintf(1, '       Gamma: [ positive scalar | {Inf} ]\n');
            fprintf(1, '      Method: [ ''bfsr'' | {''sr''} ]\n');
            fprintf(1, '       Order: [ positive integer | {min(10,length(hsvp)) + nu + ni} ]\n');
            fprintf(1, '   Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'BalanceType', [], ...
                'Gamma'     , [], ...
                'Method'    , [], ...
                'Order'     , [], ...
                'Tolerance' , []);
        end

    case 'ml_compress_fac'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_compress_ldl'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_ct_d_dss_adtf'
        if nargout == 0
            fprintf(1, '       DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '       DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '      infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '     stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '     stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'DecompEig'     , [], ...
                'DecompTol'     , [], ...
                'infdecopts'    , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'stabdecopts'   , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StoreProjection', []);
        end

    case 'ml_ct_d_dss_partstab'
        if nargout == 0
            fprintf(1, '           Beta: [ nonnegative scalar | {0.1} ]\n');
            fprintf(1, '      DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '      DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '     infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '     infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '    stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '    stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, '     StabMethod: [ ''lyap'' | {''cabe''} ]\n');
            if hasHyperlink
                fprintf(1, ' stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_cabe_sgn">ml_morlabopts(''ml_cabe_sgn'')</a> or <a href="matlab:ml_morlabopts ml_lyap_sgn">ml_morlabopts(''ml_lyap_sgn'')</a>} ]\n');
            else
                fprintf(1, ' stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_cabe_sgn'') or ml_morlabopts(''ml_lyap_sgn'')} ]\n');
            end
        else
            opts = struct( ...
                'Beta'         , [], ...
                'DecompEig'    , [], ...
                'DecompTol'    , [], ...
                'infdecopts'   , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'stabdecopts'  , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StabMethod'   , [], ...
                'stabmethodopts', ml_morlabopts('ml_cabe_sgn'));
        end

    case 'ml_ct_d_ss_adtf'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '   stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '   stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '    stabsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_sylv_sgn">ml_morlabopts(''ml_sylv_sgn'')</a>} ]\n');
            else
                fprintf(1, '    stabsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_sylv_sgn'')} ]\n');
            end
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '       UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'stabsignmopts' , ml_morlabopts('ml_signm'), ...
                'stabsylvopts'  , ml_morlabopts('ml_sylv_sgn'), ...
                'StoreProjection', [], ...
                'UnstabDim'     , []);
        end

    case 'ml_ct_d_ss_partstab'
        if nargout == 0
            fprintf(1, '           Beta: [ nonnegative scalar | {0.1} ]\n');
            fprintf(1, '     StabMethod: [ ''lyap'' | {''cabe''} ]\n');
            if hasHyperlink
                fprintf(1, ' stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_cabe_sgn">ml_morlabopts(''ml_cabe_sgn'')</a> or <a href="matlab:ml_morlabopts ml_lyap_sgn">ml_morlabopts(''ml_lyap_sgn'')</a>} ]\n');
            else
                fprintf(1, ' stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_cabe_sgn'') or ml_morlabopts(''ml_lyap_sgn'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '  stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '  stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            fprintf(1, '      UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'Beta'         , [], ...
                'StabMethod'   , [], ...
                'stabmethodopts', ml_morlabopts('ml_cabe_sgn'), ...
                'stabsignmopts', ml_morlabopts('ml_signm'), ...
                'UnstabDim'    , []);
        end

    case 'ml_decide_system_type'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_disk'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_domsubproj_proper'
        if nargout == 0
            fprintf(1, '        Order: [ positive integer | {min(10,length(sv)) + rmin} ]\n');
            fprintf(1, '    Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, ' TwoSidedProj: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'Order'      , [], ...
                'Tolerance'  , [], ...
                'TwoSidedProj', []);
        end

    case 'ml_domsubproj_soss'
        if nargout == 0
            fprintf(1, '        Order: [ positive integer | {min(10,length(hsvp))} ]\n');
            fprintf(1, '    Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
            fprintf(1, ' TwoSidedProj: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'Order'      , [], ...
                'Tolerance'  , [], ...
                'TwoSidedProj', []);
        end

    case 'ml_dt_d_dss_adtf'
        if nargout == 0
            fprintf(1, '       DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '       DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '      infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '      infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '     stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '     stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
        else
            opts = struct( ...
                'DecompEig'     , [], ...
                'DecompTol'     , [], ...
                'infdecopts'    , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'stabdecopts'   , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'StoreProjection', []);
        end

    case 'ml_dt_d_dss_partstab'
        if nargout == 0
            fprintf(1, '           Beta: [ positive scalar < 1 | {0.1} ]\n');
            fprintf(1, '      DecompEig: [ positive scalar | {[]} ]\n');
            fprintf(1, '      DecompTol: [ nonnegative scalar | {log(n)*eps} ]\n');
            if hasHyperlink
                fprintf(1, '     infdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_disk">ml_morlabopts(''ml_disk'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '     infdecopts: [ 1x1 struct | {ml_morlabopts(''ml_disk''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '    stabdecopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>, <a href="matlab:ml_morlabopts ml_getqz">ml_morlabopts(''ml_getqz'')</a>} ]\n');
            else
                fprintf(1, '    stabdecopts: [ 1x1 struct | {ml_morlabopts(''ml_signm''), ml_morlabopts(''ml_getqz'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, ' stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dlyap_smith">ml_morlabopts(''ml_dlyap_smith'')</a>} ]\n');
            else
                fprintf(1, ' stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_dlyap_smith'')} ]\n');
            end
        else
            opts = struct( ...
                'Beta'         , [], ...
                'DecompEig'    , [], ...
                'DecompTol'    , [], ...
                'infdecopts'   , merge_structs(ml_morlabopts('ml_disk'), ml_morlabopts('ml_getqz')), ...
                'stabdecopts'  , merge_structs(ml_morlabopts('ml_signm'), ml_morlabopts('ml_getqz')), ...
                'stabmethodopts', ml_morlabopts('ml_dlyap_smith'));
        end

    case 'ml_dt_d_ss_adtf'
        if nargout == 0
            if hasHyperlink
                fprintf(1, '   stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '   stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '   stabdsylvopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dsylv_smith">ml_morlabopts(''ml_dsylv_smith'')</a>} ]\n');
            else
                fprintf(1, '   stabdsylvopts: [ 1x1 struct | {ml_morlabopts(''ml_dsylv_smith'')} ]\n');
            end
            fprintf(1, ' StoreProjection: [ 1 | {0} ]\n');
            fprintf(1, '       UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'stabsignmopts' , ml_morlabopts('ml_signm'), ...
                'stabdsylvopts' , ml_morlabopts('ml_dsylv_smith'), ...
                'StoreProjection', [], ...
                'UnstabDim'     , []);
        end

    case 'ml_dt_d_ss_partstab'
        if nargout == 0
            fprintf(1, '           Beta: [ positive scalar < 1 | {0.1} ]\n');
            if hasHyperlink
                fprintf(1, ' stabmethodopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_dlyap_smith">ml_morlabopts(''ml_dlyap_smith'')</a>} ]\n');
            else
                fprintf(1, ' stabmethodopts: [ 1x1 struct | {ml_morlabopts(''ml_dlyap_smith'')} ]\n');
            end
            if hasHyperlink
                fprintf(1, '  stabsignmopts: [ 1x1 struct | {<a href="matlab:ml_morlabopts ml_signm">ml_morlabopts(''ml_signm'')</a>} ]\n');
            else
                fprintf(1, '  stabsignmopts: [ 1x1 struct | {ml_morlabopts(''ml_signm'')} ]\n');
            end
            fprintf(1, '      UnstabDim: [ integer | {-1} ]\n');
        else
            opts = struct( ...
                'Beta'         , [], ...
                'stabmethodopts', ml_morlabopts('ml_dlyap_smith'), ...
                'stabsignmopts', ml_morlabopts('ml_signm'), ...
                'UnstabDim'    , []);
        end

    case 'ml_extend_cell'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_format_output'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_getqz'
        if nargout == 0
            fprintf(1, ' Dimension: [ integer | {-1} ]\n');
            fprintf(1, '   RankTol: [ nonnegative scalar | {log(n)*eps} ]\n');
        else
            opts = struct( ...
                'Dimension', [], ...
                'RankTol' , []);
        end

    case 'ml_morlabopts'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_order'
        if nargout == 0
            fprintf(1, '     Gamma: [ positive scalar | {Inf} ]\n');
            fprintf(1, '     Order: [ positive integer | {min(10,length(hsv)+rmin)} ]\n');
            fprintf(1, ' Tolerance: [ nonnegative scalar | {1.0e-02} ]\n');
        else
            opts = struct( ...
                'Gamma'   , [], ...
                'Order'   , [], ...
                'Tolerance', []);
        end

    case 'ml_prepare_system_data'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_projtrunc_improper'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_projtrunc_proper'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_projtrunc_soss'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    case 'ml_signm'
        if nargout == 0
            fprintf(1, '  AbsTol: [ nonnegative scalar | {0} ]\n');
            fprintf(1, '    Info: [ 1 | {0} ]\n');
            fprintf(1, ' MaxIter: [ positive integer | {100} ]\n');
            fprintf(1, '  RelTol: [ nonnegative scalar | {1.0e+02*n*eps} ]\n');
        else
            opts = struct( ...
                'AbsTol', [], ...
                'Info'  , [], ...
                'MaxIter', [], ...
                'RelTol', []);
        end

    case 'ml_addpath'
        if nargout == 0
            fprintf(1, ' The requested function doesn''t use an option struct!\n');
        else
            opts = struct([]);
        end

    otherwise % Non-MORLAB functions.
        error('MORLAB:data', ...
            'The given function %s is not implemented in MORLAB.', fname);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DEFAULT HELP LINE.                                                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Default help link for more details.
if nargout == 0
    fprintf(1, '\n');
    if hasHyperlink
        fprintf(1, [' For more details see <a href="matlab:help ' ...
            '%s">%s</a>.\n'], fname, fname);
    else
        fprintf(1, ' For more details see %s.', fname);
    end
    fprintf(1, '\n');
end

end % end function


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% HELPER FUNCTION.                                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function struct = merge_structs(varargin)
%MERGE_STRUCTS Merges structs together.
%
% SYNTAX:
%   struct = MERGE_STRUCTS(struct1, struct2, ...)
%
% DESCRIPTION:
%   Takes multiple structs and combines them into one with all fields.
%   If structs share a field with the same name, the latter argument's
%   value is used.
%
% INPUTS:
%   struct1, struct2, ... - arbitrary structs
%
% OUTPUTS:
%   struct - struct, containing all fields of struct1, struct2, ...

struct = varargin{1};
for k = 2:length(varargin)
    st     = varargin{k};
    fields = fieldnames(st);

    for field = transpose(fields)
        f                 = st.(field{1});
        struct.(field{1}) = f;
    end
end

end

function substruct = local_krylovopts()
%LOCAL_KRYLOVOPTS Generates the substructure for Krylov methods.
%
% SYNTAX:
%   substruct = local_krylovopts()
%
% DESCRIPTION:
%   ---
%
% INPUTS:
%   ---
%
% OUTPUTS:
%   ---

substruct = struct( ...
    'Directs'   , [], ...
    'FreqRange' , [], ...
    'NumPts'    , [], ...
    'Orders'    , [], ...
    'Points'    , [], ...
    'RealVal'   , []);

end
