function varargout = ml_format_output(rom, ioformat, info)
%ML_FORMAT_OUTPUT Reformat internal representation to final output.
%
% SYNTAX:
%               [A, B, C, D, info] = ML_FORMAT_OUTPUT(roms, 0, info)
%            [A, B, C, D, E, info] = ML_FORMAT_OUTPUT(roms, 0, info)
%   [M, E, K, Bu, Cp, Cv, D, info] = ML_FORMAT_OUTPUT(roms, 0, info)
%
%   [roms, info] = ml_format_output(roms, 1, info)
%
%   [roms, info] = ml_format_output(roms, 2, info)
%
% DESCRIPTION:
%   This function creates the final output data in terms of state-space
%   objects, structs or matrices for given systems. It also unpacks cell
%   arrays in the info struct if only a single rom is given.
%
% INPUTS:
%   rom      - cell array, contains the struct systems
%   ioformat - {0, 1, 2}, determines the output type of the systems
%                0 - matrices
%                1 - for input type structs
%                2 - for input type state-space objects
%   info     - struct, containing additional information
%
% OUTPUTS:
%   A    - cell array of matrices with dimensions n x n, or single matrix
%   B    - cell array of matrices with dimensions n x m, or single matrix
%   C    - cell array of matrices with dimensions p x n, or single matrix
%   D    - cell array of matrices with dimensions p x m, or single matrix
%   E    - cell array of matrices with dimensions n x n, or single matrix
%   M    - cell array of matrices with dimensions n x n, or single matrix
%   K    - cell array of matrices with dimensions n x n, or single matrix
%   Bu   - cell array of matrices with dimensions n x m, or single matrix
%   Cp   - cell array of matrices with dimensions p x n, or single matrix
%   Cv   - cell array of matrices with dimensions p x n, or single matrix
%   rom  - cell array of structs or state-space objects, or a single one
%   info - struct, same as input but with alphabetically ordered fields and
%          unpacked cell arrays if only one system is given
%
% See also ml_decide_system_type.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

narginchk(3, 3);

assert(isa(rom, 'cell') && isvector(rom), ...
    'MORLAB:data', ...
    'The input argument systems has to be a cell vector!');

assert(isa(info, 'struct'), ...
    'MORLAB:data', ...
    'The input argument info has to be a struct!');

% Sort info fields.
[~, perm] = sort(lower(fieldnames(info)));
info      = orderfields(info, perm);

% Format output ROMs.
numSys = length(rom);

for k = numSys:-1:1
    % Merge stable and unstable parts if available.
    sys = structfun(@full, rom{k}, 'UniformOutput', 0);

    % Pack arrays.
    switch ioformat
        case 0 % single matrices
            if isfield(sys, 'K')
                % Second-order systems.
                varargout{1}{k} = sys.M;
                varargout{2}{k} = sys.E;
                varargout{3}{k} = sys.K;
                varargout{4}{k} = sys.Bu;
                varargout{5}{k} = sys.Cp;
                varargout{6}{k} = sys.Cv;
                varargout{7}{k} = sys.D;
            else
                % First-order systems.
                varargout{1}{k} = sys.A;
                varargout{2}{k} = sys.B;
                varargout{3}{k} = sys.C;
                varargout{4}{k} = sys.D;

                if isfield(sys, 'E')
                    varargout{5}{k} = sys.E;
                end
            end

        case 1 % MORLAB system struct
            if isfield(sys, 'K')
                % Second-order systems.
                varargout{1}{k} = struct( ...
                    'M' , sys.M, ...
                    'E' , sys.E, ...
                    'K' , sys.K, ...
                    'Bu', sys.Bu, ...
                    'Cp', sys.Cp, ...
                    'Cv', sys.Cv, ...
                    'D' , sys.D);
            else
                % First-order systems.
                varargout{1}{k} = struct(...
                    'A', sys.A, ...
                    'B', sys.B, ...
                    'C', sys.C, ...
                    'D', sys.D);

                if isfield(sys, 'E')
                    varargout{1}{k}.E = sys.E;
                end
            end

            if isfield(sys, 'Ts')
                varargout{1}{k}.Ts = sys.Ts;
            end

        case 2 % MATLAB ss-object
            if isfield(sys, 'Ts')
                Ts = sys.Ts;
            else
                Ts = 0;
            end

            if isfield(sys, 'K')
                varargout{1}{k} = mechss(sys.M, sys.E, sys.K, sys.Bu, ...
                    sys.Cp, sys.Cv, sys.D, Ts);
            elseif isfield(sys, 'E')
                varargout{1}{k} = dss(sys.A, sys.B, sys.C, sys.D, ...
                    sys.E, Ts);
            else
                varargout{1}{k} = ss(sys.A, sys.B, sys.C, sys.D, Ts);
            end

        otherwise
            error('MORLAB:data', ...
                'The input argument ioformat has to be 0, 1 or 2!');
    end
end

% A single result is unpacked.
if numSys == 1
    for k = 1:length(varargout)
        varargout{k} = varargout{k}{1};
    end

    % Iterate fields of the info struct.
    fields = fieldnames(info);
    for field = fields'
        f = info.(field{1});

        if isa(f, 'cell') && (length(f) == 1)
            info.(field{1}) = f{1};
        end
    end
end

% Forward info structure.
len              = length(varargout);
varargout{len+1} = info;
