function [K, info] = ml_dt_d_ss_partstab(A, B, opts)
%ML_DT_D_SS_PARTSTAB Stabilizing feedback for standard systems.
%
% SYNTAX:
%   [K, info] = ML_DT_D_SS_PARTSTAB(A, B)
%   [K, info] = ML_DT_D_SS_PARTSTAB(A, B, opts)
%
% DESCRIPTION:
%   Partial stabilization is used for the discrete-time system of
%   odinary differential equations
%
%       x(t+1) = Ax(t) + Bu(t)                                          (1)
%
%   to get a stabilizing feedback term K, such that all eigenvalues of
%   A-B*K are inside the open unit disk. It is assumed that A has no
%   eigenvalues on the unit disk.
%
% INPUTS:
%   A    - matrix with dimensions n x n in (1)
%   B    - matrix with dimensions n x m in (1)
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Beta            | positive scalar < 1, used as shift of the         |
%   |                 | Bass' algorithm for better conditioning           |
%   |                 | (default 0.1)                                     |
%   +-----------------+---------------------------------------------------+
%   | stabmethodopts  | structure, containing the optional parameters for |
%   |                 | the Smith iteration based Lyapunov equation solver|
%   |                 | used for the stabilization, see ml_dlyap_smith    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabsignmopts   | structure, containing the optional parameters for |
%   |                 | the matrix sign function used for the             |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, see ml_signm                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | UnstabDim       | integer, dimension of the deflating anti-stable   |
%   |                 | subspace, negative if unknown                     |
%   |                 | (default -1)                                      |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   K    - stabilizing feedback matrix of dimensions m x n
%   info - structure, containing the following information about the
%          partial stabilization method
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | infoSTABMETH    | structure, containing information about the Smith |
%   |                 | iteration based solver used for the stabilization,|
%   |                 | see ml_dlyap_smith                                |
%   +-----------------+---------------------------------------------------+
%   | infoSTABMETH2   | structure, containing information about the Smith |
%   |                 | iteration based solver used for a second          |
%   |                 | stabilization if necessary, see ml_dlyap_smith    |
%   +-----------------+---------------------------------------------------+
%   | infoSTABSIGNM   | structure, containing information about the       |
%   |                 | matrix sign function method used for the          |
%   |                 | decomposition, see ml_signm                       |
%   +-----------------+---------------------------------------------------+
%   | infoSTABSIGNM2  | structure, containing information about the       |
%   |                 | matrix sign function method used for the a second |
%   |                 | decomposition, see ml_signm                       |
%   +-----------------+---------------------------------------------------+
%   | Ns              | Number of identified stable eigenvalues           |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Number of identified anti-stable eigenvalues      |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, Partial stabilization of descriptor systems using spectral
%   projectors, in: P. Van Dooren, S. P. Bhattacharyya, R. H. Chan, V.
%   Olshevsky, A.Routray (Eds.), Numerical Linear Algebra in Signals,
%   Systems and Control, Vol. 80 of Lect. Notes Electr. Eng., Springer
%   Netherlands, 2011, pp. 55--76.
%   https://doi.org/10.1007/978-94-007-0602-6_3
%
% See also ml_ct_d_ss_partstab, ml_ct_d_dss_partstab.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(2, 3);

if (nargin < 3) || isempty(opts)
    opts = struct();
end

% Check input matrices.
n = size(A, 1);
m = size(B, 2);

assert(isequal(size(A), [n n]), ...
    'MORLAB:data', ...
    'The matrix A has to be square!');

assert(size(B, 1) == n, ...
    'MORLAB:data', ...
    'The matrix B must have the same number of rows as A!');

if issparse(A), A = full(A); end
if issparse(B), B = full(B); end

% Check and assign optional parameters.
assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The parameter opts has to be a struct!');

if ml_field_set_to_value(opts, 'Beta')
    ml_assert_posscalar(opts.Beta, 'opts.Beta');

    assert(opts.Beta < 1, ...
        'MORLAB:data', ...
        'The parameter opts.Beta needs to be smaller than 1!');
else
    opts.Beta = 0.1;
end

if ml_field_set_to_value(opts, 'stabsignmopts')
    assert(isa(opts.stabsignmopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.stabsignmopts has to be a struct!');
else
    opts.stabsignmopts = struct();
end

if ml_field_set_to_value(opts, 'stabmethodopts')
    assert(isa(opts.stabmethodopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.stabmethodopts has to be a struct!');
else
    opts.stabmethodopts = struct();
end

if ml_field_set_to_value(opts, 'UnstabDim')
    ml_assert_integer(opts.UnstabDim, 'opts.UnstabDim');
else
    opts.UnstabDim = -1;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION: ANTI-STABLE PART.                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition into stable and anti-stable parts.
if opts.UnstabDim == 0
    ns = n;
else
    [Z, infoSTABSIGNM] = ml_signm((A + eye(n)) \ (A - eye(n)), [], ...
        opts.stabsignmopts);

    if opts.UnstabDim > 0
        ns = n - opts.UnstabDim;
    else
        ns = 0.5 * (n - round(trace(Z)));
    end

    % Assign information about sign function.
    info.infoSTABSIGNM = infoSTABSIGNM;
end

if (ns > 0) && (ns < n)
    [Q, ~, ~] = qr(eye(n) - Z);

    A = Q(: , ns+1:end)' * A * Q(: , ns+1:end);
    B = Q(: , ns+1:end)' * B;
elseif ns == n
    [A, B, Q] = deal([]);
else
    Q = [];
end

nu = n - ns;


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PARTIAL STABILIZATION (I).                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Bass' Algorithm.
[X, infoSTABMETH] = ml_dlyap_smith(opts.Beta * eye(nu), (2 * B) * B', ...
    A, opts.stabmethodopts);
K1                = B' * (pinv(X + B * B') * A);

% Assign information about stabilization method.
info.infoSTABMETH = infoSTABMETH;

% Test of stability.
A              = A - B * K1;
opts.UnstabDim = sum(abs(eig(A)) > 1);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PARTIAL STABILIZATION (II).                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if opts.UnstabDim > 0
    [Z, infoSTABSIGNM2] = ml_signm((A + eye(nu)) \ (A - eye(nu)), [], ...
        opts.stabsignmopts);
    ns2                 = 0.5 * (nu - round(trace(Z)));
    nu2                 = nu - ns2;

    % Assign information about sign function.
    info.infoSTABSIGNM2 = infoSTABSIGNM2;

    if (ns2 > 0) && (ns2 < nu)
        [Q2, ~, ~] = qr(eye(nu) - Z);

        A = Q2(: , ns2+1:end)' * A * Q2(: , ns2+1:end);
        B = Q2(: , ns2+1:end)' * B;
    elseif (ns2 == 0) || (ns2 == nu)
        error('MORLAB:partstab', ...
            'Partial stabilization of the system failed!');
    end

    opts.Beta          = 1 / norm(A, 'fro');
    [X, infoSTABMETH2] = ml_dlyap_smith(opts.Beta * eye(nu2), ...
        (2 * B) * B', A, opts.stabmethodopts);
    K2                 = B' * (pinv(X + B*B') * A);

    % Assign information about stabilization method.
    info.infoSTABMETH2 = infoSTABMETH2;

    if sum(abs(eig(A - B * K2)) > 1)
        error('MORLAB:partstab', ...
            'Partial stabilization of the system failed!');
    end

    K = [zeros(m, ns), K1 + [zeros(m, ns2), K2] * Q2'];
    if not(isempty(Q))
        K = K * Q';
    end
else
    K = [zeros(m, ns), K1];
    if not(isempty(Q))
        K = K * Q';
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN INFORMATION.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Collection of all information.
info.Ns = ns;
info.Nu = nu;

[~, perm] = sort(lower(fieldnames(info)));
info      = orderfields(info, perm);
