function [K, info] = ml_dt_d_dss_partstab(A, B, E, opts)
%ML_DT_D_DSS_PARTSTAB Stabilizing feedback for descriptor systems.
%
% SYNTAX:
%   [K, info] = ML_DT_D_DSS_PARTSTAB(A, B, E)
%   [K, info] = ML_DT_D_DSS_PARTSTAB(A, B, E, opts)
%
% DESCRIPTION:
%   Partial stabilization is used for the discrete-time system of
%   differential-algebraic equations
%
%       Ex(t+1) = Ax(t) + Bu(t)                                         (1)
%
%   to get a stabilizing feedback term K, such that all finite eigenvalues
%   of s*E - A + B*K are inside the open unit disk. It is assumed that
%   the pencil s*E - A has no eigenvalues on the unit disk.
%
% INPUTS:
%   A    - matrix with dimensions n x n in (1)
%   B    - matrix with dimensions n x m in (1)
%   E    - matrix with dimensions n x n in (1)
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Beta            | positive scalar < 1, used as shift of the         |
%   |                 | Bass' algorithm for better conditioning           |
%   |                 | (default 0.1)                                     |
%   +-----------------+---------------------------------------------------+
%   | DecompEig       | positive scalar, overestimation of the absolute   |
%   |                 | value of the largest finite eigenvalue of s*E - A,|
%   |                 | if set, replaces the computation with DecompTol   |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | DecompTol       | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of E to determine the      |
%   |                 | smallest non-quasi-zero singular value of E       |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | infdecopts      | structure, containing the optional parameters for |
%   |                 | the decomposition of the finite and infinite parts|
%   |                 | of the system using the disk function and subspace|
%   |                 | extraction method, see ml_disk and ml_getqz       |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabdecopts     | structure, containing the optional parameters for |
%   |                 | the decomposition of the stable and unstable parts|
%   |                 | of the system using the sign function and subspace|
%   |                 | extraction method, see ml_signm and ml_getqz      |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabmethodopts  | structure, containing the optional parameters for |
%   |                 | the Smith iteration based Lyapunov equation solver|
%   |                 | used for the stabilization, see ml_dlyap_smith    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   K    - stabilizing feedback matrix of dimensions m x n
%   info - structure, containing the following information about the
%          generalized partial stabilization method
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | infoINFDISK     | structure, containing information about the disk  |
%   |                 | function method used for the separation of the    |
%   |                 | infinite part, see ml_disk                        |
%   +-----------------+---------------------------------------------------+
%   | infoSTABSIGNM   | structure, containing information about the sign  |
%   |                 | function method used for the separation of the    |
%   |                 | unstable part, see ml_signm                       |
%   +-----------------+---------------------------------------------------+
%   | infoSTABSIGNM2  | structure, containing information about the sign  |
%   |                 | function method used for a second separation of   |
%   |                 | the unstable part if necessary, see ml_signm      |
%   +-----------------+---------------------------------------------------+
%   | infoSTABMETH    | structure, containing information about the Smith |
%   |                 | iteration based solver used for the stabilization,|
%   |                 | see ml_dlyap_smith                                |
%   +-----------------+---------------------------------------------------+
%   | infoSTABMETH2   | structure, containing information about the Smith |
%   |                 | iteration based solver used for a second          |
%   |                 | stabilization if necessary, see ml_dlyap_smith    |
%   +-----------------+---------------------------------------------------+
%   | Ninf            | Number of identified infinite eigenvalues         |
%   +-----------------+---------------------------------------------------+
%   | Ns              | Number of identified stable eigenvalues           |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Number of identified anti-stable eigenvalues      |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, Partial stabilization of descriptor systems using spectral
%   projectors, in: P. Van Dooren, S. P. Bhattacharyya, R. H. Chan, V.
%   Olshevsky, A.Routray (Eds.), Numerical Linear Algebra in Signals,
%   Systems and Control, Vol. 80 of Lect. Notes Electr. Eng., Springer
%   Netherlands, 2011, pp. 55--76.
%   https://doi.org/10.1007/978-94-007-0602-6_3
%
% See also ml_dt_d_ss_partstab, ml_ct_d_dss_partstab.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(3, 4);

if (nargin < 4) || isempty(opts)
    opts = struct();
end

% Check input matrices.
n = size(A, 1);
m = size(B, 2);

assert(isequal(size(A), [n n]), ...
    'MORLAB:data', ...
    'The matrix A has to be square!');

assert(size(B, 1) == n, ...
    'MORLAB:data', ...
    'The matrix B must have the same number of rows as A!');

assert(isequal(size(E), [n n]), ...
    'MORLAB:data', ...
    'The matrix E must have the same dimensions as A!');

if issparse(A), A = full(A); end
if issparse(B), B = full(B); end
if issparse(E), E = full(E); end

% Check and assign optional parameters.
assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The parameter opts has to be a struct!');

if ml_field_set_to_value(opts, 'Beta')
    ml_assert_posscalar(opts.Beta, 'opts.Beta');

    assert(opts.Beta < 1, ...
        'MORLAB:data', ...
        'The parameter opts.Beta needs to be smaller than 1!');
else
    opts.Beta = 0.1;
end

if ml_field_set_to_value(opts, 'DecompEig')
    ml_assert_posscalar(opts.DecompEig, 'opts.DecompEig');
else
    opts.DecompEig = [];
end

if ml_field_set_to_value(opts, 'DecompTol')
    ml_assert_nonnegscalar(opts.DecompTol, 'opts.DecompTol');
else
    opts.DecompTol = log(n) * eps;
end

if ml_field_set_to_value(opts, 'infdecopts')
    assert(isa(opts.infdecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.infdecopts has to be a struct!');
else
    opts.infdecopts = struct();
end

if ml_field_set_to_value(opts, 'stabdecopts')
    assert(isa(opts.stabdecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.stabdecopts has to be a struct!');
else
    opts.stabdecopts = struct();
end

if ml_field_set_to_value(opts, 'stabmethodopts')
    assert(isa(opts.stabmethodopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.stabmethodopts has to be a struct!');
else
    opts.stabmethodopts = struct();
end

% Additional check of dimensions for avoiding unnecessary calls.
if ml_field_set_to_value(opts.infdecopts, 'Dimension')
    ml_assert_integer(opts.infdecopts.Dimension, ...
        'opts.infdecopts.Dimension');
else
    opts.infdecopts.Dimension = -1;
end

if ml_field_set_to_value(opts.stabdecopts, 'Dimension')
    ml_assert_integer(opts.stabdecopts.Dimension, ...
        'opts.stabdecopts.Dimension');
else
    opts.stabdecopts.Dimension = -1;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION: INFINITE PART.                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition into finite and infinite parts.
if opts.infdecopts.Dimension == 0
    ninf = 0;
else
    if isempty(opts.DecompEig) || (opts.DecompEig == 0)
        svdE  = svd(E);
        alpha = svdE(find(svdE > svdE(1) * opts.DecompTol, 1, 'last')) ...
            / (2 * norm(A, 'fro'));

        if isempty(alpha) || isinf(alpha)
            alpha = 1.0;
        end
    else
        alpha = 1 / (2 * opts.DecompEig);
    end

    [Aspace, Espace, infoINFDISK] = ml_disk(E, alpha * A, opts.infdecopts);
    [Q, Z, ninf]                  = ml_getqz(A, E, Aspace, Espace, ...
        opts.infdecopts);

    % Assign information about disk function.
    info.infoINFDISK = infoINFDISK;
end

if (ninf > 0) && (ninf < n)
    A = Q(: , ninf+1:end)' * A * Z(: , ninf+1:end);
    E = Q(: , ninf+1:end)' * E * Z(: , ninf+1:end);
    B = Q(: , ninf+1:end)' * B;
elseif ninf == n
    [A, B, E, Z] = deal([]);
else
    Z = [];
end

n = n - ninf;


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION: ANTI-STABLE PART.                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition into finite stable and unstable parts.
if opts.stabdecopts.Dimension >= 0
    opts.stabdecopts.Dimension = n - opts.stabdecopts.Dimension;
end

if (opts.stabdecopts.Dimension == 0) || (n == 0)
    ns = 0;
else
    AE                      = A + E;
    [Aspace, infoSTABSIGNM] = ml_signm(A - E, AE, opts.stabdecopts);
    [U, V, ns]              = ml_getqz(A, E, Aspace + AE,  Aspace - AE, ...
        opts.stabdecopts);

    % Assign information about disk function.
    info.infoSTABSIGNM = infoSTABSIGNM;
end

if (ns > 0) && (ns < n)
    A = U(: , ns+1:end)' * A * V(: , ns+1:end);
    E = U(: , ns+1:end)' * E * V(: , ns+1:end);
    B = U(: , ns+1:end)' * B;
elseif ns == n
    [A, B, E, V] = deal([]);
else
    V = [];
end

n = n - ns;


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PARTIAL STABILIZATION (I).                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Bass' Algorithm.
[X, infoSTABMETH] = ml_dlyap_smith(opts.Beta * E, (2 * B) * B', ...
    A, opts.stabmethodopts);
K1                = B' * (pinv(E * (X * E') + B * B') * A);

% Assign information about stabilization method.
info.infoSTABMETH = infoSTABMETH;

% Test of stability.
A                          = A - B * K1;
opts.stabdecopts.Dimension = n - sum(abs(eig(A, E)) > 1);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PARTIAL STABILIZATION (II).                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if opts.stabdecopts.Dimension < n
    AE                       = A + E;
    [Aspace, infoSTABSIGNM2] = ml_signm(A - E, AE, opts.stabdecopts);
    [U2, V2, ns2]            = ml_getqz(A, E, Aspace + AE, Aspace - AE, ...
        opts.stabdecopts);

    % Assign information about disk function.
    info.infoSTABSIGNM2 = infoSTABSIGNM2;

    if (ns2 > 0) && (ns2 < n)
        A = U2(: , ns2+1:end)' * A * V2(: , ns2+1:end);
        E = U2(: , ns2+1:end)' * E * V2(: , ns2+1:end);
        B = U2(: , ns2+1:end)' * B;
    elseif (ns2 == 0) || (ns2 == n)
        error('MORLAB:gpartstab', ...
            'Partial stabilization of the system failed!');
    end

    % Bass' Algorithm.
    [X, infoSTABMETH2] = ml_dlyap_smith(opts.Beta * E, (2 * B) * B', ...
        A, opts.stabmethodopts);
    K2                = B' * (pinv(E * (X * E') + B * B') * A);

    % Assign information about stabilization method.
    info.infoSTABMETH2 = infoSTABMETH2;

    if sum(abs(eig(A - B * K2, E)) > 1)
        error('MORLAB:gpartstab', ...
            'Partial stabilization of the system failed!');
    end

    if isempty(V)
        K = [zeros(m, ninf), [zeros(m, ns), K1 + ...
            [zeros(m, ns2), K2] * V2']];
    else
        K = [zeros(m, ninf), [zeros(m, ns), K1 + ...
            [zeros(m, ns2), K2] * V2'] * V'];
    end
    if not(isempty(Z))
        K = K * Z';
    end
else
    if isempty(V)
        K = [zeros(m, ninf), [zeros(m, ns), K1]];
    else
        K = [zeros(m, ninf), [zeros(m, ns), K1] * V'];
    end
    if not(isempty(Z))
        K = K * Z';
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN INFORMATION.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Collection of all information.
info.Ninf   = ninf;
info.Ns     = ns;
info.Nu     = n;

[~, perm] = sort(lower(fieldnames(info)));
info      = orderfields(info, perm);
