function [sys, info] = ml_dt_d_dss_adtf(sys, opts)
%ML_DT_D_DSS_ADTF Add. dec. of dense descriptor system transfer functions.
%
% SYNTAX:
%   [sys, info] = ML_DT_D_DSS_ADTF(sys)
%   [sys, info] = ML_DT_D_DSS_ADTF(sys, opts)
%
% DESCRIPTION:
%   Consider a discrete-time descriptor system of the form
%
%       E*x(t+1) = A*x(t) + B*u(t),                                     (1)
%           y(t) = C*x(t).                                              (2)
%
%   This function computes an additive decomposition of the corresponding
%   transfer function by a block diagonalization of the matrix pencil
%   s*E - A. The descriptor system is transformed, such that
%
%       [ Ef         ]         [ Af           ]       [ Bf   ]
%       [    Eu      ]z(t+1) = [     Au       ]z(t) + [ Bu   ]u(t),     (3)
%       [       Einf ]         [         Ainf ]       [ Binf ]
%
%                      y(t) = [ Cf, Cu, Cinf ]z(t),                     (4)
%
%   where s*Ef - Af contains the finite eigenvalues with absolute value
%   smaller than 1, s*Einf - Ainf the infinite eigenvalues and s*Eu - Au
%   the finite eigenvalues with absolute value larger than 1.
%
% INPUTS:
%   sys  - structure, containing the descriptor system in the form:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | E               | matrix with dimensions n x n in (1)               |
%   +-----------------+---------------------------------------------------+
%   | A               | matrix with dimensions n x n in (1)               |
%   +-----------------+---------------------------------------------------+
%   | B               | matrix with dimensions n x m in (1)               |
%   +-----------------+---------------------------------------------------+
%   | C               | matrix with dimensions p x n in (2)               |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | DecompEig       | positive scalar, overestimation of the absolute   |
%   |                 | value of the largest finite eigenvalue of s*E - A,|
%   |                 | if set, replaces the computation with DecompTol   |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | DecompTol       | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of E to determine the      |
%   |                 | smallest non-zero singular value of E             |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | infdecopts      | structure, containing the optional parameters for |
%   |                 | the decomposition of the finite and infinite parts|
%   |                 | of the system using the disk function and subspace|
%   |                 | extraction method, see ml_disk, ml_getqz          |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabdecopts     | structure, containing the optional parameters for |
%   |                 | the decomposition of the stable and unstable parts|
%   |                 | of the system using the sign function and subspace|
%   |                 | extraction method, see ml_signm and ml_getqz      |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   sys - structure, containing the transformed descriptor system:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | E               | matrix with dimensions nf x nf, see Ef in (3)     |
%   +-----------------+---------------------------------------------------+
%   | Einf            | matrix with dimensions ninf x ninf in (3)         |
%   +-----------------+---------------------------------------------------+
%   | Eu              | matrix with dimensions nu x nu in (3)             |
%   +-----------------+---------------------------------------------------+
%   | A               | matrix with dimensions nf x nf, see Af in (3)     |
%   +-----------------+---------------------------------------------------+
%   | Ainf            | matrix with dimensions ninf x ninf in (3)         |
%   +-----------------+---------------------------------------------------+
%   | Au              | matrix with dimensions nu x nu in (3)             |
%   +-----------------+---------------------------------------------------+
%   | B               | matrix with dimensions nf x m, see Bf in (3)      |
%   +-----------------+---------------------------------------------------+
%   | Binf            | matrix with dimensions ninf x m in (3)            |
%   +-----------------+---------------------------------------------------+
%   | Bu              | matrix with dimensions nu x m in (3)              |
%   +-----------------+---------------------------------------------------+
%   | C               | matrix with dimensions p x nf, see Cf in (4)      |
%   +-----------------+---------------------------------------------------+
%   | Cinf            | matrix with dimensions p x ninf in (4)            |
%   +-----------------+---------------------------------------------------+
%   | Cu              | matrix with dimensions p x nu in (4)              |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information about the
%          additive decomposition of the descriptor system
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | infoINFDISK     | structure, containing information about the disk  |
%   |                 | function method used for the separation of the    |
%   |                 | infinite part, see ml_disk                        |
%   +-----------------+---------------------------------------------------+
%   | infoSTABSIGNM   | structure, containing information about the sign  |
%   |                 | function method used for the separation of the    |
%   |                 | unstable part, see ml_signm                       |
%   +-----------------+---------------------------------------------------+
%   | Ninf            | Number of identified infinite eigenvalues         |
%   +-----------------+---------------------------------------------------+
%   | Ns              | Number of identified stable eigenvalues           |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Number of identified anti-stable eigenvalues      |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   |                 | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   |                 | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   S. Werner, Hankel-norm approximation of descriptor systems, Master's
%   thesis, Otto von Guericke University, Magdeburg, Germany (2016).
%   http://nbn-resolving.de/urn:nbn:de:gbv:ma9:1-8845
%
% See also ml_dt_d_ss_adtf, ml_ct_d_dss_adtf, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check input descriptor system.
ml_assert_descsys(sys);

if issparse(sys.A), sys.A = full(sys.A); end
if issparse(sys.B), sys.B = full(sys.B); end
if issparse(sys.C), sys.C = full(sys.C); end
if issparse(sys.E), sys.E = full(sys.E); end

n = size(sys.A, 1);

% Check and assign optional parameters.
assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The parameter opts has to be a struct!');

if ml_field_set_to_value(opts, 'DecompEig')
    ml_assert_posscalar(opts.DecompEig, 'opts.DecompEig');
else
    opts.DecompEig = [];
end

if ml_field_set_to_value(opts, 'DecompTol')
    ml_assert_nonnegscalar(opts.DecompTol, 'opts.DecompTol');
else
    opts.DecompTol = log(n) * eps;
end

if ml_field_set_to_value(opts, 'infdecopts')
    assert(isa(opts.infdecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.infdecopts has to be a struct!');
else
    opts.infdecopts = struct();
end

if ml_field_set_to_value(opts, 'stabdecopts')
    assert(isa(opts.stabdecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.stabdecopts has to be a struct!');
else
    opts.stabdecopts = struct();
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Additional check of dimensions for avoiding unnecessary calls.
if ml_field_set_to_value(opts.infdecopts, 'Dimension')
    ml_assert_integer(opts.infdecopts.Dimension, ...
        'opts.infdecopts.Dimension');
else
    opts.infdecopts.Dimension = -1;
end

if ml_field_set_to_value(opts.stabdecopts, 'Dimension')
    ml_assert_integer(opts.stabdecopts.Dimension, ...
        'opts.stabdecopts.Dimension');
else
    opts.stabdecopts.Dimension = -1;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION: INFINITE PART.                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation of deflating subspaces using the matrix disk function.
if opts.infdecopts.Dimension == 0
    ninf = 0;
else
    if isempty(opts.DecompEig) || (opts.DecompEig == 0)
        svdE  = svd(sys.E);
        alpha = svdE(find(svdE > svdE(1) * opts.DecompTol, 1, 'last')) ...
            / (4 * norm(sys.A, 'fro'));

        if isempty(alpha)
            alpha = 1.0;
        end
    else
        alpha = 1 / (2 * opts.DecompEig);
    end

    [Aspace, Espace, infoINFDISK] = ml_disk(sys.E, alpha * sys.A, ...
        opts.infdecopts);
    [Q, Z, ninf]                  = ml_getqz(sys.A, sys.E, Aspace, ...
        Espace, opts.infdecopts);

    info.infoINFDISK = infoINFDISK;
end

% Block diagonalization of the system matrices.
if (ninf > 0) && (ninf < n)
    opts.infdecopts.Dimension = n - ninf;
    [U, T, nf]                = ml_getqz(sys.A, sys.E, Espace, Aspace, ...
        opts.infdecopts);

    sys.Ainf = U(: , nf+1:end)' * sys.A * Z(: , 1:ninf);
    sys.A    = Q(: , ninf+1:end)' * sys.A * T(: , 1:nf);

    sys.Einf = U(: , nf+1:end)' * sys.E * Z(: , 1:ninf);
    sys.E    = Q(: , ninf+1:end)' * sys.E * T(: , 1:nf);

    sys.Binf = U(: , nf+1:end)' * sys.B;
    sys.B    = Q(: , ninf+1:end)' * sys.B;

    sys.Cinf = sys.C * Z(: , 1:ninf);
    sys.C    = sys.C * T(: , 1:nf);

    if opts.StoreProjection
        V = [T(: , 1:nf), Z(: , 1:ninf)];
        W = [Q(: , ninf+1:end), U(: , nf+1:end)];
    else
        W = [];
        V = [];
    end
elseif ninf == n
    sys.Ainf = sys.A;
    sys.Einf = sys.E;
    sys.Binf = sys.B;
    sys.Cinf = sys.C;

    [sys.A, sys.E] = deal([]);
    sys.B          = zeros(0, size(sys.Binf, 2));
    sys.C          = zeros(size(sys.Cinf, 1), 0);

    nf = 0;

    if opts.StoreProjection
        V = eye(n);
        W = eye(n);
    else
        W = [];
        V = [];
    end
else
    [sys.Ainf, sys.Einf] = deal([]);
    sys.Binf             = zeros(0, size(sys.B, 2));
    sys.Cinf             = zeros(size(sys.C, 1), 0);

    nf = n;

    if opts.StoreProjection
        V = eye(n);
        W = eye(n);
    else
        W = [];
        V = [];
    end
end

n = n - ninf;


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION: ANTI-STABLE PART.                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation of deflating subspaces using the matrix disk function.
if (opts.stabdecopts.Dimension == 0) || (n == 0)
    nu = 0;
else
    AE                      = sys.A + sys.E;
    [Aspace, infoSTABSIGNM] = ml_signm(sys.A - sys.E, AE, ...
        opts.stabdecopts);
    AE1                     = Aspace - AE;
    AE2                     = Aspace + AE;
    [Q, Z, nu]              = ml_getqz(sys.A, sys.E, AE1, AE2, ...
        opts.stabdecopts);

    info.infoSTABSIGNM = infoSTABSIGNM;
end

% Block diagonalization of the system matrices.
if (nu > 0) && (nu < n)
    opts.stabdecopts.Dimension = n - nu;
    [U, T, nf]                 = ml_getqz(sys.A, sys.E, AE2, AE1, ...
        opts.stabdecopts);

    sys.Au = U(: , nf+1:end)' * sys.A * Z(: , 1:nu);
    sys.A  = Q(: , nu+1:end)' * sys.A * T(: , 1:nf);

    sys.Eu = U(: , nf+1:end)' * sys.E * Z(: , 1:nu);
    sys.E  = Q(: , nu+1:end)' * sys.E * T(: , 1:nf);

    sys.Bu = U(: , nf+1:end)' * sys.B;
    sys.B  = Q(: , nu+1:end)' * sys.B;

    sys.Cu = sys.C * Z(: , 1:nu);
    sys.C  = sys.C * T(: , 1:nf);

    if opts.StoreProjection
        V = V * blkdiag([T(: , 1:nf), Z(: , 1:nu)], eye(ninf));
        W = W * blkdiag([Q(: , nu+1:end), U(: , nf+1:end)], eye(ninf));
    else
        W = [];
        V = [];
    end
elseif nu == n
    sys.Au = sys.A;
    sys.Eu = sys.E;
    sys.Bu = sys.B;
    sys.Cu = sys.C;

    [sys.A, sys.E] = deal([]);
    sys.B          = zeros(0, size(sys.Bu, 2));
    sys.C          = zeros(size(sys.Cu, 1), 0);

    if opts.StoreProjection
        V = V * eye(n + ninf);
        W = eye(n + ninf) * W;
    else
        W = [];
        V = [];
    end
else
    [sys.Au, sys.Eu] = deal([]);
    sys.Bu           = zeros(0, size(sys.B, 2));
    sys.Cu           = zeros(size(sys.C, 1), 0);

    if opts.StoreProjection
        V = V * eye(n + ninf);
        W = eye(n + ninf) * W;
    else
        W = [];
        V = [];
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN INFORMATION.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Collection of all information.
info.Ninf = ninf;
info.Ns   = nf;
info.Nu   = nu;
info.V    = V;
info.W    = W;

[~, perm] = sort(lower(fieldnames(info)));
info      = orderfields(info, perm);
