function [sys, opts, ioformat] = ml_decide_system_type(tformat, varargin)
% ML_DECIDE_SYSTEM_TYPE Analyse system structure and rearrange into struct.
%
% SYNTAX:
%   [sys, opts, ioformat] = ML_DECIDE_SYSTEM_TYPE(tformat, A, B, C, D)
%   [sys, opts, ioformat] = ...
%                          ML_DECIDE_SYSTEM_TYPE(tformat, A, B, C, D, opts)
%   [sys, opts, ioformat] = ML_DECIDE_SYSTEM_TYPE(tformat, A, B, C, D, E)
%   [sys, opts, ioformat] = ...
%                       ML_DECIDE_SYSTEM_TYPE(tformat, A, B, C, D, E, opts)
%   [sys, opts, ioformat] = ...
%                    ML_DECIDE_SYSTEM_TYPE(tformat, M, E, K, Bu, Cp, Cv, D)
%   [sys, opts, ioformat] = ...
%              ML_DECIDE_SYSTEM_TYPE(tformat, M, E, K, Bu, Cp, Cv, D, opts)
%
%   [sys, opts, ioformat] = ML_DECIDE_SYSTEM_TYPE(tformat, sys)
%   [sys, opts, ioformat] = ML_DECIDE_SYSTEM_TYPE(tformat, sys, opts)
%
% DESCRIPTION:
%   This function checks and converts a given, supported system into a
%   struct for internal use in MORLAB. If not given, the optional parameter
%   struct is initiated and the used format by the user is saved for later.
%   In the case of sparse coefficient matrices, a permutation might be
%   applied to analyze potential DAE structures.
%
% INPUTS:
%   tformat - string, used to denote the time format, e.g.,
%             'ct' for continuous-time and 'dt' for discrete-time
%   A       - matrix with dimensions n x n
%   B       - matrix with dimensions n x m
%   C       - matrix with dimensions p x n
%   D       - matrix with dimensions p x m, allowed to be empty
%   E       - matrix with dimensions n x n
%   M       - matrix with dimensions n x n
%   K       - matrix with dimensions n x n
%   Bu      - matrix with dimensions n x m
%   Cp      - matrix with dimensions p x n, allowed to be empty
%   Cv      - matrix with dimensions p x n, allowed to be empty
%   sys     - struct, ss, sparss or mechss object containing the system
%             matrices
%   opts    - struct, containing the optional parameters
%
% OUTPUTS:
%   sys      - struct, containing the system matrices, the field
%              sys.SystemType is added with a string determining type and
%              structure of the system, sys.DecidedSystemType to avoid
%              re-running structure checks and sys.SystemPermL and
%              sys.SystemPermR if a permutation has been applied to the
%              system matrices
%   opts     - struct, containing the optional parameters
%   ioformat - integer, describing the input format
%                0 - for input type matrices
%                1 - for input type struct
%                2 - for input type state-space object
%
% See also ml_format_output, ml_prepare_system_data.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(2, 9);

ml_assert_char(tformat, 'tformat');


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CONVERT TO UNIFORM STRUCT.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Rearrange matrices into struct.
if isa(varargin{1}, 'double') % Input case: matrices.
    ioformat             = 0;
    [sys, systype, opts] = set_sys_from_matrices(varargin{:});
elseif isa(varargin{1}, 'struct') % Input case: structs.
    ioformat             = 1;
    [sys, systype, opts] = set_sys_from_struct(varargin{:});
elseif isa(varargin{1}, 'ss') || isa(varargin{1}, 'sparss') % (sparse) ss.
    ioformat             = 2;
    [sys, systype, opts] = set_sys_from_ss(varargin{:});
elseif isa(varargin{1}, 'mechss') % Input case: mechss.
    ioformat             = 2;
    [sys, systype, opts] = set_sys_from_mechss(varargin{:});
else
    error('MORLAB:data', ...
        'The given type of input data is not implemented!');
end

% Check option struct.
assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The parameter opts has to be a struct!');

% Analyse matrix structure.
if not(ml_field_set_to_value(sys, 'DecidedSystemType')) ...
        || not(sys.DecidedSystemType)
    if (isfield(sys, 'A') && issparse(sys.A)) || ...
            (isfield(sys, 'K') && issparse(sys.K))
        sparsedense = '_s_';
    else
        sparsedense = '_d_';
    end

    if strcmpi(sparsedense, '_s_')
        [usfs, sys]    = determine_sparse_structure(sys, systype);
        sys.SystemType = [tformat sparsedense systype '_' usfs];
    else
        sys.SystemType = [tformat sparsedense systype];
    end

    sys.DecidedSystemType = true;
end

end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% HELPER FUNCTION: SET_SYS_FROM_MATRICES.                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [sys, systype, opts] = set_sys_from_matrices(varargin)
%SET_SYS_FROM_MATRICES Construct system structure from matrices.
%
% SYNTAX:
%   [sys, systype, opts] = SET_SYS_FROM_MATRICES(A, B, C, D)
%   [sys, systype, opts] = SET_SYS_FROM_MATRICES(A, B, C, D, opts)
%
%   [sys, systype, opts] = SET_SYS_FROM_MATRICES(A, B, C, D, E)
%   [sys, systype, opts] = SET_SYS_FROM_MATRICES(A, B, C, D, E, opts)
%
%   [sys, systype, opts] = SET_SYS_FROM_MATRICES(M, E, K, Bu, Cp, Cv, D)
%   [sys, systype, opts] = ...
%                       SET_SYS_FROM_MATRICES(M, E, K, Bu, Cp, Cv, D, opts)
%
% DESCRIPTION:
%   This function checks given matrices to determine the basic system type
%   as standard, descriptor or second-order system and constructs an
%   appropriate structure.
%
% INPUTS:
%   A    - matrix with dimensions n x n
%   B    - matrix with dimensions n x m
%   C    - matrix with dimensions p x n
%   D    - matrix with dimensions p x m, allowed to be empty
%   E    - matrix with dimensions n x n
%   M    - matrix with dimensions n x n
%   K    - matrix with dimensions n x n
%   Bu   - matrix with dimensions n x m
%   Cp   - matrix with dimensions p x n, allowed to be empty
%   Cv   - matrix with dimensions p x n, allowed to be empty
%   opts - struct containing the optional parameters
%
% OUTPUTS:
%   sys     - structure containing the system matrices
%   systype - string denoting the system type
%              'ss'   - standard state-space system
%              'dss'  - generalized/descriptor system
%              'soss' - second-order system
%   opts    - struct containing the optional parameters

narginchk(3, 8);

if (nargin < 5) || not(isa(varargin{5}, 'double')) % Standard system.
    systype = 'ss';

    sys = struct( ...
        'A', varargin{1}, ...
        'B', varargin{2}, ...
        'C', varargin{3});

    if nargin < 4
        sys.D = [];
    else
        sys.D = varargin{4};
    end

    nin  = size(sys.B, 2);
    nout = size(sys.C, 1);

    if (nargin < 5) || isempty(varargin{5})
        opts = struct();
    else
        opts = varargin{5};
    end

    ml_assert_stdsys(sys);

elseif  (nargin < 6) || not(isa(varargin{6}, 'double')) % Generalized sys.
    systype = 'dss';

    % Assign and check standard system.
    sys = struct( ...
        'A', varargin{1}, ...
        'B', varargin{2}, ...
        'C', varargin{3}, ...
        'D', varargin{4}, ...
        'E', varargin{5});

    nin  = size(sys.B, 2);
    nout = size(sys.C, 1);

    if (nargin < 6) || isempty(varargin{6})
        opts = struct();
    else
        opts = varargin{6};
    end

    ml_assert_descsys(sys);

else  % Second-order system.
    systype = 'soss';

    % Assign and check standard system.
    sys = struct( ...
        'M' , varargin{1}, ...
        'E' , varargin{2}, ...
        'K' , varargin{3}, ...
        'Bu', varargin{4}, ...
        'Cp', varargin{5}, ...
        'Cv', varargin{6});

    nin  = size(sys.Bu, 2);
    nout = max(size(sys.Cp, 1), size(sys.Cv, 1));

    if nargin < 7
        sys.D = [];
    else
        sys.D = varargin{7};
    end

    if (nargin < 8) || isempty(varargin{8})
        opts = struct();
    else
        opts = varargin{8};
    end

    ml_assert_sosys(sys);
end

if ml_field_set_to_value(sys, 'D')
    assert(isequal(size(sys.D), [nout, nin]), ...
        'MORLAB:data', ...
        'The matrix D must have the dimensions %d x %d!', ...
        nout, nin);
end

end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% HELPER FUNCTION: SET_SYS_FROM_STRUCT.                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [sys, systype, opts] = set_sys_from_struct(varargin)
%SET_SYS_FROM_STRUCT Checks given system structure for type.
%
% SYNTAX:
%   [sys, systype, opts] = SET_SYS_FROM_STRUCT(sys)
%   [sys, systype, opts] = SET_SYS_FROM_STRUCT(sys, opts)
%
% DESCRIPTION:
%   This function checks a given struct to determine the basic
%   system type as standard, descriptor or second-order system.
%
% INPUTS:
%   sys  - structure, containing the system matrices, only the entries
%          defining the system are needed:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix with dimensions n x m                      |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix with dimensions p x n                      |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix with dimensions p x m, allowed to be empty |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix with dimensions n x m                      |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix with dimensions p x n, allowed to be empty |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix with dimensions p x n, allowed to be empty |
%   +-----------------+---------------------------------------------------+
%   opts - struct containing the optional parameters
%
% OUTPUTS:
%   sys     - structure containing the system matrices
%   systype - string denoting the system type
%              'ss'   - standard state-space system
%              'dss'  - generalized/descriptor system
%              'soss' - second-order system
%   opts    - struct containing the optional parameters

narginchk(1, 2);

if (nargin < 2) || isempty(varargin{2})
    opts = struct();
else
    opts = varargin{2};
end

sys = varargin{1};

if isfield(sys, 'K') % Second-order system.
    ml_assert_sosys(sys);

    if not(ml_field_set_to_value(sys, 'Cp'))
        sys.Cp = [];
    end

    if not(ml_field_set_to_value(sys, 'Cv'))
        sys.Cv = [];
    end

    systype = 'soss';
    nin     = size(sys.Bu, 2);
    nout    = max(size(sys.Cp, 1), size(sys.Cv, 1));
elseif isfield(sys, 'E') % Generalized/descriptor system.
    ml_assert_descsys(sys);

    systype = 'dss';
    nin     = size(sys.B, 2);
    nout    = size(sys.C, 1);
else % Standard system.
    ml_assert_stdsys(sys);

    systype = 'ss';
    nin     = size(sys.B, 2);
    nout    = size(sys.C, 1);
end

if ml_field_set_to_value(sys, 'D')
    assert(isequal(size(sys.D), [nout, nin]), ...
        'MORLAB:data', ...
        'The matrix D must have the dimensions %d x %d!', ...
        nout, nin);
else
    sys.D = [];
end

end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% HELPER FUNCTION: SET_SYS_FROM_SS.                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [sys, systype, opts] = set_sys_from_ss(varargin)
%SET_SYS_FROM_SS Construct system structure from ss or sparss object.
%
% SYNTAX:
%   [sys, systype, opts] = SET_SYS_FROM_SS(sys)
%   [sys, systype, opts] = SET_SYS_FROM_SS(sys, opts)
%
% DESCRIPTION:
%   This function checks a given ss or sparss obejct to determine the basic
%   system type as standard or descriptor system and gives back a struct
%   containing the system matrices.
%
% INPUTS:
%   sys  - ss or sparss object, containing the system matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix with dimensions n x m                      |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix with dimensions p x n                      |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix with dimensions p x m                      |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix with dimensions n x n, allowed to be empty |
%   +-----------------+---------------------------------------------------+
%   opts - struct containing the optional parameters
%
% OUTPUTS:
%   sys     - structure containing the system matrices
%   systype - string denoting the system type
%              'ss'   - standard state-space system
%              'dss'  - generalized/descriptor system
%   opts    - struct containing the optional parameters

narginchk(1, 2);

if (nargin < 2) || isempty(varargin{2})
    opts = struct();
else
    opts = varargin{2};
end

myss = varargin{1};
nin  = size(myss.B, 2);
nout = size(myss.C, 1);

if isempty(myss.E) % Standard system.
    sys = struct( ...
        'A', myss.A, ...
        'B', myss.B, ...
        'C', myss.C, ...
        'D', myss.D);

    ml_assert_stdsys(sys);

    systype = 'ss';

else % Generalized/descriptor system.
    sys = struct( ...
        'A', myss.A, ...
        'B', myss.B, ...
        'C', myss.C, ...
        'D', myss.D, ...
        'E', myss.E);

    ml_assert_descsys(sys);

    systype = 'dss';
end

if ml_field_set_to_value(sys, 'D')
    assert(isequal(size(sys.D), [nout, nin]), ...
        'MORLAB:data', ...
        'The matrix D must have the dimensions %d x %d!', ...
        nout, nin);
end

% Save sampling time in structure.
if not(myss.Ts == 0)
    sys.Ts = myss.Ts;
end

end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% HELPER FUNCTION: SET_SYS_FROM_MECHSS.                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [sys, systype, opts] = set_sys_from_mechss(varargin)
%SET_SYS_FROM_MECHSS Construct system structure from mechss object.
%
% SYNTAX:
%   [sys, systype, opts] = SET_SYS_FROM_SS(sys)
%   [sys, systype, opts] = SET_SYS_FROM_SS(sys, opts)
%
% DESCRIPTION:
%   This function checks a given mechss obejct and gives back a struct
%   containing the system matrices.
%
% INPUTS:
%   sys  - ss or sparss object, containing the system matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix with dimensions n x m                      |
%   +-----------------+---------------------------------------------------+
%   |        F        | matrix with dimensions p x n                      |
%   +-----------------+---------------------------------------------------+
%   |        G        | matrix with dimensions p x n                      |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix with dimensions p x m                      |
%   +-----------------+---------------------------------------------------+
%   opts - struct containing the optional parameters
%
% OUTPUTS:
%   sys     - structure containing the system matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix with dimensions n x m                      |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix with dimensions p x n, allowed to be empty |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix with dimensions p x n, allowed to be empty |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix with dimensions p x m, allowed to be empty |
%   +-----------------+---------------------------------------------------+
%   systype - string denoting the system type
%              'soss' - second-order system
%   opts    - struct containing the optional parameters

narginchk(1, 2);

if (nargin < 2) || isempty(varargin{2})
    opts = struct();
else
    opts = varargin{2};
end

myss = varargin{1};
nin  = size(myss.B, 2);
nout = max(size(myss.F, 1), size(myss.G, 1));

sys = struct( ...
    'M' , myss.M, ...
    'E' , myss.C, ...
    'K' , myss.K, ...
    'Bu', myss.B, ...
    'Cp', myss.F, ...
    'Cv', myss.G, ...
    'D' , myss.D);

ml_assert_sosys(sys);

systype = 'soss';

if ml_field_set_to_value(sys, 'D')
    assert(isequal(size(sys.D), [nout, nin]), ...
        'MORLAB:data', ...
        'The matrix D must have the dimensions %d x %d!', ...
        nout, nin);
end

% Save sampling time in structure.
if not(myss.Ts == 0)
    sys.Ts = myss.Ts;
end

end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% HELPER FUNCTION: DETERMINE_SPARSE_STRUCTURE.                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [usfs, sys] = determine_sparse_structure(sys, systype)
%DETERMINE_SPARSE_STRUCTURE Analyse sparse structures for M-M.E.S.S. calls.
%
% SYNTAX:
%   [usfs, sys] = DETERMINE_SPARSE_STRUCTURE(sys, systype)
%
% DESCRIPTION:
%   This function analyses the structure of the given system matrices in
%   the sparse case and selects implemented structures to handle ODE and
%   DAE systems.
%
% INPUTS:
%   sys     - struct containing the system matrices
%   systype - string denoting the system type
%              'ss'   - standard state-space system
%              'dss'  - generalized/descriptor system
%              'soss' - second-order system
%
% OUTPUTS:
%   usfs - string denoting an appropriate function handle set for the found
%          structure
%   sys  - struct containing potentially permuted system matrices

switch systype
    case 'dss' % Generalized/descriptor system.
        dE = abs(diag(sys.E));

        if double(sprank(sys.E)) == size(sys.E, 1)
            sE = sum(abs(sys.E), 2);
            sE = sE - dE;

            usfs = 'default';

            if not(all(dE > sE)) && (condest(sys.E) > (1 / eps))
                warning('MORLAB:data', ...
                    ['Ill-conditioned mass matrix detected!',...
                    ' Perturbation of E by small multiples of',...
                    ' the identity may be necessary.']);
            end
        else % E is guaranteed rank-deficient.
            [dE, perm] = sort(dE, 'descend');
            E          = sys.E(perm, perm);
            zE         = find(dE == 0);

            if not(isempty(zE)) ...
                    && isempty(nonzeros(E(zE, :))) ...
                    && isempty(nonzeros(E(:, zE)))
                % Symmetric permutation to get E11 block.
                sys.E           = E;
                sys.A           = sys.A(perm, perm);
                sys.B           = sys.B(perm, :);
                sys.C           = sys.C(:, perm);
                sys.SystemPermL = perm;
                sys.SystemPermR = perm;
                nE              = 1:(zE(1) - 1);
                sys.zE          = zE(:);
                sys.nE          = nE(:);

                if condest(sys.E(nE, nE)) > (1 / eps)
                    warning('MORLAB:data', ...
                        'Ill-conditioned DAE structure detected!');
                end

                if double(sprank(sys.A(zE, zE))) == length(zE)
                    if condest(sys.A(zE, zE)) > (1 / eps)
                        warning('MORLAB:data', ...
                            ['Ill-conditioned index-1 structure' ...
                            ' detected! Perturbation of A by small' ...
                            ' multiples of the identity may be' ...
                            ' necessary.']);
                    end

                    usfs = 'dae_1';
                elseif isempty(nonzeros(sys.A(zE, zE))) ...
                        && (double(sprank(sys.A(nE, zE))) == length(zE)) ...
                        && (double(sprank(sys.A(zE, nE))) == length(zE)) ...
                        && not(any(any(sys.B(zE, :))) ...
                        && any(any(sys.C(:, zE))))
                    if condest(sys.A(zE, nE) * sys.A(nE, zE)) > (1 / eps)
                        warning('MORLAB:data', ...
                            ['Rank deficient index-2 structure' ...
                            ' detected!']);
                    end

                    usfs = 'dae_2';
                else
                    error('MORLAB:data', ...
                        'MORLAB cannot determine system structure!');
                end
            else
                % Nonsymmetric permutation to get potential E11 block.
                cols = find(not(any(sys.E, 1)));
                rows = find(not(any(sys.E, 2)));

                if length(cols) == length(rows)
                    n    = size(sys.A, 1);
                    permR = [setdiff(1:n, cols(:)'), cols(:)'];
                    permL = [setdiff(1:n, rows(:)'), rows(:)'];

                    sys.E           = sys.E(permL, permR);
                    sys.A           = sys.A(permL, permR);
                    sys.B           = sys.B(permL, :);
                    sys.C           = sys.C(:, permR);
                    sys.SystemPermL = permL;
                    sys.SystemPermR = permR;
                    zE              = (n - length(cols) + 1):n;
                    nE              = 1:(zE(1) - 1);
                    sys.zE          = zE(:);
                    sys.nE          = nE(:);

                    if condest(sys.E(nE, nE)) > (1 / eps)
                        warning('MORLAB:data', ...
                            'Ill-conditioned DAE structure detected!');
                    end

                    if double(sprank(sys.A(zE, zE))) == length(zE)
                        if condest(sys.A(zE, zE)) > (1 / eps)
                            warning('MORLAB:data', ...
                                ['Ill-conditioned index-1 structure' ...
                                ' detected! Perturbation of A by small' ...
                                ' multiples of the identity may be' ...
                                ' necessary.']);
                        end

                        usfs = 'dae_1';
                    elseif isempty(nonzeros(sys.A(zE, zE))) ...
                            && (double(sprank(sys.A(nE, zE))) == length(zE)) ...
                            && (double(sprank(sys.A(zE, nE))) == length(zE)) ...
                            && not(any(any(sys.B(zE, :))) ...
                            && any(any(sys.C(:, zE))))
                        if condest(sys.A(zE, nE) * sys.A(nE, zE)) > (1/eps)
                            warning('MORLAB:data', ...
                                ['Rank deficient index-2 structure' ...
                                ' detected!']);
                        end

                        usfs = 'dae_2';
                    else
                        error('MORLAB:data', ...
                            'MORLAB cannot determine system structure!');
                    end
                else
                    error('MORLAB:data', ...
                        'MORLAB cannot determine system structure!');
                end
            end
        end

    case 'soss' % Second-order system.
        dM = abs(diag(sys.M));

        if double(sprank(sys.M)) == size(sys.M, 1)
            sM = sum(abs(sys.M), 2);
            sM = sM - dM;

            usfs = 'so_1';

            if not(all(dM > sM)) && condest(sys.M) > (1 / eps)
                warning('MORLAB:data', ...
                    ['Ill-conditioned mass matrix detected!',...
                    ' Perturbation of M by small multiples of',...
                    ' the identity may be necessary.']);
            end
        else % M is guaranteed rank deficient.
            [dM, perm] = sort(dM, 'descend');
            M          = sys.M(perm, perm);
            zM         = find(dM == 0);

            if not(isempty(zM)) ...
                    && isempty(nonzeros(M(zM, :))) ...
                    && isempty(nonzeros(M(:, zM)))
                % Symmetric permutation to get M11 blocks.
                sys.M  = M;
                sys.E  = sys.E(perm, perm);
                sys.K  = sys.K(perm, perm);
                sys.Bu = sys.Bu(perm, :);
                if not(isempty(sys.Cp)), sys.Cp = sys.Cp(:, perm); end
                if not(isempty(sys.Cv)), sys.Cv = sys.Cv(:, perm); end

                sys.SystemPermL = perm;
                sys.SystemPermR = perm;
                nM              = 1:(zM(1) - 1);
                sys.zM          = zM(:);
                sys.nM          = nM(:);

                if condest(sys.M(nM, nM)) > (1 / eps)
                    warning('MORLAB:data', ...
                        ['Ill-conditioned second-order DAE structure' ...
                        ' detected!']);
                end

                if double(sprank(sys.K(zM, zM))) == length(zM) ...
                        && isempty(nonzeros(sys.E(zM, :))) ...
                        && isempty(nonzeros(sys.E(:, zM))) ...
                        && (isempty(nonzeros(sys.Bu(zM, :))) ...
                        || isempty(sys.Cv) ...
                        || not(any(any(sys.Cv(:, zM)))))
                    if condest(sys.K(zM, zM)) > (1 / eps)
                        warning('MORLAB:data', ...
                            ['Ill-conditioned second-order index-1' ...
                            ' structure detected! Perturbation of' ...
                            ' K by small multiples of the identity' ...
                            ' may be necessary.']);
                    end

                    usfs = 'dae_1_so';
                elseif isempty(nonzeros(sys.K(zM, zM))) ...
                        && (double(sprank(sys.K(nM, zM))) == length(zM)) ...
                        && (double(sprank(sys.K(zM, nM))) == length(zM)) ...
                        && (norm(sys.K(nM,zM) - sys.K(zM,nM)',1) == 0) ...
                        && not(any(any(sys.Bu(zM, :)))) ...
                        && (isempty(sys.Cp) ...
                        || not(any(any(sys.Cp(:, zM))))) ...
                        && (isempty(sys.Cv) ...
                        || not(any(any(sys.Cv(:, zM)))))
                    % TODO: remove norm condition when MESS allows for
                    % different matrices for constraints and lagrange
                    % multipliers.

                    if condest(sys.K(zM, nM) * sys.K(nM, zM)) > (1/eps)
                        warning('MORLAB:data', ...
                            ['Rank deficient index-3 structure' ...
                            ' detected!']);
                    end

                    usfs = 'dae_3_so';
                elseif isempty(nonzeros(sys.K(zM, zM))) ...
                        && (double(sprank(sys.K(nM, zM))) == length(zM)) ...
                        && (double(sprank(sys.E(zM, nM))) == length(zM)) ...
                        && (norm(sys.K(nM,zM) - sys.E(zM,nM)',1) == 0) ...
                        && not(any(any(sys.Bu(zM, :)))) ...
                        && (isempty(sys.Cp) ...
                        || not(any(any(sys.Cp(:, zM))))) ...
                        && (isempty(sys.Cv) ...
                        || not(any(any(sys.Cv(:, zM)))))
                    % TODO: remove norm condition when MESS allows for
                    % different matrices for constraints and lagrange
                    % multipliers.
                    % TODO: remove condition for zero improper part when
                    % MESS allows

                    if condest(sys.E(zM, nM) * sys.K(nM, zM)) > (1/eps)
                        warning('MORLAB:data', ...
                            ['Rank deficient index-2 structure' ...
                            ' detected!']);
                    end

                    usfs = 'dae_2_so';
                else
                    error('MORLAB:data', ...
                        'MORLAB cannot determine system structure!');
                end
            else
                % Nonsymmetric permutation to get potential M11 block.
                cols = find(not(any(sys.M, 1)));
                rows = find(not(any(sys.M, 2)));

                if length(cols) == length(rows)
                    n    = size(sys.M, 1);
                    permR = [setdiff(1:n, cols(:)'), cols(:)'];
                    permL = [setdiff(1:n, rows(:)'), rows(:)'];

                    sys.M          = sys.M(permL, permR);
                    sys.E          = sys.E(permL, permR);
                    sys.K          = sys.K(permL, permR);
                    sys.Bu         = sys.Bu(permL, :);
                    if not(isempty(sys.Cp)), sys.Cp = sys.Cp(:, permR); end
                    if not(isempty(sys.Cv)), sys.Cv = sys.Cv(:, permR); end

                    sys.SystemPermL = permL;
                    sys.SystemPermR = permR;
                    zM              = (n - length(cols) + 1):n;
                    nM              = 1:(zM(1) - 1);
                    sys.zM          = zM(:);
                    sys.nM          = nM(:);

                    if condest(sys.M(nM, nM)) > (1 / eps)
                        warning('MORLAB:data', ...
                            ['Ill-conditioned second-order DAE' ...
                            ' structure detected!']);
                    end

                    if double(sprank(sys.K(zM, zM))) == length(zM) ...
                            && isempty(nonzeros(sys.E(zM, :))) ...
                            && isempty(nonzeros(sys.E(:, zM))) ...
                            && (isempty(nonzeros(sys.Bu(zM, :))) ...
                            || isempty(sys.Cv) ...
                            || not(any(any(sys.Cv(:, zM)))))
                        if condest(sys.K(zM, zM)) > (1 / eps)
                            warning('MORLAB:data', ...
                                ['Ill-conditioned second-order index-1' ...
                                ' structure detected! Perturbation of' ...
                                ' K by small multiples of the identity' ...
                                ' may be necessary.']);
                        end

                        usfs = 'dae_1_so';
                    else
                        error('MORLAB:data', ...
                            'MORLAB cannot determine system structure!');
                    end
                else
                    error('MORLAB:data', ...
                        'MORLAB cannot determine system structure!');
                end
            end
        end

    otherwise % Standard system.
        usfs = 'default';
end

end
