function [Z, Y] = ml_compress_ldl(Z, Y, tol, type)
%ML_COMPRESS_LDL LDL^T row/column compression.
%
% SYNTAX:
%   [Z, Y] = ML_COMPRESS_LDL(Z, Y, tol, type)
%
% DESCRIPTION:
%   Computes a column or row compression of the matrices Z and Y using an
%   eigenvalue decomposition. Usually used to approximate the products
%   Z'*Y*Z (row compression) or Z*Y*Z' (column compression) via low-rank
%   factors.
%
% INPUTS:
%   Z    - matrix of dimensions n x m (column compression) or m x n
%          (row compression)
%   Y    - symmetric matrix of dimensions m x m
%   tol  - nonnegative scalar, tolerance multiplied with the largest
%          singular value to determine the rank of the approximation
%   type - string, switching between row and column compression
%            'c', 'col', 'column' - for column compression
%            'r', 'row'           - for row compression
%
% OUTPUTS:
%   Z    - matrix of dimensions n x r in case of column compression and
%          r x n in case of row compression
%   Y    - matrix of dimensions r x r
%
% See also ml_compress_fac.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(4, 4);

assert(isa(Z, 'double'), ...
    'MORLAB:data', ...
    'The input argument Z has to be a matrix!');

assert(isa(Y, 'double'), ...
    'MORLAB:data', ...
    'The input argument Y has to be a matrix!');

ml_assert_nonnegscalar(tol, 'tol');

assert(ischar(type), ...
    'MORLAB:data', ...
    'The input argument type has to be a string!');


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LDL^T COLUMN/ROW COMPRESSION.                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

switch lower(type)
    case {'c', 'col', 'column'}
        assert(isequal(size(Y), [size(Z, 2), size(Z, 2)]), ...
            'MORLAB:data', ...
            'Wrong dimensions of matrices Z and Y!');

        [Q, R] = qr(Z, 0);
        RYR    = R * (Y * R');
        [V, Y] = eig(RYR);
        r      = abs(diag(Y)) > max(abs(diag(Y))) * tol;
        Y      = Y(r, r);
        Z      = Q * V(:, r);

    case {'r', 'row'}
        assert(isequal(size(Y), [size(Z, 1), size(Z, 1)]), ...
            'MORLAB:data', ...
            'Wrong dimensions of matrices Z and Y!');

        [Q, R] = qr(Z', 0);
        RYR    = R * (Y * R');
        [V, Y] = eig(RYR);
        r      = abs(diag(Y)) > max(abs(diag(Y))) * tol;
        Y      = Y(r, r);
        Z      = (Q * V(:, r))';

    otherwise
        error( ...
            'MORLAB:data', ...
            'This type of compression is not implemented!');
end
