function [Z] = ml_compress_fac(Z, tol, type)
%ML_COMPRESS_FAC SVD-based column/row compression.
%
% SYNTAX:
%   Z = ML_COMPRESS_FAC(Z, tol, type)
%
% DESCRIPTION:
%   Computes a column or row compression of the matrix Z using the SVD.
%   Usually used to approximate the products Z'*Z or Z*Z' via a low-rank
%   factor.
%
% INPUTS:
%   Z    - matrix of dimensions n x m
%   tol  - nonnegative scalar, tolerance multiplied with the largest
%          singular value to determine the rank of the approximation
%   type - string, switching between row and column compression
%            'c', 'col', 'column' - for column compression
%            'r', 'row'           - for row compression
%
% OUTPUTS:
%   Z    - matrix of dimensions n x r in case of column compression and
%          r x m in case of row compression
%
% See also ml_compress_ldl.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(3, 3);

assert(isa(Z, 'double'), ...
    'MORLAB:data', ...
    'The input argument Z has to be a matrix!');

ml_assert_nonnegscalar(tol, 'tol');

assert(ischar(type), ...
    'MORLAB:data', ...
    'The input argument type has to be a string!');


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COLUMN/ROW COMPRESSION.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

switch lower(type)
    case {'c', 'col', 'column'}
        [U, S, ~] = svd(Z, 'econ');
        r         = sum(diag(S) > S(1) * tol);
        Z         = U(:, 1:r) * S(1:r, 1:r);

    case {'r', 'row'}
        [~, S, V] = svd(Z, 'econ');
        r         = sum(diag(S) > S(1) * tol);
        Z         = S(1:r, 1:r) * V(:, 1:r)';

    otherwise
        error( ...
            'MORLAB:data', ...
            'This type of compression is not implemented!');
end
