function [Vinf, Winf, hsvi] = ml_balproj_improper(sys, Rinf, Linf, opts)
%ML_BALPROJ_IMPROPER Compute projection bases for MOR of improper parts.
%
% SYNTAX:
%   [Vinf, Winf, hsvi] = ML_BALPROJ_IMPROPER(sys, Rinf, Linf)
%   [Vinf, Winf, hsvi] = ML_BALPROJ_IMPROPER(sys, Rinf, Linf, opts)
%
% DESCRIPTION:
%   This function computes the projection basis for balanced truncation of
%   improper system parts, which have been separated via an additive
%   decomposition, via the square-root or balancing-free square-root
%   method.
%
% INPUTS:
%   sys  - structure, containing the improper system in the form:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Einf            | matrix with dimensions ninf x ninf                |
%   +-----------------+---------------------------------------------------+
%   | Ainf            | matrix with dimensions ninf x ninf                |
%   +-----------------+---------------------------------------------------+
%   | Binf            | matrix with dimensions ninf x m                   |
%   +-----------------+---------------------------------------------------+
%   | Cinf            | matrix with dimensions p x ninf                   |
%   +-----------------+---------------------------------------------------+
%   Rinf - Cholesky factor of the improper controllability Gramian with
%          dimensions ninf x nr
%   Linf - Cholesky factor of the improper observability Gramian with
%          dimensions ninf x nl
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | ImproperTrunc   | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest proper Hankel singular value of the       |
%   |                 | system to truncate the improper part, if 0 no     |
%   |                 | improper balanced truncation is performed         |
%   |                 | (default log(ninf)*eps)                           |
%   +-----------------+---------------------------------------------------+
%   | Index           | nonnegative integer, index of nilpotency of the   |
%   |                 | matrix Einf used to set an upper bound on the size|
%   |                 | of the reduced improper part, if the index is     |
%   |                 | unknown Inf is set                                |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   Vinf - right projection basis of dimensions ninf x r
%   {!}
%   Winf - left projection basis of dimensions ninf x r
%   {!}
%   hsvi - vector, containing the improper Hankel singular values
%
%
% REFERENCE:
%   V. Mehrmann, T. Stykel, Balanced truncation model reduction for
%   large-scale systems in descriptor form, in: P. Benner, V. Mehrmann,
%   D. C. Sorensen (Eds.), Dimension Reduction of Large-Scale Systems,
%   Vol. 45 of Lect. Notes Comput. Sci. Eng., Springer-Verlag,
%   Berlin/Heidelberg, Germany, 2005, pp. 83--115.
%   https://doi.org/10.1007/3-540-27909-1_3
%
% See also ml_projtrunc_improper, ml_balproj_proper.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(3, 4);

if (nargin < 4) || isempty(opts)
    opts = struct();
end

ml_assert_infdescsys(sys);

if issparse(sys.Ainf), sys.Ainf = full(sys.Ainf); end
if issparse(sys.Einf), sys.Ainf = full(sys.Einf); end
if issparse(sys.Binf), sys.Ainf = full(sys.Binf); end
if issparse(sys.Cinf), sys.Ainf = full(sys.Cinf); end

ninf = size(sys.Ainf, 1);

assert(isa(Rinf, 'double') && (size(Rinf, 1) == ninf), ...
    'MORLAB:data', ...
    'The matrix Rinf must have the same number of rows as sys.Ainf!');

assert(isa(Linf, 'double') && (size(Linf, 1) == ninf), ...
    'MORLAB:data', ...
    'The matrix Linf must have the same number of rows as sys.Ainf!');

if issparse(Rinf), Rinf = full(Rinf); end
if issparse(Linf), Linf = full(Linf); end

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'ImproperTrunc')
    ml_assert_nonnegscalar(opts.ImproperTrunc, 'opts.ImproperTrunc');
else
    opts.ImproperTrunc = log(ninf) * eps;
end

if ml_field_set_to_value(opts, 'Index')
    ml_assert_nonnegintinf(opts.Index, 'opts.Index');
else
    opts.Index = Inf;
end

opts = ml_check_cell_param(opts, 'Method', @ml_assert_char, 'sr');

numProjections = length(opts.Method);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF IMPROPER HANKEL SINGULAR VALUES.                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Special case of empty data.
if ninf == 0
    Vinf = [];
    Winf = [];
    hsvi = [];
    return;
end

% Computation of improper Hankel singular values.
[Uinf, Sinf, Tinf] = svd(Linf' * (sys.Ainf * Rinf), 'econ');
hsvi               = nonzeros(diag(Sinf));
hsvi               = hsvi(1:min([length(hsvi), ninf]));


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF REDUCED ORDER.                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if isempty(hsvi) || (hsvi(1) <= opts.ImproperTrunc)
    ninf = 0;
else
    ninf = sum(hsvi >= opts.ImproperTrunc * hsvi(1));
end

rinf = min([ninf, size(sys.Ainf, 1), size(sys.Binf, 2) * opts.Index, ...
    size(sys.Cinf, 1) * opts.Index]);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF PROJECTION.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Vinf = cell(1, numProjections);
Winf = cell(1, numProjections);

for k = 1:numProjections
    switch lower(opts.Method{k})
        case 'sr' % Square root method.
            Sinf    = sparse(1:rinf, 1:rinf, 1 ./ sqrt(hsvi(1:rinf)));
            Vinf{k} = full((Rinf * Tinf(: , 1:rinf)) * Sinf);
            Winf{k} = full((Linf * Uinf(: , 1:rinf)) * Sinf);

        case 'bfsr' % Balancing-free square root method.
            [Vinf{k}, ~] = qr(Rinf * Tinf(: , 1:rinf), 0);
            [Winf{k}, ~] = qr(Linf * Uinf(: , 1:rinf), 0);

        otherwise
            error('MORLAB:data', ...
                ['The given projection computation method' ...
                ' is not implemented!']);
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FORMAT OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if numProjections == 1
    Vinf = Vinf{:};
    Winf = Winf{:};
end
