function [varargout] = ml_dt_mt(varargin)
%ML_DT_MT Discrete-time modal truncation method.
%
% SYNTAX:
%   [Ar, Br, Cr, Dr, info] = ML_DT_MT(A, B, C, D)
%   [Ar, Br, Cr, Dr, info] = ML_DT_MT(A, B, C, D, opts)
%
%   [Ar, Br, Cr, Dr, Er, info] = ML_DT_MT(A, B, C, D, E)
%   [Ar, Br, Cr, Dr, Er, info] = ML_DT_MT(A, B, C, D, E, opts)
%
%   [rom, info] = ML_DT_MT(sys)
%   [rom, info] = ML_DT_MT(sys, opts)
%
% DESCRIPTION:
%   This function computes the modal truncation for a discrete-time
%   standard system of the form
%
%         x(t+1) = A*x(t) + B*u(t),                                    (1a)
%           y(t) = C*x(t) + D*u(t),                                    (1b)
%
%   descriptor system of the form
%
%       E*x(t+1) = A*x(t) + B*u(t),                                    (2a)
%           y(t) = C*x(t) + D*u(t).                                    (2b)
%
%   The computed reduced-order models have the same format as the systems
%   given as input.
%
% INPUTS:
%   A    - matrix from (1) or (2) with dimensions n x n
%   B    - matrix from (1) or (2) with dimensions n x m
%   C    - matrix from (1) or (2) with dimensions p x n
%   D    - matrix from (1) or (2) with dimensions p x m
%   E    - matrix from (1) or (2) with dimensions n x n
%   sys  - structure or state-space object (ss, dss), containing the
%          system's matrices
%   opts - structure, containing optional paramaters, the most important
%          are shown below, for details on parameters to change underlying
%          computational routines see *_mt functions:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Alpha           | real scalar, such that all finite eigenvalues with|
%   | {!}             | absolute value smaller than Alpha are truncated   |
%   |                 | (default 0.5)                                     |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   Ar   - output matrix for (1) or (2) with dimensions r x r
%   Br   - output matrix for (1) or (2) with dimensions r x m
%   Cr   - output matrix for (1) or (2) with dimensions p x r
%   Dr   - output matrix for (1) or (2) with dimensions p x m
%   Er   - output matrix for (1) or (2) with dimensions r x r
%   rom  - struct or state-space object (ss, dss), containing the
%          reduced-order system's matrices (naming of entries is identical
%          to input argument sys)
%   info - structure, containing the information about the computations and
%          models, depending on the system type more information might be
%          available, see *_mt routines for details:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | SystemType      | character array, system structure that has been   |
%   |                 | determined for computations                       |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_mt, ml_dt_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK AND REFORMAT INPUTS.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[sys, opts, ioformat] = ml_decide_system_type('dt', varargin{:});

if ioformat == 0
    % Check for maximum number of outputs in second-order case.
    nargoutchk(0, 8);
else
    nargoutchk(0, 2);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SELECT BALANCED TRUNCATION ROUTINE.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Start and end strings for system structures.
bstype = 'discrete-time ';
estr   = ' system';

% Selections of routines for different system structures.
switch lower(sys.SystemType)
    case 'dt_d_ss'
        [roms, info]    = ml_dt_d_ss_mt(sys, opts);
        info.SystemType = [bstype 'dense standard state-space' estr];

    case 'dt_d_dss'
        [roms, info]    = ml_dt_d_dss_mt(sys, opts);
        info.SystemType = [bstype 'dense descriptor' estr];

    case 'dt_d_soss'
        error('MORLAB:notImplemented', ...
            ['Modal truncation for discrete-time second-order systems' ...
            '  is not implemented yet.']);

    case {'dt_s_dss_default', 'dt_s_ss_default', 'dt_s_dss_dae_1', ...
            'dt_s_dss_dae_2', 'dt_s_soss_so_1', ...
            'dt_s_soss_dae_1_so', 'dt_s_soss_dae_2_so', ...
            'dt_s_soss_dae_3_so'}
        error('MORLAB:notImplemented', ...
            ['Modal truncation for sparse discret-time systems is not' ...
            ' implemented yet.']);

    otherwise
        error('MORLAB:data', 'Unknow system type!');
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if not(isa(roms, 'cell')), roms = {roms}; end

[varargout{1:nargout}] = ml_format_output(roms, ioformat, info);
