function [rom, info] = ml_ct_s_soss_prbt(sys, opts)
%ML_CT_S_SOSS_PRBT PRBT for sparse second-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_S_SOSS_PRBT(sys)
%   [rom, info] = ML_CT_S_SOSS_PRBT(sys, opts)
%
% DESCRIPTION:
%   This function computes the positive-real balanced truncation for a
%   sparse, second-order system of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   Therefore, the two Riccati equations
%
%       A*P*E' + E*P*A' + B*B' ...
%           + (E*P*C' - B) * inv(D + D') * (E*P*C' - B)' = 0,
%       A'*Q*[E] + [E']*Q*A + C'*C ...
%           + (B'*Q*E - C)' * inv(D + D') * (B'*Q*E - C) = 0,
%
%   are solved for the Gramians P and Q, using a suitable companion form
%   realization for the first-order realization A, B, C, D, E.
%   As a result, a reduced model in either first-order
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   or second-order form
%
%       Mr*x''(t) = -Kr*x(t) -  Er*x'(t) + Bur*u(t)                     (5)
%            y(t) = Cpr*x(t) + Cvr*x'(t) +  Dr*u(t)                     (6)
%
%   is computed.
%
%   Note: In case of a rank-deficient D + D' term, an epsilon
%         regularization is performed, which replaces the D during the
%         computations with an identity matrix scaled by a given epsilon.
%
% INPUTS:
%   sys  - structure, containing the system matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | BalanceType     | character array, determining the type of          |
%   | {!}             | second-order balancing, only if OutputModel = 'so'|
%   |                 |   'p'   - position balancing                      |
%   |                 |   'pm'  - position balancing (diagonalized M)     |
%   |                 |   'pv'  - position-velocity balancing             |
%   |                 |   'vp'  - velocity-position balancing             |
%   |                 |   'vpm' - velocity-position balancing (diag. M)   |
%   |                 |   'v'   - velocity balancing                      |
%   |                 |   'fv'  - free velocity balancing                 |
%   |                 |   'so'  - second-order balancing                  |
%   |                 | (default 'so')                                    |
%   +-----------------+---------------------------------------------------+
%   | carecopts       | structure, containing the optional parameters for |
%   |                 | the filter Riccati equation solver, see           |
%   |                 | mess_lrri                                         |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | careoopts       | structure, containing the optional parameters for |
%   |                 | the regulator Riccati equation solver, see        |
%   |                 | mess_lrri                                         |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Epsilon         | positive scalar, used in the case of a            |
%   |                 | non-full-rank D + D' term for epsilon             |
%   |                 | regularization by multiplying with an identity    |
%   |                 | matrix of appropriate size                        |
%   |                 | (default 1.0e-03)                                 |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the hsv,  |
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 |  'sum'       - using rel. tolerance for sum of hsv|
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | OutputModel     | character array, determining if classical or      |
%   |                 | structure-preserving balanced truncation is used  |
%   |                 |  'fo' - first-order PRBT                          |
%   |                 |  'so' - second-order PRBT                         |
%   |                 | (default 'fo')                                    |
%   +-----------------+---------------------------------------------------+
%   | RiccatiSolver   | character array, determining the solver for the   |
%   |                 | dual Riccati equations                            |
%   |                 |  'ri' - Riccati iteration                         |
%   |                 | (default 'ri')                                    |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | if 'tolerance' or 'sum' is set for                |
%   |                 | OrderComputation                                  |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order model, with the
%   {!}    following entries if opts.OutputModel = 'fo'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%          and the following entries if opts.OutputModel = 'so'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (5) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (6) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed characteristic  |
%   | {!}             | positive-real values, for second-order balancing  |
%   |                 | the position and velocity characteristic          |
%   |                 | positive-real values are saved in a struct        |
%   +-----------------+---------------------------------------------------+
%   | InvAbsErrBound  | computed error bound for the absolute error of the|
%   | {!}             | inverse transfer functions in H-infinity norm, if |
%   |                 | opts.OutputModel = 'fo'                           |
%   +-----------------+---------------------------------------------------+
%   | infoCAREC       | structure, containing information about the       |
%   |                 | Riccati equation solver for the filter            |
%   |                 | (controllability) equation                        |
%   +-----------------+---------------------------------------------------+
%   | infoCAREO       | structure, containing information about the       |
%   |                 | Riccati equation solver for the regulator         |
%   |                 | (observability) equation                          |
%   +-----------------+---------------------------------------------------+
%   | M               | matrix with dimensions m x m, polynomial part of  |
%   |                 | zeroth order and used in the error bound          |
%   |                 | (or term from the epsilon regularization)         |
%   +-----------------+---------------------------------------------------+
%   | N               | Dimension of the reduced-order model              |
%   | {!}             |                                                   |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%
% See also ml_ct_s_foss_prbt, ml_ct_s_soss_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

assert(exist('mess_version', 'file') == 2, ...
    'MORLAB:mmess', ...
    ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
    ' installed!']);

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

switch lower(sys.SystemType)
    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        % No extra action in main supported case.

        % TODO: Remove when M-M.E.S.S. is fixed.
        assert(not(strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so')) ...
            && not(strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')), ...
            'MORLAB:notImplemented', ...
            'This system structure is not yet supported.');

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Prepare system data.
sys = ml_prepare_system_data(sys);

m = size(sys.pB, 2);

assert(size(sys.pC, 1) == m, ...
    'MORLAB:data', ...
    ['The system has to be square.\nThe matrix C must have the ' ...
    'same number of rows as columns of matrix B.']);

% Check and assign optional MORLAB parameters.
if ml_field_set_to_value(opts, 'Epsilon')
    ml_assert_posscalar(opts.Epsilon, 'opts.Epsilon');
else
    opts.Epsilon = 1.0e-03;
end

if ml_field_set_to_value(opts, 'OutputModel')
    assert(strcmpi(opts.OutputModel, 'fo') ...
        || strcmpi(opts.OutputModel, 'so'), ...
        'MORLAB:data', ...
        'The parameter OutputModel must be ''fo'' or ''so''.');
else
    opts.OutputModel = 'fo';
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        if strcmpi(opts.OutputModel, 'fo')
            rselect{k} = 3;
        else
            rselect{k} = 1;
        end
    elseif strcmpi(opts.OrderComputation{k}, 'sum')
        if strcmpi(opts.OutputModel, 'fo')
            rselect{k} = 3;
        else
            rselect{k} = 2;
        end
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'RiccatiSolver')
    assert(strcmpi(opts.RiccatiSolver, 'ri'), ...
        'MORLAB:data', ...
        'The requested Riccati equation solver is not implemented!');
else
    opts.RiccatiSolver = 'ri';
end

if strcmpi(opts.RiccatiSolver, 'ri')
    riccati_solver = @mess_lrri;
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Check and assign optional M-M.E.S.S. parameters.
if ml_field_set_to_value(opts, 'carecopts')
    assert(isa(opts.carecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.carecopts has to be a struct!');
else
    opts.carecopts = struct();
end

opts.carecopts = ml_check_messopts([], opts.carecopts);

if ml_field_set_to_value(opts, 'careoopts')
    assert(isa(opts.careoopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.careoopts has to be a struct!');
else
    opts.careoopts = struct();
end

opts.careoopts = ml_check_messopts([], opts.careoopts);

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Epsilon regularization.
if (rank(sys.pD + sys.pD') < m) || (norm(sys.pD, 'fro') < 1.0e+02 * eps)
    M = opts.Epsilon * eye(m);
    J = 2 * opts.Epsilon * eye(m);
    D = sqrt(2 * opts.Epsilon) * eye(m);
else
    M = sys.pD;
    J = sys.pD + sys.pD';
    D = chol(J);
end

% Create equation object.
switch lower(sys.SystemType)
    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so'}
        eqn  = struct( ...
            'M_'   , sys.M, ...
            'E_'   , sys.E, ...
            'K_'   , sys.K, ...
            'haveE', true);

        if strcmpi(sys.SystemType, 'ct_s_soss_so_1')
            oper = operatormanager(struct(), 'so_1');
        else
            eqn.manifold_dim = sys.nM(end);
            oper             = operatormanager(struct(), 'dae_1_so');
        end

    case {'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        eqn = struct( ...
            'M_'   , sys.pM, ...
            'E_'   , sys.pE, ...
            'K_'   , sys.pK, ...
            'G_'   , sys.pG, ...
            'alpha', 0, ...
            'haveE', true);

        if strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so')
            oper    = operatormanager(struct(), 'dae_2_so');
            wstring = 'index-2';
        else
            oper    = operatormanager(struct(), 'dae_3_so');
            wstring = 'index-3';
        end

        if strcmpi(opts.OutputModel, 'so')
            warning('MORLAB:data', ...
                ['Structure-preserving balancing is not supported for' ...
                ' %s systems.\nChange to Opts.OutputModel = ''fo''.'], ...
                wstring);

            opts.OutputModel = 'fo';
        end

    otherwise
        error('MORLAB:data', ...
            ['The selected system structure is not implemented in this' ...
            ' function!']);
end

eqn.B1     = ml_compress_fac(sys.pB / D, eps, 'column');
eqn.B2     = zeros(size(sys.pB, 1), 0);
eqn.C1     = ml_compress_fac(D' \ sys.pC, eps, 'row');
eqn.C2     = zeros(0, size(sys.pC, 2));
eqn.U      = -sys.pB;
eqn.V      = (J \ sys.pC)';
eqn.haveUV = true;

% Computation of low-rank Gramian factors.
if ml_field_set_to_value(opts, 'GramFacC')
    R              = opts.GramFacC;
    info.infoCAREC = [];
else
    eqn.type       = 'N';
    out            = riccati_solver(eqn, opts.carecopts, oper);
    R              = out.Z;
    info.infoCAREC = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoCAREC.IterationSteps = length(out.res);
        info.infoCAREC.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoCAREC.IterationSteps = length(out.rc);
        info.infoCAREC.Residuals      = out.rc;
    end
end

if ml_field_set_to_value(opts, 'GramFacO')
    L              = opts.GramFacO;
    info.infoCAREO = [];
else
    eqn.type       = 'T';
    out            = riccati_solver(eqn, opts.careoopts, oper);
    L              = out.Z;
    info.infoCAREO = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoCAREO.IterationSteps = length(out.res);
        info.infoCAREO.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoCAREO.IterationSteps = length(out.rc);
        info.infoCAREO.Residuals      = out.rc;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection basis matrices.
if strcmpi(opts.OutputModel, 'fo')
    opts = ml_check_cell_param(opts, 'Tolerance', ...
        @ml_assert_nonnegscalar, 1.0e-02);

    for k = 1:length(opts.Tolerance)
        opts.Tolerance{k} = opts.Tolerance{k} * min(svd(J));
    end

    [V, W, hsv] = ml_balproj_proper(sys, R, L, 0, rselect, opts);
else
    [V, W, hsv] = ml_balproj_soss(sys, R, L, rselect, opts);
end

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
if strcmpi(opts.OutputModel, 'fo')
    rom = ml_projtrunc_proper(sys, V, W);
else
    rom = ml_projtrunc_soss(sys, V, W);
end

if not(isa(rom, 'cell')), rom = {rom}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
numRoms     = length(rom);
n           = cell(1, numRoms);

if strcmpi(opts.OutputModel, 'fo')
    absErrBound = cell(1, numRoms);
    for k = 1:numRoms
        n{k}           = size(rom{k}.A, 1);
        absErrBound{k} = 2 / min(svd(J)) * sum(hsv(n{k}+1:end));
    end
    info.InvAbsErrBound = absErrBound;
else
    for k = 1:numRoms
        n{k} = size(rom{k}.K, 1);
    end
end

info.Hsv = hsv;
info.M   = M;
info.N   = n;

% Store projection bases.
if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
