function [rom, info] = ml_ct_s_soss_krylov(sys, opts)
%ML_CT_S_SOSS_KRYLOV Krylov subspace method for sparse sec.-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_S_SOSS_KRYLOV(sys)
%   [rom, info] = ML_CT_S_SOSS_KRYLOV(sys, opts)
%
% DESCRIPTION:
%   This function computes Krylov subspace approximations for sparse,
%   first-order systems of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   Therefore, shifted linear systems in expansion points are solved. As a
%   result a reduced-order system of the form
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   or second-order form
%
%       Mr*x''(t) = -Kr*x(t) -  Er*x'(t) + Bur*u(t)                     (5)
%            y(t) = Cpr*x(t) + Cvr*x'(t) +  Dr*u(t)                     (6)
%
%   is computed.
%
% INPUTS:
%   sys  - structure, containing the second-order system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | krylovVopts     | struct, optional parameters to determine          |
%   |                 | construction of right Krylov basis, with the      |
%   |                 | optional fields:                                  |
%   |                 |   Directs - matrix, tangential directions, set to |
%   |                 |             ones vector for matrix interpolation, |
%   |                 |             default setting changes to            |
%   |                 |             randn(m, NumPts) if inputs m > 4      |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   FreqRange - vector of length 2, exponents of    |
%   |                 |             frequency range, used as sampling     |
%   |                 |             range if Points are not given         |
%   |                 |             (default [-4, 4])                     |
%   |                 |   NumPts  - positive integer, used as number of   |
%   |                 |             sampling points if Directs, Orders    |
%   |                 |             and Points are not set                |
%   |                 |             (default 50)                          |
%   |                 |   Orders  - vector, derivative orders             |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   Points  - vector, expansion points              |
%   |                 |             (default 1i*logspace(FreqR., NumPts)) |
%   |                 |   RealVal - {0, 1}, turns off/on if basis is      |
%   |                 |             realified by splitting real and       |
%   |                 |             imaginary parts of vectors            |
%   |                 |             (default 1)                           |
%   +-----------------+---------------------------------------------------+
%   | krylovWopts     | struct, optional parameters to determine          |
%   |                 | construction of left Krylov basis, with the       |
%   |                 | optional fields:                                  |
%   |                 |   Directs - matrix, tangential directions, set to |
%   |                 |             ones vector for matrix interpolation, |
%   |                 |             default setting changes to            |
%   |                 |             randn(m, NumPts) if outputs p > 4     |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   FreqRange - vector of length 2, exponents of    |
%   |                 |             frequency range, used as sampling     |
%   |                 |             range if Points are not given         |
%   |                 |             (default [-4, 4])                     |
%   |                 |   NumPts  - positive integer, used as number of   |
%   |                 |             sampling points if Directs, Orders    |
%   |                 |             and Points are not set                |
%   |                 |             (default 50)                          |
%   |                 |   Orders  - vector, derivative orders             |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   Points  - vector, expansion points              |
%   |                 |             (default 1i*logspace(FreqR., NumPts)) |
%   |                 |   RealVal - {0, 1}, turns off/on if basis is      |
%   |                 |             realified by splitting real and       |
%   |                 |             imaginary parts of vectors            |
%   |                 |             (default 1)                           |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(SVs)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the SVs,  |
%   |                 |  'sum'       - using rel. tolerance for sum of SVs|
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | OutputModel     | character array, determining if classical or      |
%   |                 | structure-preserving balanced truncation is used  |
%   |                 |  'fo' - first-order BT                            |
%   |                 |  'so' - second-order BT                           |
%   |                 | (default 'fo')                                    |
%   +-----------------+---------------------------------------------------+
%   | StoreBases      | {0, 1}, used to disable/enable storing of the     |
%   |                 | uncompressed Krylov bases                         |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | if 'tolerance' or 'sum' is set for                |
%   |                 | OrderComputation                                  |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | TwoSidedProj    | {0, 1}, if turned on two-sided projection is used |
%   |                 | rather than one-sided projection                  |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | VBasis          | matrix, precomputed basis matrix right Krylov     |
%   |                 | subspace                                          |
%   +-----------------+---------------------------------------------------+
%   | WBasis          | matrix, precomputed basis matrix left Krylov      |
%   |                 | subspace                                          |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order model, with the
%   {!}    following entries if opts.OutputModel = 'fo'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%          and the following entries if opts.OutputModel = 'so'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (5) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (6) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | N {!}           | Dimension of the reduced-order model              |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | VBasis          | Right Krylov basis of the controllability         |
%   |                 | subspace, if opts.StoreGramians == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | WBasis          | Left Krylov basis of the observability            |
%   |                 | subspace, if opts.StoreGramians == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_s_foss_krylov, ml_ct_d_soss_krylov, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}

        % TODO: Remove when M-M.E.S.S. is fixed.
        assert(not(strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so')) ...
            && not(strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')), ...
            'MORLAB:notImplemented', ...
            'This system structure is not yet supported.');

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys = ml_prepare_system_data(sys);

m = size(sys.pB, 2);
p = size(sys.pC, 1);

assert(exist('mess_version', 'file') == 2, ...
    'MORLAB:mmess', ...
    ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
    ' installed!']);

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'OutputModel')
    assert(strcmpi(opts.OutputModel, 'fo') ...
        || strcmpi(opts.OutputModel, 'so'), ...
        'MORLAB:data', ...
        'The parameter OutputModel must be ''fo'' or ''so''.');
else
    opts.OutputModel = 'fo';
end

if ml_field_set_to_value(opts, 'krylovVopts')
    assert(isa(opts.krylovVopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.krylovVopts has to be a struct!');
else
    opts.krylovVopts = struct();
end

opts.krylovVopts = ml_check_krylovopts(opts.krylovVopts, m, 'ct');

if ml_field_set_to_value(opts, 'krylovWopts')
    assert(isa(opts.krylovWopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.krylovWopts has to be a struct!');
else
    opts.krylovWopts = struct();
end

opts.krylovWopts = ml_check_krylovopts(opts.krylovWopts, p, 'ct');

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 1;
    elseif strcmpi(opts.OrderComputation{k}, 'sum')
        rselect{k} = 2;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreBases')
    ml_assert_boolean(opts.StoreBases, 'opts.StoreBases');
else
    opts.StoreBases = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GENERATE KRYLOV BASES.                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

switch lower(sys.SystemType)
    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so'}
        eqn  = struct( ...
            'M_'   , sys.M, ...
            'E_'   , sys.E, ...
            'K_'   , sys.K, ...
            'haveE', true);

        if strcmpi(sys.SystemType, 'ct_s_soss_so_1')
            oper = operatormanager(struct(), 'so_1');
        else
            eqn.manifold_dim = sys.nM(end);
            oper             = operatormanager(struct(), 'dae_1_so');

            if strcmpi(opts.OutputModel, 'so')
                warning('MORLAB:data', ...
                    ['Structure-preserving Krylov methods are' ...
                    ' currently not supported for index-1 systems.\n' ...
                    'Change to Opts.OutputModel = ''fo''.']);

                opts.OutputModel = 'fo';
            end
        end

    case {'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        eqn = struct( ...
            'M_'   , sys.pM, ...
            'E_'   , sys.pE, ...
            'K_'   , sys.pK, ...
            'G_'   , sys.pG, ...
            'alpha', 0, ...
            'haveE', true);

        if strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so')
            oper    = operatormanager(struct(), 'dae_2_so');
            wstring = 'index-2';
        else
            oper    = operatormanager(struct(), 'dae_3_so');
            wstring = 'index-3';
        end

        if strcmpi(opts.OutputModel, 'so')
            warning('MORLAB:data', ...
                ['Structure-preserving Krylov methods are currently' ...
                ' not supported for %s systems.\n' ...
                'Change to Opts.OutputModel = ''fo''.'], ...
                wstring);

            opts.OutputModel = 'fo';
        end

    otherwise
        error('MORLAB:data', ...
            ['The selected system structure is not implemented in this' ...
            ' function!']);
end

[eqn, ~, oper] = oper.size_pre(eqn, struct(), oper);
[eqn, ~, oper] = oper.sol_ApE_pre(eqn, struct(), oper);
[eqn, ~, oper] = oper.mul_E_pre(eqn, struct(), oper);

if strcmpi(opts.OutputModel, 'fo')
    n = oper.size(eqn, struct(), oper);
else
    n  = size(sys.K, 1);

    Cp = sys.pC(:, 1:n);
    Cv = sys.pC(:, n+1:end);
end

if n > 0
    % Right basis.
    if ml_field_set_to_value(opts, 'VBasis')
        Vbase = opts.VBasis;
    else
        tang = not(isempty(opts.krylovVopts.Directs));

        if tang
            upt = 1;
        else
            upt = m;
        end

        Vbase = zeros(n, sum(opts.krylovVopts.Orders) * upt);
        idx   = 1:upt;

        for k = 1:length(opts.krylovVopts.Points)
            % Current interpolation point.
            s = opts.krylovVopts.Points(k);

            % Solve linear systems.
            if strcmpi(opts.OutputModel, 'fo')
                if tang
                    Vbase(:, idx) = -oper.sol_ApE(eqn, struct(), ...
                        'N', -s, 'N', ...
                        sys.pB * opts.krylovVopts.Directs(:, k), 'N');
                else
                    Vbase(:, idx) = -oper.sol_ApE(eqn, opts, ...
                        'N', -s, 'N', sys.pB, 'N');
                end

                for j = 2:opts.krylovVopts.Orders(k)
                    idx           = idx + upt;
                    rhs           = oper.mul_E(eqn, opts, ...
                        'N', Vbase(:, idx - upt), 'N');
                    Vbase(:, idx) = -oper.sol_ApE(eqn, opts, ...
                        'N', -s, 'N', rhs, 'N');
                end
            else
                if tang
                    Vbase(:, idx) = (s^2 * sys.M + s * sys.E + sys.K) ...
                        \ (sys.Bu * opts.krylovVopts.Directs(:, k));
                else
                    Vbase(:, idx) = (s^2 * sys.M + s * sys.E + sys.K) ...
                        \ sys.Bu;
                end

                for j = 2:opts.krylovVopts.Orders(k)
                    idx = idx + upt;
                    rhs = ((2 * s) * sys.M + sys.E) * Vbase(:, idx - upt);
                    if j >= 3
                        rhs = 2 * (M * rhs);
                    end
                    Vbase(:, idx) = (s^2 * sys.M + s * sys.E + sys.K) ...
                        \ rhs;
                end
            end

            % Update index range.
            idx = idx + upt;
        end
    end

    % Realification of basis.
    if opts.krylovVopts.RealVal
        R = [real(Vbase), imag(Vbase)];
    else
        R = Vbase;
    end

    % Left basis.
    if ml_field_set_to_value(opts, 'WBasis')
        Wbase = opts.WBasis;
    else
        tang = not(isempty(opts.krylovWopts.Directs));

        if tang
            upt = 1;
        else
            upt = p;
        end

        Wbase = zeros(n, sum(opts.krylovWopts.Orders) * upt);
        idx   = 1:upt;

        for k = 1:length(opts.krylovWopts.Points)
            % Current interpolation point.
            s = opts.krylovWopts.Points(k);

            % Solve linear systems.
            if strcmpi(opts.OutputModel, 'fo')
                if tang
                    Wbase(:, idx) = -oper.sol_ApE(eqn, struct(), ...
                        'T', -conj(s), 'T', ...
                        sys.pC' * opts.krylovWopts.Directs(:, k), 'N');
                else
                    Wbase(:, idx) = -oper.sol_ApE(eqn, opts, ...
                        'T', -conj(s), 'T', sys.pC', 'N');
                end

                for j = 2:opts.krylovVopts.Orders(k)
                    idx           = idx + upt;
                    rhs           = oper.mul_E(eqn, opts, ...
                        'T', Wbase(:, idx - upt), 'N');
                    Wbase(:, idx) = -oper.sol_ApE(eqn, opts, ...
                        'T', -conj(s), 'T', rhs, 'N');
                end
            else
                if tang
                    Wbase(:, idx) = (s^2 * sys.M + s * sys.E + sys.K)' ...
                        \ ((Cp + s * Cv)' ...
                        * opts.krylovWopts.Directs(:, k));
                else
                    Wbase(:, idx) = (s^2 * sys.M + s * sys.E + sys.K)' ...
                        \ (Cp + s * Cv)';
                end

                for j = 2:opts.krylovWopts.Orders(k)
                    idx = idx + upt;
                    rhs = ((2 * s) * sys.M + sys.E)' * Wbase(:, idx - upt);
                    if j >= 3
                        rhs = 2 * (sys.M' * rhs);
                    end
                    Wbase(:, idx) = (s^2 * sys.M + s * sys.E + sys.K)' ...
                        \ rhs;
                end
            end

            % Update index range.
            idx = idx + upt;
        end
    end

    % Realification of basis.
    if opts.krylovWopts.RealVal
        L = [real(Wbase), imag(Wbase)];
    else
        L = Wbase;
    end
else
    [R, L, Vbase, Wbase] = deal([]);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% Compute singular values of projection bases.
if strcmpi(opts.OutputModel, 'fo')
    [V, W] = ml_domsubproj_proper(sys, R, L, 0, rselect, opts);
else
    [V, W] = ml_domsubproj_soss(sys, R, L, rselect, opts);
end

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce the system.
if strcmpi(opts.OutputModel, 'fo')
    rom = ml_projtrunc_proper(sys, V, W);
else
    rom = ml_projtrunc_soss(sys, V, W);
end

if not(isa(rom, 'cell')), rom = {rom}; end

numRoms = length(rom);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Information about the method.
n = cell(1, numRoms);
if strcmpi(opts.OutputModel, 'fo')
    for k = 1:numRoms
        n{k} = size(rom{k}.A, 1);
    end
else
    for k = 1:numRoms
        n{k} = size(rom{k}.K, 1);
    end
end
info.N = n;

% Store final projection bases.
if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreBases
    info.VBasis = Vbase;
    info.WBasis = Wbase;
else
    info.VBasis = [];
    info.WBasis = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
