function [rom, info] = ml_ct_s_soss_bt(sys, opts)
%ML_CT_S_SOSS_BT Balanced truncation for sparse second-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_S_SOSS_BT(sys)
%   [rom, info] = ML_CT_S_SOSS_BT(sys, opts)
%
% DESCRIPTION:
%   This function computes the balanced truncation for sparse, second-order
%   systems of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   Therefore, the two generalized Lyapunov equations
%
%       A*P*E' + E*P*A' + B*B' = 0,
%       A'*Q*E + E'*Q*A + C'*C = 0
%
%   are solved for the system Gramians P and Q, using a suitable companion
%   form realization for the first-order realization A, B, C, D, E.
%   As a result, a reduced model in either first-order
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   or second-order form
%
%       Mr*x''(t) = -Kr*x(t) -  Er*x'(t) + Bur*u(t)                     (5)
%            y(t) = Cpr*x(t) + Cvr*x'(t) +  Dr*u(t)                     (6)
%
%   is computed.
%
%   Note: The system is assumed to be asymptotically stable.
%
% INPUTS:
%   sys  - structure, containing the second-order system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | BalanceType     | character array, determining the type of          |
%   | {!}             | second-order balancing, only if OutputModel = 'so'|
%   |                 |   'p'   - position balancing                      |
%   |                 |   'pm'  - position balancing (diagonalized M)     |
%   |                 |   'pv'  - position-velocity balancing             |
%   |                 |   'vp'  - velocity-position balancing             |
%   |                 |   'vpm' - velocity-position balancing (diag. M)   |
%   |                 |   'v'   - velocity balancing                      |
%   |                 |   'fv'  - free velocity balancing                 |
%   |                 |   'so'  - second-order balancing                  |
%   |                 | (default 'so')                                    |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | lyapcopts       | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time            |
%   |                 | controllability Lyapunov equation,                |
%   |                 | see mess_lradi                                    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | lyapoopts       | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time            |
%   |                 | observability Lyapunov equation,                  |
%   |                 | see mess_lradi                                    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the hsv,  |
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 |  'sum'       - using rel. tolerance for sum of hsv|
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | OutputModel     | character array, determining if classical or      |
%   |                 | structure-preserving balanced truncation is used  |
%   |                 |  'fo' - first-order BT                            |
%   |                 |  'so' - second-order BT                           |
%   |                 | (default 'fo')                                    |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | if 'tolerance' or 'sum' is set for                |
%   |                 | OrderComputation                                  |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order model, with the
%   {!}    following entries if opts.OutputModel = 'fo'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%          and the following entries if opts.OutputModel = 'so'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (5) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (6) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | reduced-order model in H-infinity norm, if        |
%   |                 | opts.OutputModel = 'fo'                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed Hankel singular |
%   | {!}             | values, for second-order balancing the position   |
%   |                 | and velocity Hankel singular values are saved in  |
%   |                 | a struct                                          |
%   +-----------------+---------------------------------------------------+
%   | infoLYAPC       | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation solver for the  |
%   |                 | controllability equation, see mess_lradi          |
%   +-----------------+---------------------------------------------------+
%   | infoLYAPO       | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation solver for the  |
%   |                 | observability equation, see mess_lradi            |
%   +-----------------+---------------------------------------------------+
%   | N {!}           | Dimension of the reduced-order model              |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   T. Reis, T. Stykel, Balanced truncation model reduction of second-order
%   systems, Math. Comput. Model. Dyn. Syst. 14 (5) (2008) 391--406.
%   https://doi.org/10.1080/13873950701844170
%
% See also ml_ct_s_foss_bt, ml_ct_d_soss_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

assert(exist('mess_version', 'file') == 2, ...
    'MORLAB:mmess', ...
    ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
    ' installed!']);

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

switch lower(sys.SystemType)
    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        % No extra action in main supported case.

        % TODO: Remove when M-M.E.S.S. is fixed.
        assert(not(strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so')) ...
            && not(strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')), ...
            'MORLAB:notImplemented', ...
            'This system structure is not yet supported.');

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Prepare system data.
sys = ml_prepare_system_data(sys);

% Check and assign optional MORLAB parameters.
if ml_field_set_to_value(opts, 'OutputModel')
    assert(strcmpi(opts.OutputModel, 'fo') ...
        || strcmpi(opts.OutputModel, 'so'), ...
        'MORLAB:data', ...
        'The parameter OutputModel must be ''fo'' or ''so''.');
else
    opts.OutputModel = 'fo';
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        if strcmpi(opts.OutputModel, 'fo')
            rselect{k} = 3;
        else
            rselect{k} = 1;
        end
    elseif strcmpi(opts.OrderComputation{k}, 'sum')
        if strcmpi(opts.OutputModel, 'fo')
            rselect{k} = 3;
        else
            rselect{k} = 2;
        end
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Check and assign optional M-M.E.S.S. parameters.
if ml_field_set_to_value(opts, 'lyapcopts')
    assert(isa(opts.lyapcopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapcopts has to be a struct!');
else
    opts.lyapcopts = struct();
end

opts.lyapcopts = ml_check_messopts({'global', 'shifts', 'adi'}, ...
    opts.lyapcopts);

if ml_field_set_to_value(opts, 'lyapoopts')
    assert(isa(opts.lyapoopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapoopts has to be a struct!');
else
    opts.lyapoopts = struct();
end

opts.lyapoopts = ml_check_messopts({'global', 'shifts', 'adi'}, ...
    opts.lyapoopts);

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Create equation object.
switch lower(sys.SystemType)
    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so'}
        eqn  = struct( ...
            'M_'   , sys.M, ...
            'E_'   , sys.E, ...
            'K_'   , sys.K, ...
            'haveE', true);

        if strcmpi(sys.SystemType, 'ct_s_soss_so_1')
            oper = operatormanager(struct(), 'so_1');
        else
            eqn.manifold_dim = sys.nM(end);
            oper             = operatormanager(struct(), 'dae_1_so');
        end

    case {'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        eqn = struct( ...
            'M_'   , sys.pM, ...
            'E_'   , sys.pE, ...
            'K_'   , sys.pK, ...
            'G_'   , sys.pG, ...
            'alpha', 0, ...
            'haveE', true);

        if strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so')
            oper    = operatormanager(struct(), 'dae_2_so');
            wstring = 'index-2';
        else
            oper    = operatormanager(struct(), 'dae_3_so');
            wstring = 'index-3';
        end

        if strcmpi(opts.OutputModel, 'so')
            warning('MORLAB:data', ...
                ['Structure-preserving balancing is not supported for' ...
                ' %s systems.\nChange to Opts.OutputModel = ''fo''.'], ...
                wstring);

            opts.OutputModel = 'fo';
        end

    otherwise
        error('MORLAB:data', ...
            ['The selected system structure is not implemented in this' ...
            ' function!']);
end

eqn.B = ml_compress_fac(sys.pB, eps, 'column');
eqn.C = ml_compress_fac(sys.pC, eps, 'row');

% Computation of low-rank Gramian factors.
if ml_field_set_to_value(opts, 'GramFacC')
    R              = opts.GramFacC;
    info.infoLYAPC = [];
else
    eqn.type       = 'N';
    out            = mess_lradi(eqn, opts.lyapcopts, oper);
    R              = out.Z;
    info.infoLYAPC = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoLYAPC.IterationSteps = length(out.res);
        info.infoLYAPC.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoLYAPC.IterationSteps = length(out.rc);
        info.infoLYAPC.Residuals      = out.rc;
    end
end

if ml_field_set_to_value(opts, 'GramFacO')
    L              = opts.GramFacO;
    info.infoLYAPO = [];
else
    eqn.type       = 'T';
    out            = mess_lradi(eqn, opts.lyapoopts, oper);
    L              = out.Z;
    info.infoLYAPO = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoLYAPO.IterationSteps = length(out.res);
        info.infoLYAPO.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoLYAPO.IterationSteps = length(out.rc);
        info.infoLYAPO.Residuals      = out.rc;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection basis matrices.
if strcmpi(opts.OutputModel, 'fo')
    [V, W, hsv] = ml_balproj_proper(sys, R, L, 0, rselect, opts);
else
    [V, W, hsv] = ml_balproj_soss(sys, R, L, rselect, opts);
end

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
if strcmpi(opts.OutputModel, 'fo')
    rom = ml_projtrunc_proper(sys, V, W);
else
    rom = ml_projtrunc_soss(sys, V, W);
end

if not(isa(rom, 'cell')), rom = {rom}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
numRoms = length(rom);
n       = cell(1, numRoms);
if strcmpi(opts.OutputModel, 'fo')
    absErrBound = cell(1, numRoms);
    for k = 1:numRoms
        n{k}           = size(rom{k}.A, 1);
        absErrBound{k} = 2 * sum(hsv(n{k}+1:end));
    end
    info.AbsErrBound = absErrBound;
else
    for k = 1:numRoms
        n{k} = size(rom{k}.K, 1);
    end
end

info.Hsv = hsv;
info.N   = n;

% Store projection bases.
if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
