function [rom, info] = ml_ct_s_foss_brbt(sys, opts)
%ML_CT_S_FOSS_BRBT BRBT for sparse first-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_S_FOSS_BRBT(sys)
%   [rom, info] = ML_CT_S_FOSS_BRBT(sys, opts)
%
% DESCRIPTION:
%   This function computes the bounded-real balanced truncation for a
%   sparse, standard or descriptor system of the form
%
%    [E]x'(t) = A*x(t) + B*u(t),                                        (1)
%        y(t) = C*x(t) + D*u(t).                                        (2)
%
%   Therefore, the two standard Lyapunov equations
%
%       A*P*[E'] + [E]*P*A' + B*B' ...
%           + ([E]*P*C' + B*D') * inv(Rb) * ([E]*P*C' + B*D')' = 0,
%       A'*Q*[E] + [E']*Q*A + C'*C ...
%           + (B'*Q*[E] + D'*C)' * inv(Rc) * (B'*Q*[E] + D'*C) = 0,
%
%   are solved for the Gramians P and Q, with
%
%       Rb = I - D*D',
%       Rc = I - D'*D.
%
%   As result, a reduced-order system of the form
%
%     [Er]*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%        y(t) = Cr*x(t) + Dr*u(t)                                       (4)
%
%   is computed, such that for the original transfer function G with the
%   spectral factors W'*W = I - G'*G and  V*V' = I - G*G', and the r-th
%   order transfer function Gr with the spectral factors
%   Wr'*Wr = I - Gr'*Gr and  Vr*Vr' = I - Gr*Gr' it holds
%
%       max(||[G - Gr; W - Wr]||_{\infty}, ||[G - Gr; V - Vr]||_{\infty})
%       <= 2 * (Hsv(r+1) + ... + Hsv(n)),
%
%   with Hsv, a vector containing the bounded-real singular values of the
%   system.
%
% INPUTS:
%   sys  - structure, containing the system matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n, optional   |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | carecopts       | structure, containing the optional parameters for |
%   |                 | the filter Riccati equation solver, see           |
%   |                 | mess_lrri                                         |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | careoopts       | structure, containing the optional parameters for |
%   |                 | the regulator Riccati equation solver, see        |
%   |                 | mess_lrri                                         |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using absolute error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | RiccatiSolver   | character array, determining the solver for the   |
%   |                 | dual Riccati equations                            |
%   |                 |  'ri' - Riccati iteration                         |
%   |                 | (default 'ri')                                    |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions n x n, optional   |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | reduced-order model (and the spectral factors of  |
%   |                 | I - G*G' and I - G'*G) in H-infinity norm         |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed bounded-real    |
%   |                 | singular values                                   |
%   +-----------------+---------------------------------------------------+
%   | infoCAREC       | structure, containing information about the       |
%   |                 | Riccati equation solver for the filter            |
%   |                 | (controllability) equation                        |
%   +-----------------+---------------------------------------------------+
%   | infoCAREO       | structure, containing information about the       |
%   |                 | Riccati equation solver for the regulator         |
%   |                 | (observability) equation                          |
%   +-----------------+---------------------------------------------------+
%   | N               | Dimension of the reduced-order model              |
%   | {!}             |                                                   |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%
% See also ml_ct_d_ss_brbt, ml_ct_d_dss_brbt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

assert(exist('mess_version', 'file') == 2, ...
    'MORLAB:mmess', ...
    ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
    ' installed!']);

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

switch lower(sys.SystemType)
    case {'ct_s_ss_default', 'ct_s_dss_default', 'ct_s_dss_dae_1', ...
            'ct_s_dss_dae_2'}
        % No extra action in main supported case.

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Prepare system data.
sys = ml_prepare_system_data(sys);

% Check and assign optional MORLAB parameters.
opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 3;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'RiccatiSolver')
    assert(strcmpi(opts.RiccatiSolver, 'ri'), ...
        'MORLAB:data', ...
        'The requested Riccati equation solver is not implemented!');
else
    opts.RiccatiSolver = 'ri';
end

if strcmpi(opts.RiccatiSolver, 'ri')
    riccati_solver = @mess_lrri;
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Check and assign optional M-M.E.S.S. parameters.
if ml_field_set_to_value(opts, 'carecopts')
    assert(isa(opts.carecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.carecopts has to be a struct!');
else
    opts.carecopts = struct();
end

opts.carecopts = ml_check_messopts([], opts.carecopts);

if ml_field_set_to_value(opts, 'careoopts')
    assert(isa(opts.careoopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.careoopts has to be a struct!');
else
    opts.careoopts = struct();
end

opts.careoopts = ml_check_messopts([], opts.careoopts);

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Create equation object.
if isempty(sys.pD)
    eqn = struct( ...
        'A_', sys.A, ...
        'B1', ml_compress_fac(sys.pB, eps, 'column'), ...
        'B2', zeros(size(sys.pB, 1), 0), ...
        'C1', ml_compress_fac(sys.pC, eps, 'row'), ...
        'C2', zeros(0, size(sys.pC, 2)));
else
    Rb = eye(size(sys.pC, 1)) - sys.pD * sys.pD';
    Lb = chol(Rb);
    Rc = eye(size(sys.pB, 2)) - sys.pD' * sys.pD;
    Lc = chol(Rc);

    eqn = struct( ...
        'A_'    , sys.A, ...
        'B1'    , ml_compress_fac(sys.pB / Lc, eps, 'column'), ...
        'B2'    , zeros(size(sys.pB, 1), 0), ...
        'C1'    , ml_compress_fac(Lb' \ sys.pC, eps, 'row'), ...
        'C2'    , zeros(0, size(sys.pC, 2)), ...
        'U'     , sys.pB, ...
        'V'     , sys.pC' * (Rb \ sys.pD), ...
        'haveUV', true);
end

% Select user-supplied function handles and mass matrices.
switch lower(sys.SystemType)
    case 'ct_s_ss_default'
        eqn.haveE = false;
        oper      = operatormanager(struct(), 'default');

    case 'ct_s_dss_default'
        if norm(sys.E - speye(size(sys.E, 1)), 'inf') == 0
            eqn.haveE = false;
        else
            eqn.E_    = sys.E;
            eqn.haveE = true;
        end
        oper = operatormanager(struct(), 'default');

    case {'ct_s_dss_dae_1', 'ct_s_dss_dae_2'}
        eqn.E_           = sys.E;
        eqn.haveE        = true;
        eqn.manifold_dim = sys.nE(end);

        if strcmpi(sys.SystemType, 'ct_s_dss_dae_1')
            oper = operatormanager(struct(), 'dae_1');
        else
            oper = operatormanager(struct(), 'dae_2');
        end

    otherwise
        error('MORLAB:data', ...
            ['The selected system structure is not implemented in this' ...
            ' function!']);
end

% Computation of low-rank Gramian factors.
if ml_field_set_to_value(opts, 'GramFacC')
    R              = opts.GramFacC;
    info.infoCAREC = [];
else
    eqn.type       = 'N';
    out            = riccati_solver(eqn, opts.carecopts, oper);
    R              = out.Z;
    info.infoCAREC = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoCAREC.IterationSteps = length(out.res);
        info.infoCAREC.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoCAREC.IterationSteps = length(out.rc);
        info.infoCAREC.Residuals      = out.rc;
    end
end

if ml_field_set_to_value(opts, 'GramFacO')
    L              = opts.GramFacO;
    info.infoCAREO = [];
else
    eqn.type       = 'T';
    out            = riccati_solver(eqn, opts.careoopts, oper);
    L              = out.Z;
    info.infoCAREO = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoCAREO.IterationSteps = length(out.res);
        info.infoCAREO.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoCAREO.IterationSteps = length(out.rc);
        info.infoCAREO.Residuals      = out.rc;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection basis matrices.
[V, W, hsv] = ml_balproj_proper(sys, R, L, 0, rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
rom = ml_projtrunc_proper(sys, V, W);

if not(isa(rom, 'cell')), rom = {rom}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
numRoms     = length(rom);
n           = cell(1, numRoms);
absErrBound = cell(1, numRoms);
for k = 1:numRoms
    n{k}           = size(rom{k}.A, 1);
    absErrBound{k} = 2 * sum(hsv(n{k}+1:end));
end

info.AbsErrBound = absErrBound;
info.Hsv         = hsv;
info.N           = n;

% Store projection bases.
if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
