function [varargout] = ml_ct_bst(varargin)
%ML_CT_BST Continuous-time balanced stochastic truncation method.
%
% SYNTAX:
%   [Ar, Br, Cr, Dr, info] = ML_CT_BST(A, B, C, D)
%   [Ar, Br, Cr, Dr, info] = ML_CT_BST(A, B, C, D, opts)
%
%   [Ar, Br, Cr, Dr, Er, info] = ML_CT_BST(A, B, C, D, E)
%   [Ar, Br, Cr, Dr, Er, info] = ML_CT_BST(A, B, C, D, E, opts)
%
%   [Ar, Br, Cr, Dr, Er, info] = ML_CT_BST(M, E, K, Bu, Cp, Cv, D)
%   [Ar, Br, Cr, Dr, Er, info] = ML_CT_BST(M, E, K, Bu, Cp, Cv, D, opts)
%   [Mr, Er, Kr, Bur, Cpr, Cvr, Dr, info] = ...
%                                         ML_CT_BST(M, E, K, Bu, Cp, Cv, D)
%   [Mr, Er, Kr, Bur, Cpr, Cvr, Dr, info] = ...
%                                   ML_CT_BST(M, E, K, Bu, Cp, Cv, D, opts)
%
%   [rom, info] = ML_CT_BST(sys)
%   [rom, info] = ML_CT_BST(sys, opts)
%
% DESCRIPTION:
%   This function computes the balanced truncation for a continuous-time
%   standard system of the form
%
%         x'(t) = A*x(t) + B*u(t),                                     (1a)
%          y(t) = C*x(t) + D*u(t),                                     (1b)
%
%   descriptor system of the form
%
%       E*x'(t) = A*x(t) + B*u(t),                                     (2a)
%          y(t) = C*x(t) + D*u(t),                                     (2b)
%
%   or second-order system of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t),                       (3a)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t).                       (3b)
%
%   The computed reduced-order models usually have the format of (1)
%   or (2), but for (3) there is the option to generate
%   structure-preserving reduced-order models.
%
%   Note: Note that unstable systems are only supported in the case of
%   dense first-order systems.
%
% INPUTS:
%   A    - matrix from (1) or (2) with dimensions n x n
%   B    - matrix from (1) or (2) with dimensions n x m
%   C    - matrix from (1) or (2) with dimensions p x n
%   D    - matrix from (1), (2) or (3) with dimensions p x m
%   E    - matrix from (1), (2) or (3) with dimensions n x n
%   M    - matrix from (3) with dimensions n x n
%   K    - matrix from (3) with dimensions n x n
%   Bu   - matrix from (3) with dimensions n x m
%   Cp   - matrix from (3) with dimensions p x n
%   Cv   - matrix from (3) with dimensions p x n
%   sys  - structure or state-space object (ss, dss, sparss, mechss),
%          containing the system's matrices
%   opts - structure, containing optional paramaters, the most important
%          are shown below, for details on parameters to change underlying
%          computational routines see *_bst functions:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Epsilon         | positive scalar, used in the case of a            |
%   |                 | non-full-rank D term for epsilon regularization   |
%   |                 | by multiplying with an identity matrix of         |
%   |                 | appropriate size                                  |
%   |                 | (default 1.0e-03)                                 |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using absolute error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | OutputModel     | character array, determining if classical or      |
%   |                 | structure-preserving balanced truncation is used  |
%   |                 | for systems of type (3)                           |
%   |                 |  'fo' - first-order BST                           |
%   |                 |  'so' - second-order BST                          |
%   |                 | (default 'fo')                                    |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   Ar   - output matrix for (1) or (2) with dimensions r x r
%   Br   - output matrix for (1) or (2) with dimensions r x m
%   Cr   - output matrix for (1) or (2) with dimensions p x r
%   Dr   - output matrix for (1), (2) or (3) with dimensions p x m
%   Er   - output matrix for (1), (2) or (3) with dimensions r x r
%   Mr   - output matrix for (3) with dimensions r x r
%   Kr   - output matrix for (3) with dimensions r x r
%   Bur  - output matrix for (3) with dimensions r x m
%   Cpr  - output matrix for (3) with dimensions p x r
%   Cvr  - output matrix for (3) with dimensions p x r
%   rom  - struct or state-space object (ss, dss, mechss), containing the
%          reduced-order system's matrices (naming of entries is identical
%          to input argument sys)
%   info - structure, containing the information about the computations and
%          models, depending on the system type more information might be
%          available, see *_bst routines for details:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | reduced-order model in H-infinity norm, if        |
%   |                 | output system is of type (1) or (2)               |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | RelErrBound     | computed error bound for the relative error of the|
%   | {!}             | of the reduced-order model in H-infinity norm, if |
%   |                 | output system is of type (1) or (2)               |
%   +-----------------+---------------------------------------------------+
%   | SystemType      | character array, system structure that has been   |
%   |                 | determined for computations                       |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_bt, ml_ct_lqgbt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK AND REFORMAT INPUTS.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[sys, opts, ioformat] = ml_decide_system_type('ct', varargin{:});

if ioformat == 0
    % Check for maximum number of outputs in second-order case.
    nargoutchk(0, 8);
else
    % Struct or state-space object outputs.
    nargoutchk(0, 2);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SELECT BALANCED TRUNCATION ROUTINE.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Start and end strings for system structures.
bstype = 'continuous-time ';
estr   = ' system';

% Selections of routines for different system structures.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        [roms, info]    = ml_ct_d_ss_bst(sys, opts);
        info.SystemType = [bstype 'dense standard state-space' estr];

    case 'ct_d_dss'
        [roms, info]    = ml_ct_d_dss_bst(sys, opts);
        info.SystemType = [bstype 'dense descriptor' estr];

    case 'ct_d_soss'
        [roms, info]    = ml_ct_d_soss_bst(sys, opts);
        info.SystemType = [bstype 'dense second-order' estr];

    case 'ct_s_ss_default'
        [roms, info]    = ml_ct_s_foss_bst(sys, opts);
        info.SystemType = [bstype 'sparse standard state-space' estr];

    case 'ct_s_dss_default'
        [roms, info]    = ml_ct_s_foss_bst(sys, opts);
        info.SystemType = [bstype 'sparse generalized state-space' estr];

    case 'ct_s_dss_dae_1'
        [roms, info]    = ml_ct_s_foss_bst(sys, opts);
        info.SystemType = [bstype 'sparse index-1 descriptor' estr];

    case 'ct_s_dss_dae_2'
        [roms, info]    = ml_ct_s_foss_bst(sys, opts);
        info.SystemType = [bstype 'sparse index-2 descriptor' estr];

    case 'ct_s_soss_so_1'
        [roms, info]    = ml_ct_s_soss_bst(sys, opts);
        info.SystemType = [bstype 'sparse second-order' estr];

    case 'ct_s_soss_dae_1_so'
        [roms, info]    = ml_ct_s_soss_bst(sys, opts);
        info.SystemType = [bstype 'sparse index-1 second-order' estr];

    case 'ct_s_soss_dae_2_so'
        % TODO: Remove when M-M.E.S.S. is fixed.
        error('MORLAB:notImplemented', ...
            ['This function is not implemented yet for sparse ' ...
            ' index-2 second-order systems.']);
        % [roms, info]    = ml_ct_s_soss_bst(sys, opts);
        % info.SystemType = [bstype 'sparse index-2 second-order' estr];

    case 'ct_s_soss_dae_3_so'
        % TODO: Remove when M-M.E.S.S. is fixed.
        error('MORLAB:notImplemented', ...
            ['This function is not implemented yet for sparse ' ...
            ' index-3 second-order systems.']);
        % [roms, info]    = ml_ct_s_soss_bst(sys, opts);
        % info.SystemType = [bstype 'sparse index-3 second-order' estr];

    otherwise
        error('MORLAB:data', 'Unknow system type!');
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if not(isa(roms, 'cell')), roms = {roms}; end

[varargout{1:nargout}] = ml_format_output(roms, ioformat, info);
