function [rom, info] = ml_ct_d_ss_lqgbt(sys, opts)
%ML_CT_D_SS_LQGBT LQG balanced truncation for dense, standard systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SS_LQGBT(sys)
%   [rom, info] = ML_CT_D_SS_LQGBT(sys, opts)
%
% DESCRIPTION:
%   This function computes the linear-quadratic Gaussian balanced
%   truncation for a standard system of the form
%
%       x'(t) = A*x(t) + B*u(t),                                        (1)
%        y(t) = C*x(t) + D*u(t).                                        (2)
%
%   Therefor, the two algebraic Riccati equations
%
%       A*P  + P*A' + B*B' - (P*C' + B*D') * inv(Rb) * (P*C' + B*D')' = 0,
%       A'*Q + Q*A  + C'*C - (B'*Q + D'*C)' * inv(Rc) * (B'*Q + D'*C) = 0,
%
%   are solved for the Gramians P and Q, with
%
%       Rb = I + D*D',
%       Rc = I + D'*D.
%
%   As result, a reduced-order system of the form
%
%       x'(t) = Ar*x(t) + Br*u(t),                                      (3)
%        y(t) = Cr*x(t) + Dr*u(t)                                       (4)
%
%   is computed, such that for the original transfer function G with the
%   (right) coprime factorization G = N*inv(M) and the r-th order transfer
%   function Gr with the (right) coprime factorization Gr = Nr*inv(Mr) it
%   holds
%
%       ||[N; M] - [Nr; Mr]||_{\infty} <= 2 * (Hsv(r+1)/sqrt(1
%           + Hsv(r+1)^2) + ... + Hsv(n)/sqrt(1 + Hsv(n)^2)),
%
%   with Hsv the vector containing the characteristic LQG singular values
%   of the system.
%
% INPUTS:
%   sys  - structure, containing the standard system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Beta            | nonnegative scalar, used as shift of the          |
%   |                 | in Bass' algorithm for better conditioning if     |
%   |                 | StabMethod == 'lyap' is chosen only used if       |
%   |                 | RiccatiSolver = 'newton'                          |
%   |                 | (default 0.1)                                     |
%   +-----------------+---------------------------------------------------+
%   | careopts        | structure, containing the optional parameters for |
%   |                 | the Riccati equation solver, see ml_caredl_sgn_fac|
%   |                 | and ml_care_nwt_fac                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the filter Gramian             |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the regulator Gramian          |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)) + Nu)                |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using absolute error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | RiccatiSolver   | character array, determining the solver for the   |
%   |                 | dual Riccati equations                            |
%   |                 |  'newton' - Newton iteration                      |
%   |                 |  'sign'   - dual sign function method             |
%   |                 | (default 'sign')                                  |
%   +-----------------+---------------------------------------------------+
%   | StabMethod      | character array, determining the method of        |
%   |                 | stabilization of the system, only used if         |
%   |                 | RiccatiSolver = 'newton'                          |
%   |                 |   'cabe' - partial stabilization with the         |
%   |                 |            algebraic Bernoulli equation           |
%   |                 |   'lyap' - partial stabilization with Bass'       |
%   |                 |            algorithm                              |
%   |                 | (default 'cabe')                                  |
%   +-----------------+---------------------------------------------------+
%   | stabmethodopts  | structure, containing the optional parameters for |
%   |                 | the sign function based Lyapunov or Bernoulli     |
%   |                 | equation solver used for the stabilization, only  |
%   |                 | used if RiccatiSolver = 'newton', see ml_cabe_sgn |
%   |                 | or ml_lyap_sgn                                    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabsignmopts   | structure, containing the optional parameters for |
%   |                 | the matrix sign function used for the             |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, only used if RiccatiSolver = 'newton',     |
%   |                 | see ml_signm                                      |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | UnstabDim       | integer, dimension of the deflating anti-stable   |
%   |                 | subspace in the control and filter Riccati        |
%   |                 | equations, only used if RiccatiSolver = 'newton', |
%   |                 | negative if unknown                               |
%   |                 | (default -1)                                      |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | (right) coprime factorization in H-infinity norm  |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the filter Gramian, if         |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the regulator Gramian, if      |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed characteristic  |
%   |                 | LQG values                                        |
%   +-----------------+---------------------------------------------------+
%   | infoCARE        | structure, containing information about the       |
%   |                 | sign function solver for the dual Riccati         |
%   |                 | equations, see ml_caredl_sgn_fac                  |
%   +-----------------+---------------------------------------------------+
%   | infoCAREC       | structure, containing information about the       |
%   |                 | Newton solver for the filter Riccati equation,    |
%   |                 | see ml_care_nwt_fac                               |
%   +-----------------+---------------------------------------------------+
%   | infoCAREO       | structure, containing information about the       |
%   |                 | Newton solver for the regulator Riccati equation, |
%   |                 | see ml_care_nwt_fac                               |
%   +-----------------+---------------------------------------------------+
%   | infoPARTSTABC   | structure, containing information about the       |
%   |                 | partial stabilization used for the controllability|
%   |                 | Riccati equation, see ml_ct_d_ss_partstab         |
%   +-----------------+---------------------------------------------------+
%   | infoPARTSTABO   | structure, containing information about the       |
%   |                 | partial stabilization used for the observability  |
%   |                 | Riccati equation, see ml_ct_d_ss_partstab         |
%   +-----------------+---------------------------------------------------+
%   | N {!}           | Dimension of the reduced-order model              |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, T. Stykel, Model order reduction for differential-algebraic
%   equations: A survey, in: A. Ilchmann, T. Reis (Eds.), Surveys in
%   Differential-Algebraic Equations IV, Differential-Algebraic Equations
%   Forum, Springer International Publishing, Cham, 2017, pp. 107--160.
%   https://doi.org/10.1007/978-3-319-46618-7_3
%
% See also ml_ct_d_dss_lqgbt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        % No extra action in main supported case.

    case 'ct_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to' ...
                ' full. Use ml_ct_s_foss_lqgbt to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse standard system detected.' ...
                ' Use ml_ct_s_foss_lqgbt to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'ct_d_ss';
sys            = ml_prepare_system_data(sys);

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'careopts')
    assert(isa(opts.careopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.careopts has to be a struct!');
else
    opts.careopts = struct();
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 5;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'RiccatiSolver')
    assert(strcmpi(opts.RiccatiSolver, 'newton') ...
        || strcmpi(opts.RiccatiSolver, 'sign'), ...
        'MORLAB:data', ...
        'The desired Riccati equation solver is not implemented!');
else
    opts.RiccatiSolver = 'sign';
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PARTIAL STABILIZATION AND SOLVE MATRIX EQUATIONS.                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation of matrices for Riccati equations.
A = sys.A;
B = sys.B;
C = sys.C;

if not(isempty(sys.D))
    Rb = eye(size(sys.B, 2)) + sys.D' * sys.D;
    Lb = chol(Rb);
    Rc = eye(size(sys.C, 1)) + sys.D * sys.D';
    Lc = chol(Rc);

    A = A - sys.B * (sys.D' * (Rc \ sys.C));
    B = B / Lb;
    C = Lc' \ C;
end

if strcmpi(opts.RiccatiSolver, 'sign')
    if ml_field_set_to_value(opts, 'GramFacC') ...
            && ml_field_set_to_value(opts, 'GramFacO')
        % Both precomputed Gramian factors.
        R = opts.GramFacC;
        L = opts.GramFacO;
    elseif ml_field_set_to_value(opts, 'GramFacC')
        % Precomputed filter factor.
        opts.careopts.EqnType = 'primal';
        [L, ~, infoCARE]      = ml_caredl_sgn_fac(A, B, C, [], -1, ...
            opts.careopts);
        R                     = opts.GramFacC;
        info.infoCARE         = infoCARE;
    elseif ml_field_set_to_value(opts, 'GramFacO')
        % Precomputed regulator factor.
        opts.careopts.EqnType = 'dual';
        [~, R, infoCARE]      = ml_caredl_sgn_fac(A, B, C, [], -1, ...
            opts.careopts);
        L                     = opts.GramFacO;
        info.infoCARE         = infoCARE;
    else
        % No Gramian factors precomputed.
        opts.careopts.EqnType = 'both';
        [L, R, infoCARE]      = ml_caredl_sgn_fac(A, B, C, [], -1, ...
            opts.careopts);
        info.infoCARE         = infoCARE;
    end
else
    if ml_field_set_to_value(opts, 'GramFacC') ...
            && ml_field_set_to_value(opts, 'GramFacO')
        % Both precomputed Gramian factors.
        R = opts.GramFacC;
        L = opts.GramFacO;
    elseif ml_field_set_to_value(opts, 'GramFacC')
        % Precomputed filter factor.
        [K2, infoPARTSTABO] = ml_ct_d_ss_partstab(A, B, opts);
        info.infoPARTSTABO  = infoPARTSTABO;

        opts.careopts.K0 = K2;
        [L, infoCAREO]   = ml_care_nwt_fac(A, B, C, [], opts.careopts);
        info.infoCAREO   = infoCAREO;

        R = opts.GramFacC;
    elseif ml_field_set_to_value(opts, 'GramFacO')
        % Precomputed regulator factor.
        [K1, infoPARTSTABC] = ml_ct_d_ss_partstab(A', C', opts);
        info.infoPARTSTABC  = infoPARTSTABC;

        opts.careopts.K0 = K1;
        [R, infoCAREC]   = ml_care_nwt_fac(A', C', B', [], opts.careopts);
        info.infoCAREC   = infoCAREC;

        L = opts.GramFacO;
    else
        % No Gramian factors precomputed.
        [K1, infoPARTSTABC] = ml_ct_d_ss_partstab(A', C', opts);
        info.infoPARTSTABC  = infoPARTSTABC;

        opts.careopts.K0 = K1;
        [R, infoCAREC]   = ml_care_nwt_fac(A', C', B', [], opts.careopts);
        info.infoCAREC   = infoCAREC;

        [K2, infoPARTSTABO] = ml_ct_d_ss_partstab(A, B, opts);
        info.infoPARTSTABO  = infoPARTSTABO;

        opts.careopts.K0 = K2;
        [L, infoCAREO]   = ml_care_nwt_fac(A, B, C, [], opts.careopts);
        info.infoCAREO   = infoCAREO;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection basis matrices.
[V, W, hsv] = ml_balproj_proper(sys, R, L, 0, rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
rom = ml_projtrunc_proper(sys, V, W);

if not(isa(rom, 'cell')), rom = {rom}; end

numRoms = length(rom);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
n           = cell(1, numRoms);
absErrBound = cell(1, numRoms);
for k = 1:numRoms
    n{k}           = size(rom{k}.A, 1);
    absErrBound{k} = 2 * sum(hsv(n{k}+1:end) ...
        ./ sqrt(1 + hsv(n{k}+1:end).^2));
end

info.AbsErrBound = absErrBound;
info.Hsv         = hsv;
info.N           = n;

% Store final projection bases.
if opts.StoreProjection
    for k = 1:numRoms
        info.V{k} = V{k};
        info.W{k} = W{k};
    end
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
