function [rom, info] = ml_ct_d_ss_bst(sys, opts)
%ML_CT_D_SS_BST Balanced stochastic truncation for dense, standard systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SS_BST(sys)
%   [rom, info] = ML_CT_D_SS_BST(sys, opts)
%
% DESCRIPTION:
%   This function computes the balanced stochastic truncation for a
%   standard system of the form
%
%       x'(t) = A*x(t) + B*u(t),                                        (1)
%        y(t) = C*x(t) + D*u(t).                                        (2)
%
%   Therefore, first the Lyapunov equation
%
%       A*P + P*A' + B*B' = 0,
%
%   is solved for the controllability Gramian P and then, the corresponding
%   Riccati equation
%
%       A'*Q + Q*A + (C - Bw' * Q)' * inv(D*D') * (C - Bw' * Q) = 0
%
%   is solved for the Gramian Q, with
%
%       Bw = B*D' + P*C'.
%
%   As result, a reduced-order system of the form
%
%       x'(t) = Ar*x(t) + Br*u(t),                                      (3)
%        y(t) = Cr*x(t) + Dr*u(t)                                       (4)
%
%   is computed, such that for the original transfer function G and and the
%   r-th order transfer function Gr it holds
%
%       ||G - Gr||_{\infty} / ||G||_{\infty}
%       <= ((1 + Hsv(r+1))/(1 - Hsv(r+1)) * ...
%          * (1 + Hsv(n))/(1 - Hsv(n))) + 1,
%
%   with Hsv, a vector containing the characteristic stochastic singular
%   values of the system. If the transfer function is invertible it holds
%
%       ||inv(G)*(G - Gr)||_{\infty}
%       <= ((1 + Hsv(r+1))/(1 - Hsv(r+1)) * ...
%          * (1 + Hsv(n))/(1 - Hsv(n))) - 1.
%
%   Notes:
%     1) The equations above are defined for the case of p < m. Otherwise
%        the system is transposed, then reduced and back transposed.
%     2) In case of a rank-deficient D term, an epsilon regularization is
%        performed, which replaces the D during the computations with an
%        identity matrix scaled by a given epsilon.
%     3) For unstable systems, first an additive decomposition into the
%        stable and anti-stable parts is performed and then only the
%        stable part will be reduced.
%
% INPUTS:
%   sys  - structure, containing the standard system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | careopts        | structure, containing the optional parameters for |
%   |                 | the Riccati equation solver, see ml_caredl_sgn_fac|
%   |                 | and ml_care_nwt_fac                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Epsilon         | positive scalar, used in the case of a            |
%   |                 | non-full-rank D term for epsilon regularization   |
%   |                 | by multiplying with an identity matrix of         |
%   |                 | appropriate size                                  |
%   |                 | (default 1.0e-03)                                 |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | lyapopts        | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time algebraic  |
%   |                 | Lyapunov equation, see ml_lyap_sgn_fac            |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)) + Nu)                |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using relative error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | RiccatiSolver   | character array, determining the solver for the   |
%   |                 | dual Riccati equations                            |
%   |                 |  'newton' - Newton iteration                      |
%   |                 |  'sign'   - dual sign function method             |
%   |                 | (default 'sign')                                  |
%   +-----------------+---------------------------------------------------+
%   | stabsignmopts   | structure, containing the optional parameters for |
%   |                 | the matrix sign function used for the             |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, see ml_signm                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabsylvopts    | structure, containing the optional parameters for |
%   |                 | the Sylvester equation solver used for the        |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, see ml_sylv_sgn                            |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an relative error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | UnstabDim       | integer, dimension of the deflating anti-stable   |
%   |                 | subspace, negative if unknown                     |
%   |                 | (default -1)                                      |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed characteristic  |
%   |                 | stochastic singular values                        |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | stable and anti-stable parts, see ml_ct_ss_adtf   |
%   +-----------------+---------------------------------------------------+
%   | infoCARE        | structure, containing information about the       |
%   |                 | sign function solver for the dual Riccati         |
%   |                 | equations, see ml_caredl_sgn_fac                  |
%   +-----------------+---------------------------------------------------+
%   | infoLYAP        | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation sovler for the  |
%   |                 | controllability Gramian, see ml_lyap_sgn_fac      |
%   +-----------------+---------------------------------------------------+
%   | infoPCARE       | structure, containing information about the       |
%   |                 | continuous-time algebraic positive Riccati        |
%   |                 | equation for the observability Gramian, see       |
%   |                 | ml_pcare_nwt_fac                                  |
%   +-----------------+---------------------------------------------------+
%   | Ns              | Dimension of the stable part of the reduced-order |
%   | {!}             | model                                             |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Dimension of the anti-stable part of the reduced- |
%   |                 | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | RelErrBound     | computed error bound for the relative error of the|
%   | {!}             | of the reduced-order model in H-infinity norm     |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, E. S. Quintana-Orti, G. Quintana-Orti, Efficient numerical
%   algorithms for balanced stochastic truncation, Int. J. Appl. Math.
%   Comput. Sci. 11 (5) (2001) 1123--1150.
%
% See also ml_ct_d_dss_bst, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        % No extra action in main supported case.

    case 'ct_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to' ...
                ' full. Use ml_ct_s_foss_bst to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse standard system detected.' ...
                ' Use ml_ct_s_foss_bst to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'ct_d_ss';
sys            = ml_prepare_system_data(sys);

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'careopts')
    assert(isa(opts.careopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.careopts has to be a struct!');
else
    opts.careopts = struct();
end

if ml_field_set_to_value(opts, 'Epsilon')
    ml_assert_posscalar(opts.Epsilon, 'opts.Epsilon');
else
    opts.Epsilon = 1.0e-03;
end

if ml_field_set_to_value(opts, 'lyapopts')
    assert(isa(opts.lyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapopts has to be a struct!');
else
    opts.lyapopts = struct();
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 4;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'RiccatiSolver')
    assert(strcmpi(opts.RiccatiSolver, 'newton') ...
        || strcmpi(opts.RiccatiSolver, 'sign'), ...
        'MORLAB:data', ...
        'The desired Riccati equation solver is not implemented!');
else
    opts.RiccatiSolver = 'sign';
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION OF TRANSFER FUNCTION.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition of the system into the stable and anti-stable parts.
[sys, infoADTF] = ml_ct_d_ss_adtf(sys, opts);

% Assign information about additive decomposition.
info.infoADTF = rmfield(infoADTF, {'V'; 'W'});


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

m = size(sys.B, 2);
p = size(sys.C, 1);

% Transpose system, if there are more outputs than inputs.
if m < p
    A = sys.A';
    B = sys.C';
    C = sys.B';
    D = sys.D';
else
    A = sys.A;
    B = sys.B;
    C = sys.C;
    D = sys.D;
end

% Computation of the full-rank factors of the system Gramians.
if size(sys.A, 1) > 0
    if (m >= p) && ml_field_set_to_value(opts, 'GramFacC')
        R             = opts.GramFacC;
        info.infoLYAP = [];
    elseif (m < p) && ml_field_set_to_value(opts, 'GramFacO')
        R             = opts.GramFacO;
        info.infoLYAP = [];
    else
        [R, infoLYAP] = ml_lyap_sgn_fac(A, B, [], opts.lyapopts);
        info.infoLYAP = infoLYAP;
    end

    if (m >= p) && ml_field_set_to_value(opts, 'GramFacO')
        L             = opts.GramFacO;
        info.infoCARE = [];
    elseif (m < p) && ml_field_set_to_value(opts, 'GramFacC')
        L             = opts.GramFacC;
        info.infoCARE = [];
    else
        % Epsilon regularization.
        if (rank(D) < min(m, p)) ...
                || (norm(D, 'fro') < 1.0e+02 * eps)
            D  = opts.Epsilon * eye(min(m, p));
            Bw = B * opts.Epsilon * eye(min(m, p), max(m, p))' + ...
                R * (R' * C');
        else
            Bw     = B * D' + R * (R' * C');
            [~, D] = qr(D', 0);
        end

        A = A - Bw * ((D' * D) \ C);
        B = Bw / D;
        C = D' \ C;

        if strcmpi(opts.RiccatiSolver, 'sign')
            opts.careopts.EqnType = 'primal';
            [L, ~, infoCARE]      = ml_caredl_sgn_fac(A, B, C, [], 1, ...
                opts.careopts);
            info.infoCARE         = infoCARE;
        else
            [L, infoPCARE] = ml_pcare_nwt_fac(A, B, C, [], opts.careopts);
            info.infoPCARE = infoPCARE;
        end
    end
else
    R = [];
    L = [];
end

% Swap Gramian factors if system was transposed.
if m < p
    [R, L] = deal(L, R);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection basis matrices.
[V, W, hsv] = ml_balproj_proper(sys, R, L, size(sys.Au, 1), rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
stabrom = ml_projtrunc_proper(sys, V, W);

% Construct final reduced-order models from stable and unstable parts.
if not(isa(stabrom, 'cell')), stabrom = {stabrom}; end

numRoms = length(stabrom);
rom     = cell(1, numRoms);
for k = 1:numRoms
    rom{k}.A = blkdiag(stabrom{k}.A, sys.Au);
    rom{k}.B = [stabrom{k}.B; sys.Bu];
    rom{k}.C = [stabrom{k}.C, sys.Cu];
    rom{k}.D = sys.D;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
n           = cell(1, numRoms);
relErrBound = cell(1, numRoms);
for k = 1:numRoms
    n{k}           = size(stabrom{k}.A, 1);
    relErrBound{k} = prod((1 + hsv(n{k}+1:end)) ...
        ./ (1 - hsv(n{k}+1:end))) - 1;
end

info.Hsv         = hsv;
info.Ns          = n;
info.Nu          = size(sys.Au, 1);
info.RelErrBound = relErrBound;

% Store final projection bases.
if opts.StoreProjection
    for k = 1:numRoms
        info.V{k} = infoADTF.V * blkdiag(V{k}, eye(info.Nu));
        info.W{k} = infoADTF.W * blkdiag(W{k}, eye(info.Nu));
    end
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
