function [rom, info] = ml_ct_d_soss_mt(sys, opts)
%ML_CT_D_SOSS_MT Modal trunc. for dense, continuous second-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SOSS_BT(sys)
%   [rom, info] = ML_CT_D_SOSS_BT(sys, opts)
%
% DESCRIPTION:
%   This function computes the modal truncation for a second-order
%   system of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   Therefore, a block diagonalization of the matrix pencil s*E - A is
%   performed using the matrix disk function, such that
%
%            [ Ei 0  0  ]       [ Ai 0  0  ]
%       E2 = [ 0  E0 0  ], A2 = [ 0  A0 0  ],
%            [ 0  0  E1 ]       [ 0  0  A1 ]
%
%                                 [ Ci ]
%       B2 = [ Bi, B0, B1 ], C2 = [ C0 ],
%                                 [ C1 ]
%
%   where the matrix pencil s*Ei - Ai contains all infinite eigenvalues
%   and s*E0 - A0 the finite eigenvalues with the real part larger than a
%   given alpha using the linearization
%
%           [ I  0 ]      [ 0   I ]      [ 0  ]
%       E = [      ], A = [       ], B = [    ], C = [Cp, Cv].
%           [ 0  M ]      [ -K -E ]      [ Bu ]
%
%   As a result, a reduced model in either first-order
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t).                                   (4)
%
%   Note: There is currently no structure-preserving variant implemented.
%         This function calls ml_ct_d_dss_mt to construct the first-order
%         reduced-order model.
%
% INPUTS:
%   sys  - structure, containing the second-order system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Alpha           | real scalar, such that all finite eigenvalues with|
%   | {!}             | the real part smaller than Alpha are truncated    |
%   |                 | (default -1.0)                                    |
%   +-----------------+---------------------------------------------------+
%   | DecompEig       | positive scalar, overestimation of the absolute   |
%   |                 | value of the largest finite eigenvalue of s*E - A,|
%   |                 | if set, replaces the computation with DecompTol   |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | DecompTol       | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of E to determine the      |
%   |                 | smallest non-quasi-zero singular value of E       |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | gdlyapopts      | structure, containing the optional parameters for |
%   |                 | the computation of the generalized discrete-time  |
%   |                 | Lyapunov equations, see ml_gdlyapdl_smith_fac     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the observability improper     |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | ImproperTrunc   | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest improper Hankel singular value of the     |
%   |                 | system to truncate the improper part, if 0 no     |
%   |                 | improper balanced truncation is performed         |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Index           | nonnegative integer, index of the descriptor      |
%   |                 | system used to set an upper bound on the size of  |
%   |                 | the reduced improper part, Inf if unknown         |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | signmopts       | structure, containing the optional parameters for |
%   |                 | the sign function method, see ml_signm            |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | infdecopts      | structure, containing the optional parameters for |
%   |                 | the decomposition of the finite and infinite parts|
%   |                 | of the system using the disk function and subspace|
%   |                 | extraction method, see ml_disk and ml_getqz       |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | OutputModel     | character array, determining if classical or      |
%   |                 | structure-preserving balanced truncation is used  |
%   |                 |  'fo' - first-order MT, see ml_ct_d_dss_mt for    |
%   |                 |         for optional parameters                   |
%   |                 |  'so' - second-order MT (not yet implemented)     |
%   |                 | (default 'fo')                                    |
%   +-----------------+---------------------------------------------------+
%   | RankTol         | nonnegative scalar, tolerance used for the        |
%   |                 | determination of deflating subspaces              |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order model, with the
%   {!}    following entries if opts.OutputModel = 'fo'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%          and the following entries if opts.OutputModel = 'so'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (5) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (6) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Hsvi            | a vector, containing the computed Hankel singular |
%   |                 | values of the improper part of the system         |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the improper observability     |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | infoSIGNM       | structure, containing information about the sign  |
%   | {!}             | function method used for the modal truncation of  |
%   |                 | the system, see ml_signm                          |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | infinite and finite parts, see ml_ct_d_dss_adtf   |
%   +-----------------+---------------------------------------------------+
%   | Np              | Dimension of the finite part in the reduced-order |
%   | {!}             | model                                             |
%   +-----------------+---------------------------------------------------+
%   | Ni              | Dimension of the infinite part in the reduced-    |
%   |                 | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_s_ss_mt, ml_ct_d_dss_mt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_soss'
        % No extra action in main supported case.

    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        error('MORLAB:notImplemented', ...
            ['Large-scale sparse second-order system detected!' ...
            ' This functionality is not implemented yet.']);

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Enforce DAE handling.
if strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so') ...
        && strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so') ...
        && strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')
    if ml_field_set_to_value(opts, 'OutputModel') ...
            && strcmpi(opts.OutputModel, 'so')
        warning('MORLAB:data', ...
            ['Dense second-order descriptor systems cannot be reduced' ...
            ' structure-preserving. Changed opts.OutputModel to ''fo''.']);
    end

    opts.OutputModel = 'fo';
end

sys.SystemType = 'ct_d_soss';
sys            = ml_prepare_system_data(sys);

% Temporary variables.
n = size(sys.K, 1);
A = [zeros(n), eye(n); -sys.K, -sys.E];
B = sys.pB;
C = sys.pC;
D = sys.pD;
E = [eye(n), zeros(n); zeros(n), sys.M];

% Change to descriptor balanced truncation if necessary.
if not(ml_field_set_to_value(opts, 'OutputModel')) ...
        || (ml_field_set_to_value(opts, 'OutputModel') ...
        && strcmpi(opts.OutputModel, 'fo'))

    sysfo = struct( ...
        'A', A, ...
        'B', B, ...
        'C', C, ...
        'D', D, ...
        'E', E);

    [rom, info] = ml_ct_d_dss_mt(sysfo, opts);

    return;
elseif strcmpi(opts.OutputModel, 'so')
    error('MORLAB:notImplemented', ...
        ['A structure-preserving modal truncation is not' ...
        ' implemented yet.']);
elseif not(strcmpi(opts.OutputModel, 'so'))
    error('MORLAB:data', ...
        'Unknown value for opts.OutputModel.');
end
