function [rom, info] = ml_ct_d_soss_hna(sys, opts)
%ML_CT_D_SOSS_HNA HNA for dense, continuous second-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SOSS_HNA(sys)
%   [rom, info] = ML_CT_D_SOSS_HNA(sys, opts)
%
% DESCRIPTION:
%   This function computes the generalized Hankel-norm approximation of a
%   second-order system
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   Therefore, first a balanced realization is computed by using the
%   generalized balanced truncation square-root method with an appropriate
%   tolerance for the minimal realization of the given system. Then the
%   strictly proper part of the system is transformed using the formulas
%   for all-pass systems. As result, a reduced-order system of the form
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   is computed, such that for the original transfer function G and the
%   reduced-order transfer function Gr with an r-th order strictly proper
%   part it holds
%
%       ||G - Gr||_{H}       = Hsvp(r+1),
%       ||G - Gr||_{\infty} <= 2 * (Hsvp(r+1) + ... + Hsvp(n)),
%
%   with Hsvp, a vector containing the proper Hankel singular values of the
%   system.
%
%   Note: For unstable systems, an additional additive decomposition into
%         the stable and anti-stable parts is performed and then only the
%         stable part will be reduced. That does not change the error
%         formulas.
%
% INPUTS:
%   sys  - structure or state-space object, containing the descriptor
%          system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | DecompEig       | positive scalar, overestimation of the absolute   |
%   |                 | value of the largest finite eigenvalue of s*E - A,|
%   |                 | if set, replaces the computation with DecompTol   |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | DecompTol       | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of E to determine the      |
%   |                 | smallest non-quasi-zero singular value of E       |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | gdlyapopts      | structure, containing the optional parameters for |
%   |                 | the computation of the generalized discrete-time  |
%   |                 | Lyapunov equations, see ml_gdlyapdl_smith_fac     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | hankeldecopts   | structure, containing the optional parameters for |
%   |                 | the disk function used for the decomposition      |
%   |                 | after the transformation to an all-pass system    |
%   |                 | see ml_disk and ml_getqz                          |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the observability improper     |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | ImproperTrunc   | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest proper Hankel singular value of the       |
%   |                 | system to truncate the improper part, if 0 no     |
%   |                 | improper balanced truncation is performed         |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Index           | nonnegative integer, index of the descriptor      |
%   |                 | system used to set an upper bound on the size of  |
%   |                 | the reduced improper part, Inf if unknown         |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | infdecopts      | structure, containing the optional parameters for |
%   |                 | the decomposition of the finite and infinite parts|
%   |                 | of the system using the disk function and subspace|
%   |                 | extraction method, see ml_disk and ml_getqz       |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | lyapopts        | structure, containing the optional parameters for |
%   |                 | the computation of the generalized continuous-time|
%   |                 | Lyapunov equations, see ml_lyapdl_sgn_fac         |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | MinRelTol       | nonnegative scalar, tolerance multiplied with the |
%   | {!}             | largest characteristic value to determine a       |
%   |                 | minimal realization                               |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsvp)) + Nu + Ni)          |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using absolute error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | stabdecopts     | structure, containing the optional parameters for |
%   |                 | the decomposition of the stable and unstable parts|
%   |                 | of the system using the sign function and subspace|
%   |                 | extraction method, see ml_signm and ml_getqz      |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   rom  - structure or state-space object, containing the reduced-order
%          descriptor system:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | reduced-order model in H-infinity norm            |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsvi            | a vector, containing the computed Hankel singular |
%   |                 | values of the improper part of the system         |
%   +-----------------+---------------------------------------------------+
%   | Hsvp            | a vector, containing the computed Hankel singular |
%   |                 | values of the proper part of the system           |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the improper observability     |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | infinite, finite stable and finite anti-stable    |
%   |                 | parts, see ml_ct_soss_adtf                         |
%   +-----------------+---------------------------------------------------+
%   | infoGDLYAP      | structure, containing information about the       |
%   |                 | generalized discrete-time Lyapunov equation solver|
%   |                 | for the improper Gramians,                        |
%   |                 | see ml_gdlyapdl_smith_fac                         |
%   +-----------------+---------------------------------------------------+
%   | infoHAADTF      | structure, containing information about the       |
%   |                 | disk function method, see ml_ct_soss_adtf          |
%   +-----------------+---------------------------------------------------+
%   | infoLYAP        | structure, containing information about the       |
%   |                 | continuous-time dual Lyapunov equations solver,   |
%   |                 | see ml_lyapdl_sgn_fac                             |
%   +-----------------+---------------------------------------------------+
%   | Ni              | Dimension of the improper part in the reduced-    |
%   | {!}             | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | Np              | Dimension of the proper part in the reduced-order |
%   | {!}             | model                                             |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Dimension of the unstable part in the reduced-    |
%   |                 | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | Sigma           | Chosen proper Hankel singular value, exact        |
%   | {!}             | approximation error in the Hankel-norm            |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   S. Werner, Hankel-norm approximation of descriptor systems, Master's
%   thesis, Otto von Guericke University, Magdeburg, Germany (2016).
%   http://nbn-resolving.de/urn:nbn:de:gbv:ma9:1-8845
%
% See also ml_ct_dss_hna, ml_ct_d_soss_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_soss'
        % No extra action in main supported case.

    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        if size(sys.K, 1) <= 2500
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to' ...
                ' full. Use ml_ct_s_soss_hna to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse second-order system detected! ' ...
                ' Use ml_ct_s_soss_hna to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'ct_d_soss';
sys            = ml_prepare_system_data(sys);

% Temporary variables.
n = size(sys.K, 1);
A = [zeros(n), eye(n); -sys.K, -sys.E];
B = sys.pB;
C = sys.pC;
D = sys.pD;
E = [eye(n), zeros(n); zeros(n), sys.M];

% Change to descriptor balanced truncation if necessary.
sysfo = struct( ...
    'A', A, ...
    'B', B, ...
    'C', C, ...
    'D', D, ...
    'E', E);

[rom, info] = ml_ct_d_dss_hna(sysfo, opts);
