function [rom, info] = ml_ct_d_soss_flbt(sys, opts)
%ML_CT_D_SOSS_FLBT FLBT for dense, continuous second-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SOSS_FLBT(sys)
%   [rom, info] = ML_CT_D_SOSS_FLBT(sys, opts)
%
% DESCRIPTION:
%   This function computes the freuqency-limited balanced truncation for a
%   second-order system of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   Therefore, the two generalized Lyapunov equations
%
%       A*P*E' + E*P*A' + BF*B' + B*BF' = 0,
%       A'*Q*E + E'*Q*A + CF'*C + C'*CF = 0,
%
%   where BF and CF are frequency-dependent matrices, are solved for the
%   system Gramians P and Q, with
%
%           [ I  0 ]      [ 0   I ]      [ 0  ]
%       E = [      ], A = [       ], B = [    ], C = [Cp, Cv].
%           [ 0  M ]      [ -K -E ]      [ Bu ]
%
%   As a result, a reduced model in either first-order
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   or second-order form
%
%       Mr*x''(t) = -Kr*x(t) -  Er*x'(t) + Bur*u(t)                     (5)
%            y(t) = Cpr*x(t) + Cvr*x'(t) +  Dr*u(t)                     (6)
%
%   is computed. Using the modified Gramian approach can lead to a more
%   stable reduced-order model but it is at most stable as the classical
%   second-order balanced truncation method.
%
%   Note: In the structure-preserving version, only asymptotically stable
%         systems without algebraic constraints are allowed. In the
%         first-order version ml_ct_d_dss_flbt is used such that additional
%         optional arguments are available and the info struct has
%         more fields, which will not be particularly outlined below.
%
% INPUTS:
%   sys  - structure, containing the second-order system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | BalanceType     | character array, determining the type of          |
%   | {!}             | second-order balancing, only if OutputModel = 'so'|
%   |                 |   'p'   - position balancing                      |
%   |                 |   'pm'  - position balancing (diagonalized M)     |
%   |                 |   'pv'  - position-velocity balancing             |
%   |                 |   'vp'  - velocity-position balancing             |
%   |                 |   'vpm' - velocity-position balancing (diag. M)   |
%   |                 |   'v'   - velocity balancing                      |
%   |                 |   'fv'  - free velocity balancing                 |
%   |                 |   'so'  - second-order balancing                  |
%   |                 | (default 'so')                                    |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | lyapopts        | structure, containing the optional parameters for |
%   |                 | the computation of the generalized continuous-time|
%   |                 | Lyapunov equations, see ml_lyapdl_sgn_ldl if      |
%   |                 | ModGramian = 0 and ml_lyapdl_sgn_fac if           |
%   |                 | ModGramian = 1                                    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | ModGramian      | {0, 1}, used to disable/enable the modified       |
%   |                 | Gramian approach                                  |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the hsv,  |
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 |  'sum'       - using rel. tolerance for sum of hsv|
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | OutputModel     | character array, determining if classical or      |
%   |                 | structure-preserving balanced truncation is used  |
%   |                 |  'fo' - first-order FLBT, see ml_ct_d_dss_flbt    |
%   |                 |         for optional parameters                   |
%   |                 |  'so' - second-order FLBT                         |
%   |                 | (default 'fo')                                    |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | if 'tolerance' or 'sum' is set for                |
%   |                 | OrderComputation                                  |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order model, with the
%   {!}    following entries if opts.OutputModel = 'fo'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%          and the following entries if opts.OutputModel = 'so'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (5) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (6) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed Hankel singular |
%   | {!}             | values, for second-order balancing the position   |
%   |                 | and velocity Hankel singular values are saved in  |
%   |                 | a struct                                          |
%   +-----------------+---------------------------------------------------+
%   | infoLYAP        | structure, containing information about the       |
%   |                 | continuous-time dual Lyapunov equations solver,   |
%   |                 | see ml_lyapdl_sgn_ldl or ml_lyapdl_sgn_fac        |
%   +-----------------+---------------------------------------------------+
%   | N {!}           | Dimension of the reduced-order model              |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE: TODO!
%
% See also ml_ct_d_soss_bt, ml_ct_d_dss_flbt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_soss'
        % No extra action in main supported case.

    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        if size(sys.K, 1) <= 2500
            warning('MORLAB:data', ...
                'System matrices are converted from sparse to full.');
        else
            error('MORLAB:notImplemented', ...
                ['Large-scale sparse second-order system detected.' ...
                ' For this system structure there is no method' ...
                ' implemented yet.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Enforce DAE handling.
if strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so') ...
        && strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so') ...
        && strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')
    if ml_field_set_to_value(opts, 'OutputModel') ...
            && strcmpi(opts.OutputModel, 'so')
        warning('MORLAB:data', ...
            ['Dense second-order descriptor systems cannot be reduced' ...
            ' structure-preserving. Changed opts.OutputModel to ''fo''.']);
    end

    opts.OutputModel = 'fo';
end

sys.SystemType = 'ct_d_soss';
sys            = ml_prepare_system_data(sys);

% Temporary variables.
n = size(sys.K, 1);
A = [zeros(n), eye(n); -sys.K, -sys.E];
B = sys.pB;
C = sys.pC;
D = sys.pD;
E = [eye(n), zeros(n); zeros(n), sys.M];

% Change to descriptor balanced truncation if necessary.
if not(ml_field_set_to_value(opts, 'OutputModel')) ...
        || (ml_field_set_to_value(opts, 'OutputModel') ...
        && strcmpi(opts.OutputModel, 'fo'))

    sysfo = struct( ...
        'A', A, ...
        'B', B, ...
        'C', C, ...
        'D', D, ...
        'E', E);

    [rom, info] = ml_ct_d_dss_flbt(sysfo, opts);

    return;
elseif not(strcmpi(opts.OutputModel, 'so'))
    error('MORLAB:data', ...
        'Unknown value for opts.OutputModel.');
end

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'FreqRange')
    ml_assert_nonnegvector(opts.FreqRange, 'opts.FreqRange');

    assert(mod(length(opts.FreqRange), 2) == 0, ...
        'MORLAB:data', ...
        'The parameter opts.FreqRange has to be an even vector!');

    opts.FreqRange = sort(opts.FreqRange);
else
    opts.FreqRange = [0, 1.0e+03];
end

if ml_field_set_to_value(opts, 'lyapopts')
    assert(isa(opts.lyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapopts has to be a struct!');
else
    opts.lyapopts = struct();
end

if ml_field_set_to_value(opts, 'ModGramian')
    ml_assert_boolean(opts.ModGramian, 'opts.ModGramian');
else
    opts.ModGramian = 0;
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 1;
    elseif strcmpi(opts.OrderComputation{k}, 'sum')
        rselect{k} = 2;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

n = size(A, 1);
m = size(B, 2);
p = size(C, 1);

% Computation of the full-rank factors of the proper Gramians.
if n > 0
    if ml_field_set_to_value(opts, 'GramFacC') ...
            && ml_field_set_to_value(opts, 'GramFacO')
        % Both precomputed Gramian factors.
        R = opts.GramFacC;
        L = opts.GramFacO;
    else
        % Compute frequency-dependent right-hand side.
        if opts.FreqRange(1) == 0
            F      = -E \ A - 1i * opts.FreqRange(2) * eye(n);
            kstart = 2;
        else
            F      = eye(n);
            kstart = 1;
        end

        for k = kstart:1:length(opts.FreqRange)/2
            F = F * ((A + 1i * opts.FreqRange(2*k-1) * E) ...
                \ (A + 1i * opts.FreqRange(2*k) * E));
        end

        F  = (1i / pi) * logm(F);
        BF = E * real(F * (E \ B));
        CF = real(C * F);
        BQ = [zeros(m), eye(m); eye(m), zeros(m)];
        CQ = [zeros(p), eye(p); eye(p), zeros(p)];

        if opts.ModGramian
            [CF, Y] = ml_compress_ldl([C; CF], CQ, eps, 'row');
            Y       = abs(diag(Y));
            CF      = diag(sqrt(Y)) * CF;

            [BF, Y] = ml_compress_ldl([B, BF], BQ, eps, 'col');
            Y       = abs(diag(Y));
            BF      = BF * diag(sqrt(Y));
        end

        if ml_field_set_to_value(opts, 'GramFacC')
            % Precomputed controllability factor.
            if opts.ModGramian
                % Modified Gramian approach.
                [L, infoLYAP] = ml_lyap_sgn_fac(A', CF', E', ...
                    opts.lyapopts);
            else
                % Classical method.
                [L, LY, infoLYAP] = ml_lyap_sgn_ldl(A', ...
                    [C; CF]', CQ', E', opts.lyapopts);

                LY  = diag(LY);
                idx = (LY > 0);
                L   = L(:, idx) * diag(sqrt(LY(idx)));
            end

            R = opts.GramFacC;
        elseif ml_field_set_to_value(opts, 'GramFacO')
            % Precomputed observability factor.
            if opts.ModGramian
                % Modified Gramian approach.
                [R, infoLYAP] = ml_lyap_sgn_fac(A, BF, E, ...
                    opts.lyapopts);
            else
                % Classical method.
                [R, RY, infoLYAP] = ml_lyap_sgn_ldl(A, ...
                    [B, BF], BQ, E, opts.lyapopts);

                RY  = diag(RY);
                idx = (RY > 0);
                R   = R(:, idx) * diag(sqrt(RY(idx)));
            end

            L = opts.GramFacO;
        else
            % No Gramian factors precomputed.
            if opts.ModGramian
                % Modified Gramian approach.
                [R, L, infoLYAP] = ml_lyapdl_sgn_fac(A, BF, CF, ...
                    E, opts.lyapopts);
            else
                % Classical method.
                [R, RY, L, LY, infoLYAP] = ml_lyapdl_sgn_ldl(A, ...
                    [B, BF], BQ, [C; CF], CQ, E, opts.lyapopts);

                RY  = diag(RY);
                idx = (RY > 0);
                R   = R(:, idx) * diag(sqrt(RY(idx)));

                LY  = diag(LY);
                idx = (LY > 0);
                L   = L(:, idx) * diag(sqrt(LY(idx)));
            end
        end

        % Assig information about Lyapunov equation solver.
        info.infoLYAP = infoLYAP;
    end
else
    [R, L] = deal([]);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[V, W, hsv] = ml_balproj_soss(sys, R, L, rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
rom = ml_projtrunc_soss(sys, V, W);

if not(isa(rom, 'cell')), rom = {rom}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
numRoms  = length(rom);
info.Hsv = hsv;
info.N   = cell(1, numRoms);
for k = 1:numRoms
    info.N{k} = size(rom{k}.M, 1);
end

if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

[rom, info] = ml_format_output(rom, 1, info);
