function [rom, info] = ml_ct_d_soss_bst(sys, opts)
%ML_CT_D_SOSS_BST BST for dense, continuous second-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SOSS_BST(sys)
%   [rom, info] = ML_CT_D_SOSS_BST(sys, opts)
%
% DESCRIPTION:
%   This function computes the bounded-real balanced truncation for a
%   second-order system of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   Therefore, the continuous-time Lyapunov equation
%
%       A*P*E' + E*P*A' + B*B' = 0
%
%   is solved for P and then, the corresponding Riccati equation
%
%       A'*Q*E + E'*Q*A
%       + (C - Bw' * Q * E)' * inv(R) * (C - Bw' * Q * E) = 0
%
%   is solved for Q, with R = D * D'. As a result, a reduced model in
%   either first-order
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   or second-order form
%
%       Mr*x''(t) = -Kr*x(t) -  Er*x'(t) + Bur*u(t)                     (5)
%            y(t) = Cpr*x(t) + Cvr*x'(t) +  Dr*u(t)                     (6)
%
%   is computed.
%
%   Note: In the structure-preserving version, only asymptotically stable
%         systems without algebraic constraints are allowed. In the
%         first-order version ml_ct_d_dss_bst is used such that
%         additional optional arguments are available and the info struct
%         has more fields, which will not be particularly outlined below.
%
% INPUTS:
%   sys  - structure, containing the second-order system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | BalanceType     | character array, determining the type of          |
%   | {!}             | second-order balancing, only if OutputModel = 'so'|
%   |                 |   'p'   - position balancing                      |
%   |                 |   'pm'  - position balancing (diagonalized M)     |
%   |                 |   'pv'  - position-velocity balancing             |
%   |                 |   'vp'  - velocity-position balancing             |
%   |                 |   'vpm' - velocity-position balancing (diag. M)   |
%   |                 |   'v'   - velocity balancing                      |
%   |                 |   'fv'  - free velocity balancing                 |
%   |                 |   'so'  - second-order balancing                  |
%   |                 | (default 'so')                                    |
%   +-----------------+---------------------------------------------------+
%   | careopts        | structure, containing the optional parameters for |
%   |                 | the Riccati equation solver, see ml_caredl_sgn_fac|
%   |                 | and ml_care_nwt_fac                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Epsilon         | positive scalar, used in the case of a            |
%   |                 | non-full-rank D + D' term for epsilon             |
%   |                 | regularization by multiplying with an identity    |
%   |                 | matrix of appropriate size                        |
%   |                 | (default 1.0e-03)                                 |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the filter Gramian             |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the regulator Gramian          |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | lyapopts        | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time algebraic  |
%   |                 | Lyapunov equation, see ml_lyap_sgn_fac            |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the hsv,  |
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 |  'sum'       - using rel. tolerance for sum of hsv|
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | OutputModel     | character array, determining if classical or      |
%   |                 | structure-preserving balanced truncation is used  |
%   |                 |  'fo' - first-order BST, see ml_ct_d_dss_bst      |
%   |                 |         for optional parameters                   |
%   |                 |  'so' - second-order BST                          |
%   |                 | (default 'fo')                                    |
%   +-----------------+---------------------------------------------------+
%   | RiccatiSolver   | character array, determining the solver for the   |
%   |                 | dual Riccati equations                            |
%   |                 |  'newton' - Newton iteration                      |
%   |                 |  'sign'   - dual sign function method             |
%   |                 | (default 'sign')                                  |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | if 'tolerance' or 'sum' is set for                |
%   |                 | OrderComputation                                  |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order model, with the
%   {!}    following entries if opts.OutputModel = 'fo'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%          and the following entries if opts.OutputModel = 'so'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (5) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (6) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the filter Gramian, if         |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the regulator Gramian, if      |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed characteristic  |
%   | {!}             | stochastic values, for second-order balancing     |
%   |                 | the position and velocity characteristic          |
%   |                 | stochastic values are saved in a struct           |
%   +-----------------+---------------------------------------------------+
%   | infoCARE        | structure, containing information about the       |
%   |                 | sign function solver for the dual Riccati         |
%   |                 | equations, see ml_caredl_sgn_fac                  |
%   +-----------------+---------------------------------------------------+
%   | infoLYAP        | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation sovler for the  |
%   |                 | controllability Gramian, ml_lyap_sgn_fac          |
%   +-----------------+---------------------------------------------------+
%   | N {!}           | Dimension of the reduced-order model              |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_s_soss_bt, ml_ct_d_dss_bst, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_soss'
        % No extra action in main supported case.

    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        if size(sys.K, 1) <= 2500
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to full.' ...
                ' Use ml_ct_s_soss_bst to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse second-order system detected! ' ...
                ' Use ml_ct_s_soss_bst to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Enforce DAE handling.
if strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so') ...
        && strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so') ...
        && strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')
    if ml_field_set_to_value(opts, 'OutputModel') ...
            && strcmpi(opts.OutputModel, 'so')
        warning('MORLAB:data', ...
            ['Dense second-order descriptor systems cannot be reduced' ...
            ' structure-preserving. Changed opts.OutputModel to ''fo''.']);
    end

    opts.OutputModel = 'fo';
end

sys.SystemType = 'ct_d_soss';
sys            = ml_prepare_system_data(sys);

% Temporary variables.
n = size(sys.K, 1);
A = [zeros(n), eye(n); -sys.K, -sys.E];
B = sys.pB;
C = sys.pC;
D = sys.pD;
E = [eye(n), zeros(n); zeros(n), sys.M];

% Change to descriptor balanced truncation if necessary.
if not(ml_field_set_to_value(opts, 'OutputModel')) ...
        || (ml_field_set_to_value(opts, 'OutputModel') ...
        && strcmpi(opts.OutputModel, 'fo'))

    sysfo = struct( ...
        'A', A, ...
        'B', B, ...
        'C', C, ...
        'D', D, ...
        'E', E);

    [rom, info] = ml_ct_d_dss_bst(sysfo, opts);

    return;
elseif not(strcmpi(opts.OutputModel, 'so'))
    error('MORLAB:data', ...
        'Unknown value for opts.OutputModel.');
end

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'careopts')
    assert(isa(opts.careopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.careopts has to be a struct!');
else
    opts.careopts = struct();
end

if ml_field_set_to_value(opts, 'Epsilon')
    ml_assert_posscalar(opts.Epsilon, 'opts.Epsilon');
else
    opts.Epsilon = 1.0e-03;
end

if ml_field_set_to_value(opts, 'lyapopts')
    assert(isa(opts.lyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapopts has to be a struct!');
else
    opts.lyapopts = struct();
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 1;
    elseif strcmpi(opts.OrderComputation{k}, 'sum')
        rselect{k} = 2;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'RiccatiSolver')
    assert(strcmpi(opts.RiccatiSolver, 'newton') ...
        || strcmpi(opts.RiccatiSolver, 'sign'), ...
        'MORLAB:data', ...
        'The desired Riccati equation solver is not implemented!');
else
    opts.RiccatiSolver = 'sign';
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

m = size(B, 2);
p = size(C, 1);

% Transpose system, if there are more outputs than inputs.
if m < p
    A      = A';
    D      = D';
    E      = E';
    [B, C] = deal(C', B');
end

% Solve Lyapunov equation.
if (m >= p) && ml_field_set_to_value(opts, 'GramFacC')
    R             = opts.GramFacC;
    info.infoLYAP = [];
elseif (m < p) && ml_field_set_to_value(opts, 'GramFacO')
    R             = opts.GramFacO;
    info.infoLYAP = [];
else
    [R, infoLYAP] = ml_lyap_sgn_fac(A, B, E, opts.lyapopts);
    info.infoLYAP = infoLYAP;
end

% Solve Riccati equation.
if (m >= p) && ml_field_set_to_value(opts, 'GramFacO')
    L             = opts.GramFacO;
    info.infoCARE = [];
elseif (m < p) && ml_field_set_to_value(opts, 'GramFacC')
    L             = opts.GramFacC;
    info.infoCARE = [];
else
    % Epsilon regularization.
    if (rank(D) < min(m, p)) || (norm(D, 'fro') < 1.0e+02 * eps)
        D  = opts.Epsilon * eye(min(m, p));
        Bw = B * opts.Epsilon * eye(min(m, p), max(m, p))' + ...
            E * (R * (R' * C'));
    else
        Bw     = B * D' + E * (R * (R' * C'));
        [~, D] = qr(D', 0);
    end

    A = A - Bw * ((D' * D) \ C);
    B = Bw / D;
    C = D' \ C;

    if strcmpi(opts.RiccatiSolver, 'sign')
        opts.careopts.EqnType = 'primal';
        [L, ~, infoCARE]      = ml_caredl_sgn_fac(A, B, C, E, 1, ...
            opts.careopts);
        info.infoCARE         = infoCARE;
    else
        [L, infoCARE] = ml_pcare_nwt_fac(A, B, C, E, opts.careopts);
        info.infoCARE = infoCARE;
    end
end

% Swap Gramian factors if system was transposed.
if m < p
    [R, L] = deal(L, R);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[V, W, hsv] = ml_balproj_soss(sys, R, L, rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
rom = ml_projtrunc_soss(sys, V, W);

if not(isa(rom, 'cell')), rom = {rom}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
numRoms  = length(rom);
info.Hsv = hsv;
info.N   = cell(1, numRoms);
for k = 1:numRoms
    info.N{k} = size(rom{k}.M, 1);
end

if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

[rom, info] = ml_format_output(rom, 1, info);
