function [rom, info] = ml_ct_d_soss_brbt(sys, opts)
%ML_CT_D_SOSS_BRBT PRBT for dense, continuous second-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SOSS_BRBT(sys)
%   [rom, info] = ML_CT_D_SOSS_BRBT(sys, opts)
%
% DESCRIPTION:
%   This function computes the bounded-real balanced truncation for a
%   second-order system of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   Therefore, the two Riccati equations
%
%       A*P*E' + E*P*A' + B*B'
%           + (E*P*C' + B*D') * inv(Rb) * (E*P*C' + B*D')' = 0,
%       A'*Q*E + E'*Q*A + C'*C
%           + (B'*Q*E + D'*Cp)' * inv(Rc) * (B'*Q*E + D'*C) = 0,
%
%   are solved for the system Gramians P and Q, where
%
%           [ I  0 ]      [ 0   I ]      [ 0  ]
%       E = [      ], A = [       ], B = [    ], C = [Cp, Cv],
%           [ 0  M ]      [ -K -E ]      [ Bu ]
%
%   and
%
%       Rb = I - D*D'   and
%       Rc = I - D'*D,
%
%   As a result, a reduced model in either first-order
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   or second-order form
%
%       Mr*x''(t) = -Kr*x(t) -  Er*x'(t) + Bur*u(t)                     (5)
%            y(t) = Cpr*x(t) + Cvr*x'(t) +  Dr*u(t)                     (6)
%
%   is computed.
%
%   Note: In the structure-preserving version, only asymptotically stable
%         systems without algebraic constraints are allowed. In the
%         first-order version ml_ct_d_dss_brbt is used such that
%         additional optional arguments are available and the info struct
%         has more fields, which will not be particularly outlined below.
%
% INPUTS:
%   sys  - structure, containing the second-order system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | BalanceType     | character array, determining the type of          |
%   | {!}             | second-order balancing, only if OutputModel = 'so'|
%   |                 |   'p'   - position balancing                      |
%   |                 |   'pm'  - position balancing (diagonalized M)     |
%   |                 |   'pv'  - position-velocity balancing             |
%   |                 |   'vp'  - velocity-position balancing             |
%   |                 |   'vpm' - velocity-position balancing (diag. M)   |
%   |                 |   'v'   - velocity balancing                      |
%   |                 |   'fv'  - free velocity balancing                 |
%   |                 |   'so'  - second-order balancing                  |
%   |                 | (default 'so')                                    |
%   +-----------------+---------------------------------------------------+
%   | careopts        | structure, containing the optional parameters for |
%   |                 | the Riccati equation solver, see ml_caredl_sgn_fac|
%   |                 | and ml_care_nwt_fac                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the filter Gramian             |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the regulator Gramian          |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the hsv,  |
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 |  'sum'       - using rel. tolerance for sum of hsv|
%   |                 |                becomes error bound if             |
%   |                 |                OutputModel = 'fo'                 |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | OutputModel     | character array, determining if classical or      |
%   |                 | structure-preserving balanced truncation is used  |
%   |                 |  'fo' - first-order BRBT, see ml_ct_d_dss_brbt    |
%   |                 |         for optional parameters                   |
%   |                 |  'so' - second-order BRBT                         |
%   |                 | (default 'fo')                                    |
%   +-----------------+---------------------------------------------------+
%   | RiccatiSolver   | character array, determining the solver for the   |
%   |                 | dual Riccati equations                            |
%   |                 |  'newton' - Newton iteration                      |
%   |                 |  'sign'   - dual sign function method             |
%   |                 | (default 'sign')                                  |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | if 'tolerance' or 'sum' is set for                |
%   |                 | OrderComputation                                  |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order model, with the
%   {!}    following entries if opts.OutputModel = 'fo'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%          and the following entries if opts.OutputModel = 'so'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (5) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (6) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the filter Gramian, if         |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the regulator Gramian, if      |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed characteristic  |
%   | {!}             | bounded-real values, for second-order balancing   |
%   |                 | the position and velocity characteristic          |
%   |                 | bounded-real values are saved in a struct         |
%   +-----------------+---------------------------------------------------+
%   | infoCARE        | structure, containing information about the       |
%   |                 | sign function solver for the dual Riccati         |
%   |                 | equations, see ml_caredl_sgn_fac                  |
%   +-----------------+---------------------------------------------------+
%   | infoCAREC       | structure, containing information about the       |
%   |                 | Newton solver for the regulator Riccati equation, |
%   |                 | see ml_care_nwt_fac                               |
%   +-----------------+---------------------------------------------------+
%   | infoCAREO       | structure, containing information about the       |
%   |                 | Newton solver for the filter Riccati equation,    |
%   |                 | see ml_care_nwt_fac                               |
%   +-----------------+---------------------------------------------------+
%   | N {!}           | Dimension of the reduced-order model              |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_s_soss_bt, ml_ct_d_dss_brbt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_soss'
        % No extra action in main supported case.

    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        if size(sys.K, 1) <= 2500
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to full.' ...
                ' Use ml_ct_s_soss_brbt to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse second-order system detected! ' ...
                ' Use ml_ct_s_soss_brbt to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Enforce DAE handling.
if strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so') ...
        && strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so') ...
        && strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')
    if ml_field_set_to_value(opts, 'OutputModel') ...
            && strcmpi(opts.OutputModel, 'so')
        warning('MORLAB:data', ...
            ['Dense second-order descriptor systems cannot be reduced' ...
            ' structure-preserving. Changed opts.OutputModel to ''fo''.']);
    end

    opts.OutputModel = 'fo';
end

sys.SystemType = 'ct_d_soss';
sys            = ml_prepare_system_data(sys);

% Temporary variables.
n = size(sys.K, 1);
A = [zeros(n), eye(n); -sys.K, -sys.E];
B = sys.pB;
C = sys.pC;
D = sys.pD;
E = [eye(n), zeros(n); zeros(n), sys.M];

% Change to descriptor balanced truncation if necessary.
if not(ml_field_set_to_value(opts, 'OutputModel')) ...
        || (ml_field_set_to_value(opts, 'OutputModel') ...
        && strcmpi(opts.OutputModel, 'fo'))

    sysfo = struct( ...
        'A', A, ...
        'B', B, ...
        'C', C, ...
        'D', D, ...
        'E', E);

    [rom, info] = ml_ct_d_dss_brbt(sysfo, opts);

    return;
elseif not(strcmpi(opts.OutputModel, 'so'))
    error('MORLAB:data', ...
        'Unknown value for opts.OutputModel.');
end

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'careopts')
    assert(isa(opts.careopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.careopts has to be a struct!');
else
    opts.careopts = struct();
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 1;
    elseif strcmpi(opts.OrderComputation{k}, 'sum')
        rselect{k} = 2;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'RiccatiSolver')
    assert(strcmpi(opts.RiccatiSolver, 'newton') ...
        || strcmpi(opts.RiccatiSolver, 'sign'), ...
        'MORLAB:data', ...
        'The desired Riccati equation solver is not implemented!');
else
    opts.RiccatiSolver = 'sign';
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if not(isempty(D))
    Rb = eye(size(B, 2)) - D' * D;
    Lb = chol(Rb);
    Rc = eye(size(C, 1)) - D * D';
    Lc = chol(Rc);

    A = A + B * (D' * (Rc \ C));
    B = B / Lb;
    C = Lc' \ C;
end

% Computation of the full-rank factors of the Gramians.
if strcmpi(opts.RiccatiSolver, 'sign')
    if ml_field_set_to_value(opts, 'GramFacC') ...
            && ml_field_set_to_value(opts, 'GramFacO')
        % Both precomputed Gramian factors.
        R = opts.GramFacC;
        L = opts.GramFacO;
    elseif ml_field_set_to_value(opts, 'GramFacC')
        % Precomputed filter factor.
        opts.careopts.EqnType = 'primal';
        [L, ~, infoCARE]      = ml_caredl_sgn_fac(A, B, C, E, ...
            1, opts.careopts);
        R                     = opts.GramFacC;
        info.infoCARE         = infoCARE;
    elseif ml_field_set_to_value(opts, 'GramFacO')
        % Precomputed regulator factor.
        opts.careopts.EqnType = 'dual';
        [~, R, infoCARE]      = ml_caredl_sgn_fac(A, B, C, E, ...
            1, opts.careopts);
        L                     = opts.GramFacO;
        info.infoCARE         = infoCARE;
    else
        % No Gramian factors precomputed.
        opts.careopts.EqnType = 'both';
        [L, R, infoCARE]      = ml_caredl_sgn_fac(A, B, C, E, ...
            1, opts.careopts);
        info.infoCARE         = infoCARE;
    end
else
    if ml_field_set_to_value(opts, 'GramFacC') ...
            && ml_field_set_to_value(opts, 'GramFacO')
        % Both precomputed Gramian factors.
        R = opts.GramFacC;
        L = opts.GramFacO;
    elseif ml_field_set_to_value(opts, 'GramFacC')
        % Precomputed filter factor.
        [L, infoCAREO] = ml_pcare_nwt_fac(A, B, C, E, opts.careopts);
        info.infoCAREO = infoCAREO;

        R = opts.GramFacC;
    elseif ml_field_set_to_value(opts, 'GramFacO')
        % Precomputed regulator factor.
        [R, infoCAREC] = ml_pcare_nwt_fac(A', C', B', E', opts.careopts);
        info.infoCAREC = infoCAREC;

        L = opts.GramFacO;
    else
        % No Gramian factors precomputed.
        [R, infoCAREC] = ml_pcare_nwt_fac(A', C', B', E', opts.careopts);
        info.infoCAREC = infoCAREC;

        [L, infoCAREO] = ml_pcare_nwt_fac(A, B, C, E, opts.careopts);
        info.infoCAREO = infoCAREO;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[V, W, hsv] = ml_balproj_soss(sys, R, L, rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
rom = ml_projtrunc_soss(sys, V, W);

if not(isa(rom, 'cell')), rom = {rom}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
numRoms  = length(rom);
info.Hsv = hsv;
info.N   = cell(1, numRoms);
for k = 1:numRoms
    info.N{k} = size(rom{k}.M, 1);
end

if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

[rom, info] = ml_format_output(rom, 1, info);
