function [rom, info] = ml_ct_d_dss_hna(sys, opts)
%ML_CT_D_DSS_HNA Hankel-norm approximation for dense, descriptor systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_DSS_HNA(sys)
%   [rom, info] = ML_CT_D_DSS_HNA(sys, opts)
%
% DESCRIPTION:
%   This function computes the generalized Hankel-norm approximation of a
%   descriptor system
%
%       E*x'(t) = A*x(t) + B*u(t),                                      (1)
%          y(t) = C*x(t) + D*u(t).                                      (2)
%
%   Therefore, first a balanced realization is computed by using the
%   generalized balanced truncation square-root method with an appropriate
%   tolerance for the minimal realization of the given system. Then the
%   strictly proper part of the system is transformed using the formulas
%   for all-pass systems. As result, a reduced-order system of the form
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   is computed, such that for the original transfer function G and the
%   reduced-order transfer function Gr with an r-th order strictly proper
%   part it holds
%
%       ||G - Gr||_{H}       = Hsvp(r+1),
%       ||G - Gr||_{\infty} <= 2 * (Hsvp(r+1) + ... + Hsvp(n)),
%
%   with Hsvp, a vector containing the proper Hankel singular values of the
%   system.
%
%   Note: For unstable systems, an additional additive decomposition into
%         the stable and anti-stable parts is performed and then only the
%         stable part will be reduced. That does not change the error
%         formulas.
%
% INPUTS:
%   sys  - structure or state-space object, containing the descriptor
%          system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | DecompEig       | positive scalar, overestimation of the absolute   |
%   |                 | value of the largest finite eigenvalue of s*E - A,|
%   |                 | if set, replaces the computation with DecompTol   |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | DecompTol       | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of E to determine the      |
%   |                 | smallest non-quasi-zero singular value of E       |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | gdlyapopts      | structure, containing the optional parameters for |
%   |                 | the computation of the generalized discrete-time  |
%   |                 | Lyapunov equations, see ml_gdlyapdl_smith_fac     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | hankeldecopts   | structure, containing the optional parameters for |
%   |                 | the disk function used for the decomposition      |
%   |                 | after the transformation to an all-pass system    |
%   |                 | see ml_disk and ml_getqz                          |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the observability improper     |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | ImproperTrunc   | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest proper Hankel singular value of the       |
%   |                 | system to truncate the improper part, if 0 no     |
%   |                 | improper balanced truncation is performed         |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Index           | nonnegative integer, index of the descriptor      |
%   |                 | system used to set an upper bound on the size of  |
%   |                 | the reduced improper part, Inf if unknown         |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | infdecopts      | structure, containing the optional parameters for |
%   |                 | the decomposition of the finite and infinite parts|
%   |                 | of the system using the disk function and subspace|
%   |                 | extraction method, see ml_disk and ml_getqz       |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | lyapopts        | structure, containing the optional parameters for |
%   |                 | the computation of the generalized continuous-time|
%   |                 | Lyapunov equations, see ml_lyapdl_sgn_fac         |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | MinRelTol       | nonnegative scalar, tolerance multiplied with the |
%   | {!}             | largest characteristic value to determine a       |
%   |                 | minimal realization                               |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsvp)) + Nu + Ni)          |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using absolute error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | stabdecopts     | structure, containing the optional parameters for |
%   |                 | the decomposition of the stable and unstable parts|
%   |                 | of the system using the sign function and subspace|
%   |                 | extraction method, see ml_signm and ml_getqz      |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   rom  - structure or state-space object, containing the reduced-order
%          descriptor system:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | reduced-order model in H-infinity norm            |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsvi            | a vector, containing the computed Hankel singular |
%   |                 | values of the improper part of the system         |
%   +-----------------+---------------------------------------------------+
%   | Hsvp            | a vector, containing the computed Hankel singular |
%   |                 | values of the proper part of the system           |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the improper observability     |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | infinite, finite stable and finite anti-stable    |
%   |                 | parts, see ml_ct_dss_adtf                         |
%   +-----------------+---------------------------------------------------+
%   | infoGDLYAP      | structure, containing information about the       |
%   |                 | generalized discrete-time Lyapunov equation solver|
%   |                 | for the improper Gramians,                        |
%   |                 | see ml_gdlyapdl_smith_fac                         |
%   +-----------------+---------------------------------------------------+
%   | infoHAADTF      | structure, containing information about the       |
%   |                 | disk function method, see ml_ct_dss_adtf          |
%   +-----------------+---------------------------------------------------+
%   | infoLYAP        | structure, containing information about the       |
%   |                 | continuous-time dual Lyapunov equations solver,   |
%   |                 | see ml_lyapdl_sgn_fac                             |
%   +-----------------+---------------------------------------------------+
%   | Ni              | Dimension of the improper part in the reduced-    |
%   | {!}             | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | Np              | Dimension of the proper part in the reduced-order |
%   | {!}             | model                                             |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Dimension of the unstable part in the reduced-    |
%   |                 | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | Sigma           | Chosen proper Hankel singular value, exact        |
%   | {!}             | approximation error in the Hankel-norm            |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   S. Werner, Hankel-norm approximation of descriptor systems, Master's
%   thesis, Otto von Guericke University, Magdeburg, Germany (2016).
%   http://nbn-resolving.de/urn:nbn:de:gbv:ma9:1-8845
%
% See also ml_ct_ss_hna, ml_ct_d_dss_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        warning('MORLAB:data', ...
            ['No descriptor matrix E found, set to identity E = I_n.' ...
            ' Use ml_ct_d_ss_hna to handle standard systems.']);

        sys.E = eye(size(sys.A));

    case 'ct_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['No descriptor matrix E found, set to identity.', ...
                ' System matrices were converted from sparse to' ...
                ' full. Use ml_ct_s_foss_hna to handle sparse systems.']);
            sys.E = eye(size(sys.A));
        else
            error('MORLAB:data', ...
                ['Large-scale sparse standard system detected.' ...
                ' Use ml_ct_s_foss_hna to handle such systems.']);
        end

    case 'ct_d_dss'
        % No extra action in main supported case.

    case {'ct_s_dss_default', 'ct_s_dss_dae_1', 'ct_s_dss_dae_2'}
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to' ...
                ' full. Use ml_ct_s_foss_hna to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse descriptor system detected.' ...
                ' Use ml_ct_s_foss_hna to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'ct_d_dss';
sys            = ml_prepare_system_data(sys);

% Fill the D matrix.
if isempty(sys.D)
    sys.D = zeros(size(sys.C, 1), size(sys.B, 2));
end

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'gdlyapopts')
    assert(isa(opts.gdlyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.gdlyapopts has to be a struct!');
else
    opts.gdlyapopts = struct();
end

if ml_field_set_to_value(opts, 'hankeldecopts')
    assert(isa(opts.hankeldecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.hankeldecopts has to be a struct!');
else
    opts.hankeldecopts = struct();
end

if ml_field_set_to_value(opts, 'lyapopts')
    assert(isa(opts.lyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapopts has to be a struct!');
else
    opts.lyapopts = struct();
end

opts = ml_check_cell_param(opts, 'MinRelTol', ...
    @ml_assert_nonnegscalar, log(size(sys.A, 1)) * eps);

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 3;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION OF TRANSFER FUNCTION.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition of the system into unstable, finite and infinite parts.
[sys, infoADTF] = ml_ct_d_dss_adtf(sys, opts);

ninf = size(sys.Ainf, 1);

% Assign information about additive decomposition.
info.infoADTF = infoADTF;


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE MINIMAL, BALANCED REALIZATION.                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation of the full-rank factors of the proper Gramians.
if size(sys.A, 1) > 0
    if ml_field_set_to_value(opts, 'GramFacC') ...
            && ml_field_set_to_value(opts, 'GramFacO')
        % Both precomputed Gramian factors.
        R = opts.GramFacC;
        L = opts.GramFacO;
    elseif ml_field_set_to_value(opts, 'GramFacC')
        % Precomputed controllability factor.
        [L, infoLYAP] = ml_lyap_sgn_fac(sys.A', sys.C', sys.E', ...
            opts.lyapopts);
        R             = opts.GramFacC;
        info.infoLYAP = infoLYAP;
    elseif ml_field_set_to_value(opts, 'GramFacO')
        % Precomputed observability factor.
        [R, infoLYAP] = ml_lyap_sgn_fac(sys.A, sys.B, sys.E, ...
            opts.lyapopts);
        L             = opts.GramFacO;
        info.infoLYAP = infoLYAP;
    else
        % No Gramian factors precomputed.
        [R, L, infoLYAP] = ml_lyapdl_sgn_fac(sys.A, sys.B, sys.C, ...
            sys.E, opts.lyapopts);
        info.infoLYAP    = infoLYAP;
    end
else
    [R, L] = deal([]);
end

% Computation of the full-rank factors of the improper Gramians.
if ninf > 0
    if ml_field_set_to_value(opts, 'IGramFacC') ...
            && ml_field_set_to_value(opts, 'IGramFacO')
        % Both precomputed Gramian factors.
        Rinf = opts.IGramFacC;
        Linf = opts.IGramFacO;
    elseif ml_field_set_to_value(opts, 'IGramFacC')
        % Precomputed controllability factor.
        [Linf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf', ...
            sys.Cinf', sys.Einf', opts.gdlyapopts);
        Rinf               = opts.IGramFacC;
        info.infoGDLYAP    = infoGDLYAP;
    elseif ml_field_set_to_value(opts, 'IGramFacO')
        % Precomputed observability factor.
        [Rinf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf, ...
            sys.Binf, sys.Einf, opts.gdlyapopts);
        Linf               = opts.IGramFacO;
        info.infoGDLYAP    = infoGDLYAP;
    else
        % No Gramian factors precomputed.
        [Rinf, Linf, infoGDLYAP] = ml_gdlyapdl_smith_fac( ...
            sys.Ainf, sys.Binf, sys.Cinf, sys.Einf, opts.gdlyapopts);
        info.infoGDLYAP          = infoGDLYAP;
    end
else
    [Rinf, Linf] = deal([]);
end

if ml_field_set_to_value(opts, 'Tolerance')
    tmp = opts.Tolerance;
else
    tmp = [];
end
opts.Tolerance = opts.MinRelTol;
opts.Method    = 'sr';
[V, W, hsvp]   = ml_balproj_proper(sys, R, L, 0, 1, opts);
opts.Tolerance = tmp;

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

minsys = ml_projtrunc_proper(sys, V, W);

if not(isa(minsys, 'cell')), minsys = {minsys}; end

for k = 1:length(minsys)
    minsys{k}.Au = sys.Au;
    minsys{k}.Bu = sys.Bu;
    minsys{k}.Cu = sys.Cu;
    minsys{k}.Eu = sys.Eu;
end

nh = ml_order(hsvp, size(minsys{1}.Au, 1), rselect, opts);

if not(isa(nh, 'cell')), nh = {nh}; end

[sysFull, nh] = ml_extend_cell(minsys, nh);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% RECUCE IMPROPER SYSTEM PART.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

opts.Method        = 'bfsr';
[Vinf, Winf, hsvi] = ml_balproj_improper(sys, Rinf, Linf, opts);
infrom             = ml_projtrunc_improper(sys, Vinf, Winf);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% TRANSFORMATION TO REDUCED-ORDER MODEL.                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

numRoms   = length(nh);
rom       = cell(1, numRoms);
sigmaFull = cell(1, numRoms);
nhFull    = nh;

for r = 1:numRoms
    nh  = nhFull{r};
    sys = sysFull{r};
    n   = size(sys.A, 1);

    if nh < n
        sigmaFull{r} = hsvp(nh + 1);
        sigma        = sigmaFull{r};

        % Get multiplicity of the chosen Hankel singular value.
        k = 1;

        while k < n - nh
            if abs(sigma - hsvp(nh + 1 + k)) > sigma * log(n) * sqrt(eps)
                break;
            end

            k = k + 1;
        end

        % Permutation of the system.
        if nh + k < n
            p = [1:nh, nh+k+1:n, nh+1:nh+k];

            sys.A = sys.A(p,p);
            sys.B = sys.B(p, :);
            sys.C = sys.C(: , p);
            hsvh  = hsvp(p);
        else
            hsvh = hsvp(1:n);
        end

        % Transformation to an all-pass error system.
        nk = n - k;

        U = sigma * pinv(sys.C(: , nk+1:n)') * sys.B(nk+1:n, :);
        S = sparse(1:nk, 1:nk, hsvh(1:nk));
        A = sys.A(1:nk, 1:nk);
        B = sys.B(1:nk, :);
        C = sys.C(: , 1:nk);

        model = struct( ...
            'A', sigma^2 * A' + S * A * S + C' * U * B', ...
            'B', S * B - C' * U, ...
            'C', C * S - U * B', ...
            'D', sys.D + U, ...
            'E', diag(hsvh(1:nk).^2 - sigma^2));

        % Additive decomposition.
        hankelopts                       = struct();
        hankelopts.infdecopts.Dimension  = 0;
        hankelopts.stabdecopts           = opts.hankeldecopts;
        hankelopts.stabdecopts.Dimension = nk - nh;


        [model, infoHANKEL] = ml_ct_d_dss_adtf(model, hankelopts);

        % Assign information about additive decomposition.
        info.infoHAADTF = infoHANKEL;
    else
        sigmaFull{r} = 0;
        model        = sys;
    end

    rom{r} = struct( ...
        'A', blkdiag(model.A, sysFull{k}.Au, infrom.A), ...
        'B', [model.B; sysFull{k}.Bu; infrom.B], ...
        'C', [model.C, sysFull{k}.Cu, infrom.C], ...
        'D', model.D, ...
        'E', blkdiag(model.E, sysFull{k}.Eu, infrom.E));
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
absErrBound = cell(1, numRoms);
for k = 1:numRoms
    absErrBound{k} = 2 * sum(hsvp(nhFull{k}+1:end));
end

info.AbsErrBound = absErrBound;
info.Hsvi        = hsvi;
info.Hsvp        = hsvp;
info.Ni          = size(infrom.A, 1);
info.Np          = nhFull;
info.Nu          = size(sysFull{1}.Au, 1);
info.Sigma       = sigmaFull;

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC  = R;
    info.GramFacO  = L;
    info.IGramFacC = Rinf;
    info.IGramFacO = Linf;
else
    info.GramFacC  = [];
    info.GramFacO  = [];
    info.IGramFacC = [];
    info.IGramFacO = [];
end

[rom, info] = ml_format_output(rom, 1, info);
