function [t, y, info] = ml_dt_dss_simulate(varargin)
%ML_DT_DSS_SIMULATE Discrete-time simulation of descriptor systems.
%
% SYNTAX:
%   [t, y, info] = ML_DT_DSS_SIMULATE(sys1, ..., sysN)
%   [t, y, info] = ML_DT_DSS_SIMULATE(sys1, ..., sysN, opts)
%
%   [t, y, info] = ML_DT_DSS_SIMULATE(sys, rom1, ..., romM)
%   [t, y, info] = ML_DT_DSS_SIMULATE(sys, rom1, ..., romM, opts)
%
% DESCRIPTION:
%   This function can be used to simulate first-order discrete-time
%   dynamical systems of the form
%
%       E*x(t+1) = A*x(t) + B*u(t),
%           y(t) = C*x(t) + D*u(t),
%
%   where E is assumed to be invertible. If no input function u is given,
%   the unit step response is computed.
%
% INPUTS:
%   sys1, ..., sysN - struct, containing first-order descriptor systems
%   {!}
%   rom1, ..., romM - struct, containing first-order descriptor systems
%   {!}
%   opts            - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | DiffMode        | character array, determining type of plot         |
%   |                 |  'off' - time response plots of sys1, sys2, ...   |
%   |                 |  'abs' - absolute errors sys-rom1, sys-rom2, ...  |
%   |                 |  'rel' - relative errors sys-rom1, sys-rom2, ...  |
%   |                 |  'all' - all plot types are shown                 |
%   |                 | (default 'off')                                   |
%   +-----------------+---------------------------------------------------+
%   | Info            | {0, 1}, used to disable/enable display of         |
%   |                 | sampling steps                                    |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | InputFcn        | function handle, input function u(t) which takes a|
%   |                 | time argument and returns an input vector of      |
%   |                 | appropriate size                                  |
%   |                 | (default ones(m, 1))                              |
%   +-----------------+---------------------------------------------------+
%   | InputRange      | vector of length <= 2, time range in which the    |
%   |                 | input function u(t) is applied to the system,     |
%   |                 | otherwise u = 0, if only a scalar tu0 is given,   |
%   |                 | the input time interval is taken to be [tu0, tf]  |
%   |                 | (default opts.TimeRange)                          |
%   +-----------------+---------------------------------------------------+
%   | LineSpecs       | character array, determining the line             |
%   | {!}             | specifications for the plots                      |
%   |                 | (default '-')                                     |
%   +-----------------+---------------------------------------------------+
%   | ShowPlot        | {0, 1}, used to disable/enable showing of plots   |
%   |                 | (default 1)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreStates     | {0, 1}, used to disable/enable storing the        |
%   |                 | computed system states additionally               |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | TimeRange       | nonnegative vector of length <= 2, time range in  |
%   |                 | which the system is simulated, if only a scalar tf|
%   |                 | is given, time range is set to be [0, tf]         |
%   |                 | (default [0 10])                                  |
%   +-----------------+---------------------------------------------------+
%   | X0              | vector, initial states of x(t)                    |
%   | {!}             | (default zeros(n, 1))                             |
%   +-----------------+---------------------------------------------------+
%
%   Note: Arguments/Parameters marked with {!} may also be a cell array
%         containing multiple arguments.
%
% OUTPUTS:
%   t    - vector of size (number samples), conatining the time sampling
%          points
%   y    - 3-D array of size p x (number samples) x (number systems),
%          containing the outputs of the systems at the time points t
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErr          | if opts.DiffMode == 'off' empty, otherwise matrix |
%   |                 | of size p x (number samples) x (number systems-1),|
%   |                 | containing the computed absolute error values     |
%   +-----------------+---------------------------------------------------+
%   | Npts            | positive integer, number of plotted points        |
%   +-----------------+---------------------------------------------------+
%   | RelErr          | if opts.DiffMode == 'off' empty, otherwise matrix |
%   |                 | of size p x (number samples) x (number systems-1),|
%   |                 | containing the computed relative error values     |
%   +-----------------+---------------------------------------------------+
%   | X               | if opts.StoreStates == 0 empty, otherwise cell    |
%   |                 | array of length (number systems) containing       |
%   |                 | matrices of size n x (number samples) with the    |
%   |                 | computed states                                   |
%   +-----------------+---------------------------------------------------+
%
% See also ml_dt_ss_simulate, ml_ct_dss_simulate_ie.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Check all input systems.
systems  = cell(1);
iosizes  = cell(1);
orders   = cell(1);
tsamples = cell(1);
numSys   = 0;
hasOpts  = 0;

for k = 1:nargin()
    sysInput = varargin{k};

    if isa(sysInput, 'cell')
        assert(isvector(sysInput), ...
            'MORLAB:data', ...
            'The cell inputs must be a vector!');
        sys_length = length(sysInput);
    else
        sys_length = 1;
    end

    for j = 1:sys_length
        if isa(sysInput, 'cell')
            sys = sysInput{j};
        else
            sys = sysInput;
        end

        if isa(sys, 'struct') && isfield(sys, 'A')
            % First-order system case (struct).
            numSys = numSys + 1;

            ml_assert_stdsys(sys);

            if ml_field_set_to_value(sys, 'D')
                assert(isequal(size(sys.D), [size(sys.C, 1), ...
                    size(sys.B, 2)]), ...
                    'MORLAB:data', ...
                    'The matrix D must have the dimensions %d x %d!', ...
                    size(sys.C, 1), size(sys.B, 2));
            else
                sys.D = zeros(size(sys.C, 1), size(sys.B, 2));
            end

            if ml_field_set_to_value(sys, 'E')
                assert(isequal(size(sys.E), size(sys.A)), ...
                    'MORLAB:data', ...
                    'The matrix E must have the same dimensions as A!');
            else
                if issparse(sys.A)
                    sys.E = speye(size(sys.A));
                else
                    sys.E = eye(size(sys.A));
                end
            end

            if ml_field_set_to_value(sys, 'Ts')
                ml_assert_scalar(sys.Ts, 'sys.Ts');
                assert(not(sys.Ts == 0), ...
                    'MORLAB:data', ...
                    ['The sample time for discrete-time systems ' ...
                    'cannot be 0!']);

                if sys.Ts < 0
                    sys.Ts = 1;
                end
            else
                sys.Ts = 1;
            end

            tsamples{numSys} = sys.Ts;
            systems{numSys}  = sys;
            orders{numSys}   = size(sys.A, 1);
            iosizes{numSys}  = size(sys.D);
        elseif isa(sys, 'ss')
            % First-order system case (state-space).
            numSys = numSys + 1;

            if isempty(sys.e)
                sys.e = eye(size(sys.a));
            end

            assert(not(sys.Ts == 0), ...
                'MORLAB:data', ...
                ['The sample time for discrete-time systems ' ...
                'cannot be 0!']);

            if sys.Ts < 0
                sys.Ts = 1;
            end

            tsamples{numSys} = sys.Ts;
            systems{numSys}  = sys;
            orders{numSys}   = size(sys.A, 1);
            iosizes{numSys}  = size(sys.D);
        elseif isa(sys, 'sparss')
            % First-order system case (sparse state-space).
            numSys = numSys + 1;

            if isempty(sys.e)
                sys.e = speye(size(sys.a, 1));
            end

            assert(not(sys.Ts == 0), ...
                'MORLAB:data', ...
                ['The sample time for discrete-time systems ' ...
                'cannot be 0!']);

            if sys.Ts < 0
                sys.Ts = 1;
            end

            tsamples{numSys} = sys.Ts;
            systems{numSys}  = sys;
            orders{numSys}   = size(sys.A, 1);
            iosizes{numSys}  = size(sys.D);
        elseif k < nargin()
            % Unsupported input.
            if isa(sys, 'struct')
                error('MORLAB:data', ...
                    'The given system structure is not supported!');
            else
                error('MORLAB:data', ...
                    'The systems must be structs or ss objects!');
            end
        else
            % Wrong format of option structure.
            assert(isa(sys, 'struct'), ...
                'MORLAB:data', ...
                'The input argument opts must be a structure!');

            hasOpts = 1;
        end
    end
end

% Check minimum number of given systems.
assert(numSys > 0, ...
    'MORLAB:data', ...
    'Give at least one system for plotting!');

% Check number of inputs and outputs.
assert(all(cellfun(@(s) isequal(iosizes{1}, s), iosizes)), ...
    'MORLAB:data', ...
    'Number of inputs and outputs have to be the same for all systems!');

% Get last input as possible option struct if not a system.
if hasOpts
    opts = varargin{end};
else
    opts = struct();
end

% Check and set optional parameters.
if ml_field_set_to_value(opts, 'DiffMode')
    ml_assert_char(opts.DiffMode, 'opts.DiffMode')
    assert(strcmpi(opts.DiffMode, 'off') ...
        || strcmpi(opts.DiffMode, 'abs') ...
        || strcmpi(opts.DiffMode, 'rel') ...
        || strcmpi(opts.DiffMode, 'all'), ...
        'MORLAB:data', ...
        'The requested difference mode is not implemented!');

    if (numSys == 1) && not(strcmpi(opts.DiffMode, 'off'))
        warning('MORLAB:data', ...
            ['The difference mode needs at least 2 systems.\n' ...
            'Difference mode is turned off!']);
        opts.DiffMode = 'off';
    end
else
    opts.DiffMode = 'off';
end

if ml_field_set_to_value(opts, 'Info')
    ml_assert_boolean(opts.Info, 'opts.Info');
else
    opts.Info = false;
end

opts         = ml_check_cell_param(opts, 'LineSpecs', ...
    @ml_assert_char, '-');
numLineSpecs = length(opts.LineSpecs);

if ml_field_set_to_value(opts, 'ShowPlot')
    ml_assert_boolean(opts.ShowPlot, 'opts.ShowPlot');
else
    opts.ShowPlot = true;
end

if ml_field_set_to_value(opts, 'StoreStates')
    ml_assert_boolean(opts.StoreStates, 'opts.StoreStates');
else
    opts.StoreStates = false;
end

if ml_field_set_to_value(opts, 'TimeRange')
    ml_assert_nonnegvector(opts.TimeRange, 'opts.TimeRange');
    assert(length(opts.TimeRange) <= 2, ...
        'MORLAB:data', ...
        'The parameter opts.TimeRange must be a vector of length <= 2!');

    if length(opts.TimeRange) == 1
        opts.TimeRange = [0, opts.TimeRange];
    end
else
    opts.TimeRange = [0, 10];
end

if ml_field_set_to_value(opts, 'InputFcn')
    assert(isa(opts.InputFcn, 'function_handle'), ...
        'MORLAB:data', ...
        'The parameter opts.InputFcn has to be a function handle!');

    tmp = opts.InputFcn(opts.TimeRange(1));
    assert(isequal(size(tmp), [iosizes{1}(2), 1]), ...
        'MORLAB:data', ...
        'The input function has to generate vectors of length %d!', ...
        iosizes{1}(2));
else
    opts.InputFcn = @(t) ones(iosizes{1}(2), 1);
end

if ml_field_set_to_value(opts, 'InputRange')
    ml_assert_vector(opts.InputRange, 'opts.InputRange');
    assert(length(opts.InputRange) <= 2, ...
        'MORLAB:data', ...
        'The parameter opts.InputRange must be a vector of length <= 2!');

    if length(opts.InputRange) == 1
        opts.InputRange = [opts.InputRange, opts.TimeRange(2)];
    end
else
    opts.InputRange = opts.TimeRange;
end

opts = ml_check_cell_vector(opts, 'X0', numSys, orders, @zeros);

% Check sample times.
assert(all(cellfun(@(s) isequal(tsamples{1}, s), tsamples)) || ...
    strcmpi(opts.DiffMode, 'off'), ...
    'MORLAB:data', ...
    'Sample times need to be the same for error computation!');


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INITIALIZATION.                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Precompute LU decomposition.
spSolves = zeros(1, numSys);
L        = cell(1, numSys);
U        = cell(1, numSys);
p        = cell(1, numSys);
q        = cell(1, numSys);
R        = cell(1, numSys);

for j = 1:numSys
    spSolves(j) = issparse(systems{j}.E);

    if spSolves(j)
        [L{j}, U{j}, p{j}, q{j}, R{j}] = lu(systems{j}.E, 'vector');
    else
        [L{j}, U{j}, p{j}] = lu(systems{j}.E, 'vector');
    end
end

% Determine minimum length of time grid.
nt = ceil(max(cellfun(@(s) abs(opts.TimeRange(2) - opts.TimeRange(1)) ...
    / s, tsamples)));

% Initial states and outputs.
u = opts.InputFcn;
t = zeros(nt + 1, numSys);
y = zeros(iosizes{1}(1), nt + 1, numSys);

if opts.StoreStates
    X = cell(1, numSys);

    for j = 1:numSys
        X{j} = zeros(orders{j}, nt + 1);
    end
end

if not(strcmpi(opts.DiffMode, 'off'))
    abserr = zeros(iosizes{1}(1), nt + 1, numSys - 1);
    relerr = zeros(iosizes{1}(1), nt + 1, numSys - 1);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% IMPLICIT EULER INTEGRATION SCHEME.                                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for j = 1:numSys
    if opts.Info
        fprintf(1, 'Simulate System %2d\n', j);
        fprintf(1, '===================\n');
    end

    x          = opts.X0{j};
    y(:, 1, j) = systems{j}.C * x;
    t(1, j)    = opts.TimeRange(1);

    if (t(1, j) >= min(opts.InputRange)) ...
            && (t(1, j) <= max(opts.InputRange))
        y(:, 1, j) = y(:, 1, j) + systems{j}.D * u(t(1));
    end

    if opts.StoreStates
        X{j}(:, 1) = x;
    end

    for k = 1:nt
        if opts.Info
            fprintf(1, 'SIMULATE Step %4d / %d\n', k, nt);
        end

        t(k+1, j) = t(k, j) + tsamples{j};
        x         = systems{j}.A * x;

        if (t(k, j) >= min(opts.InputRange)) ...
                && (t(k, j) <= max(opts.InputRange))
            x = x + systems{j}.B * u(t(k, j));
        end

        if spSolves(j)
            x(q{j}, :) = U{j} \ (L{j} \ (R{j}(:, p{j}) \ x));
        else
            x = U{j} \ (L{j} \ x(p{j}));
        end

        y(:, k+1, j) = systems{j}.C * x;
        if (t(k+1, j) >= min(opts.InputRange)) ...
                && (t(k+1, j) <= max(opts.InputRange))
            y(:, k+1, j) = y(:, k+1, j) + systems{j}.D * u(t(k+1, j));
        end

        if opts.StoreStates
            X{j}(:, k+1) = x;
        end
    end

    if opts.Info && (j < numSys)
        fprintf(1, '\n');
    end
end

% Compute errors if requested.
if not(strcmpi(opts.DiffMode, 'off'))
    maxY            = max(abs(y(:, :, 1)), [], 2);
    maxY(maxY == 0) = eps;

    for j = 2:numSys
        abserr(:, :, j-1) = abs(y(:, :, 1) - y(:, :, j));

        for kp = 1:iosizes{1}(1)
            relerr(kp, :, j-1) = abserr(kp, :, j-1) / maxY(kp);
        end
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CONSTRUCTION OF PLOTS.                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Save hold status for figure window.
if opts.ShowPlot
    holdStatus = ishold();
else
    holdStatus = 1;
end

% Set line specifications for plots.
lineSpecs = cell(1, numSys);
for j = 1:numSys
    if numLineSpecs == 1
        lineSpecs{j} = opts.LineSpecs{1};
    elseif j <= numLineSpecs
        lineSpecs{j} = opts.LineSpecs{j};
    else
        lineSpecs{j} = '-';
    end
end

% Correct color choices for 'all' plots by setting first one black.
if strcmpi(opts.DiffMode, 'all')
    setColor  = zeros(1, numSys);

    for colors = {'r', 'g', 'b', 'c', 'm', 'y', 'k', 'w'}
        setColor = setColor + cellfun(@(s) ismember(colors{1}, s), ...
            opts.LineSpecs);
    end

    if not(any(setColor))
        lineSpecs{1} = [lineSpecs{1} 'k'];
    end
end

% Plot time simulation.
if opts.ShowPlot
    switch lower(opts.DiffMode)
        case 'off'
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys
                    stairs(t(:, j), y(k, :, j)', lineSpecs{j});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end

                if abs(opts.TimeRange(1) - opts.TimeRange(2)) > 0
                    xlim(opts.TimeRange);
                end
            end

        case 'abs'
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys-1
                    stairs(t(:, j), abserr(k, :, j)', lineSpecs{j+1});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

        case 'rel'
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys-1
                    stairs(t(:, j), relerr(k, :, j)', lineSpecs{j+1});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

        case 'all'
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys
                    stairs(t(:, j), y(k, :, j)', lineSpecs{j});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

            figure();
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys-1
                    stairs(t(:, j), abserr(k, :, j)', lineSpecs{j+1});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

            figure();
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys-1
                    stairs(t(:, j), relerr(k, :, j)', lineSpecs{j+1});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

    end
end

% Turn hold status to normal.
if not(holdStatus)
    hold off;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT INFORMATION.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if strcmpi(opts.DiffMode, 'off')
    [abserr, relerr] = deal([]);
end

if not(opts.StoreStates)
    X = [];
end

info = struct( ...
    'AbsErr', abserr, ...
    'Npts'  , nt, ...
    'RelErr', relerr, ...
    'X'     , {X});
