function [t, y, info] = ml_ct_soss_simulate_ie(varargin)
%ML_CT_SOSS_SIMULATE_IE Implicit Euler second-order integrator.
%
% SYNTAX:
%   [t, y, info] = ML_CT_SOSS_SIMULATE_IE(sys1, ..., sysN)
%   [t, y, info] = ML_CT_SOSS_SIMULATE_IE(sys1, ..., sysN, opts)
%
%   [t, y, info] = ML_CT_SOSS_SIMULATE_IE(sys, rom1, ..., romM)
%   [t, y, info] = ML_CT_SOSS_SIMULATE_IE(sys, rom1, ..., romM, opts)
%
% DESCRIPTION:
%   This function can be used to simulate second-order dynamical systems
%   of the form
%
%       M*x''(t) + E*x'(t) + K*x(t) = Bu*u(t),
%                              y(t) = Cp*x(t) + Cv*x'(t) + D*u(t),
%
%   where the second-order implicit Euler method is used.
%   If no input function u is given, the unit step response is computed.
%
% INPUTS:
%   sys1, ..., sysN - struct, containing second-order systems
%   {!}
%   rom1, ..., romM - struct, containing second-order systems
%   {!}
%   opts            - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | DiffMode        | character array, determining type of plot         |
%   |                 |  'off' - time response plots of sys1, sys2, ...   |
%   |                 |  'abs' - absolute errors sys-rom1, sys-rom2, ...  |
%   |                 |  'rel' - relative errors sys-rom1, sys-rom2, ...  |
%   |                 |  'all' - all plot types are shown                 |
%   |                 | (default 'off')                                   |
%   +-----------------+---------------------------------------------------+
%   | Info            | {0, 1}, used to disable/enable display of         |
%   |                 | sampling steps                                    |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | InputFcn        | function handle, input function u(t) which takes a|
%   |                 | time argument and returns an input vector of      |
%   |                 | appropriate size                                  |
%   |                 | (default ones(m, 1))                              |
%   +-----------------+---------------------------------------------------+
%   | InputRange      | vector of length <= 2, time range in which the    |
%   |                 | input function u(t) is applied to the system,     |
%   |                 | otherwise u = 0, if only a scalar tu0 is given,   |
%   |                 | the input time interval is taken to be [tu0, tf]  |
%   |                 | (default opts.TimeRange)                          |
%   +-----------------+---------------------------------------------------+
%   | LineSpecs       | character array, determining the line             |
%   | {!}             | specifications for the plots                      |
%   |                 | (default '-')                                     |
%   +-----------------+---------------------------------------------------+
%   | ShowPlot        | {0, 1}, used to disable/enable showing of plots   |
%   |                 | (default 1)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreStates     | {0, 1}, used to disable/enable storing the        |
%   |                 | computed system states additionally               |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | TimePoints      | positive integer, number of equally distributed   |
%   |                 | points in the time range, only used if            |
%   |                 | opts.TimeSample == 'equalpts'                     |
%   |                 | (default 500)                                     |
%   +-----------------+---------------------------------------------------+
%   | TimeRange       | vector of length <= 2, time range in which the    |
%   |                 | system is simulated, if only a scalar tf is given,|
%   |                 | time range is set to be [0, tf]                   |
%   |                 | (default [0 10])                                  |
%   +-----------------+---------------------------------------------------+
%   | TimeSample      | vector or character array, determining the way of |
%   |                 | sampling the time points                          |
%   |                 |  'equalpts'  - equally distributed opts.TimePoints|
%   |                 |  'equalstep' - equally distributed opts.TimeStep  |
%   |                 |  [numeric]   - given time values are used         |
%   |                 | (default 'equalpts')                              |
%   +-----------------+---------------------------------------------------+
%   | TimeStep        | scalar, time step size, only use if               |
%   |                 | opts.TimeSample == 'equalstep'                    |
%   |                 | (default (tf - t0) / 500)                         |
%   +-----------------+---------------------------------------------------+
%   | Xp0             | vector, initial states of the positions x(t)      |
%   | {!}             | (default zeros(n, 1))                             |
%   +-----------------+---------------------------------------------------+
%   | Xv0             | vector, initial states of the velocities x'(t)    |
%   | {!}             | (default zeros(n, 1))                             |
%   +-----------------+---------------------------------------------------+
%
%   Note: Arguments/Parameters marked with {!} may also be a cell array
%         containing multiple arguments.
%
% OUTPUTS:
%   t    - vector of size (number samples), conatining the time sampling
%          points
%   y    - 3-D array of size p x (number samples) x (number systems),
%          containing the outputs of the systems at the time points t
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErr          | if opts.DiffMode == 'off' empty, otherwise matrix |
%   |                 | of size p x (number samples) x (number systems-1),|
%   |                 | containing the computed absolute error values     |
%   +-----------------+---------------------------------------------------+
%   | Npts            | positive integer, number of plotted points        |
%   +-----------------+---------------------------------------------------+
%   | RelErr          | if opts.DiffMode == 'off' empty, otherwise matrix |
%   |                 | of size p x (number samples) x (number systems-1),|
%   |                 | containing the computed relative error values     |
%   +-----------------+---------------------------------------------------+
%   | Xv              | if opts.StoreStates == 0 empty, otherwise cell    |
%   |                 | array of length (number systems) containing       |
%   |                 | matrices of size n x (number samples) with the    |
%   |                 | computed states                                   |
%   +-----------------+---------------------------------------------------+
%   | Xv              | if opts.StoreStates == 0 empty, otherwise cell    |
%   |                 | array of length (number systems) containing       |
%   |                 | matrices of size n x (number samples) with the    |
%   |                 | computed first derivatives                        |
%   +-----------------+---------------------------------------------------+
%
% See also ml_ct_soss_simulate_ss22.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Check all input systems.
systems = cell(1);
iosizes = cell(1);
orders  = cell(1);
numSys  = 0;
hasOpts = 0;

for k = 1:nargin()
    sysInput = varargin{k};

    if isa(sysInput, 'cell')
        assert(isvector(sysInput), ...
            'MORLAB:data', ...
            'The cell inputs must be a vector!');
        sys_length = length(sysInput);
    else
        sys_length = 1;
    end

    for j = 1:sys_length
        if isa(sysInput, 'cell')
            sys = sysInput{j};
        else
            sys = sysInput;
        end

        if isa(sys, 'struct') && isfield(sys, 'K')
            % Second-order system case.
            numSys = numSys + 1;

            ml_assert_sosys(sys);

            if not(ml_field_set_to_value(sys, 'Cp'))
                sys.Cp = zeros(size(sys.Cv));
            end

            if not(ml_field_set_to_value(sys, 'Cv'))
                sys.Cv = zeros(size(sys.Cp));
            end

            if ml_field_set_to_value(sys, 'D')
                if ml_field_set_to_value(sys, 'Cp')
                    assert(isequal(size(sys.D), ...
                        [size(sys.Cp, 1), size(sys.Bu, 2)]), ...
                        'MORLAB:data', ...
                        ['The matrix D must have the dimensions ' ...
                        '%d x %d!'], ...
                        size(sys.Cp, 1), size(sys.Bu, 2));
                else
                    assert(isequal(size(sys.D), ...
                        [size(sys.Cv, 1), size(sys.Bu, 2)]), ...
                        'MORLAB:data', ...
                        ['The matrix D must have the dimensions ' ...
                        '%d x %d!'], ...
                        size(sys.Cv, 1), size(sys.Bu, 2));
                end
            else
                if ml_field_set_to_value(sys, 'Cp')
                    sys.D = zeros(size(sys.Cp, 1), size(sys.Bu, 2));
                else
                    sys.D = zeros(size(sys.Cv, 1), size(sys.Bu, 2));
                end
            end

            systems{numSys} = sys;
            orders{numSys}   = size(sys.M, 1);
            iosizes{numSys}  = size(sys.D);
        elseif isa(sys, 'mechss')
            numSys = numSys + 1;

            sys = struct('M', sys.M, 'E', sys.C, 'K', sys.K, ...
                'Bu', sys.B, 'Cp', sys.F, 'Cv', sys.G, 'D', sys.D);

            systems{numSys} = sys;
            orders{numSys}  = size(sys.M, 1);
            iosizes{numSys} = size(sys.D);
        elseif k < nargin()
            % Unsupported input.
            if isa(sys, 'struct')
                error('MORLAB:data', ...
                    'The given system structure is not supported!');
            else
                error('MORLAB:data', ...
                    'The systems must be structs or ss/dss objects!');
            end
        else
            % Wrong format of option structure.
            assert(isa(sys, 'struct'), ...
                'MORLAB:data', ...
                'The input argument opts must be a structure!');

            hasOpts = 1;
        end
    end
end

% Check minimum number of given systems.
assert(numSys > 0, ...
    'MORLAB:data', ...
    'Give at least one system for plotting!');

% Check number of inputs and outputs.
assert(all(cellfun(@(s) isequal(iosizes{1}, s), iosizes)), ...
    'MORLAB:data', ...
    'Number of inputs and outputs have to be the same for all systems!');

% Get last input as possible option struct if not a system.
if hasOpts
    opts = varargin{end};
else
    opts = struct();
end

% Check and set optional parameters.
if ml_field_set_to_value(opts, 'DiffMode')
    ml_assert_char(opts.DiffMode, 'opts.DiffMode')
    assert(strcmpi(opts.DiffMode, 'off') ...
        || strcmpi(opts.DiffMode, 'abs') ...
        || strcmpi(opts.DiffMode, 'rel') ...
        || strcmpi(opts.DiffMode, 'all'), ...
        'MORLAB:data', ...
        'The requested difference mode is not implemented!');

    if (numSys == 1) && not(strcmpi(opts.DiffMode, 'off'))
        warning('MORLAB:data', ...
            ['The difference mode needs at least 2 systems.\n' ...
            'Difference mode is turned off!']);
        opts.DiffMode = 'off';
    end
else
    opts.DiffMode = 'off';
end

if ml_field_set_to_value(opts, 'Info')
    ml_assert_boolean(opts.Info, 'opts.Info');
else
    opts.Info = false;
end

opts         = ml_check_cell_param(opts, 'LineSpecs', ...
    @ml_assert_char, '-');
numLineSpecs = length(opts.LineSpecs);

if ml_field_set_to_value(opts, 'ShowPlot')
    ml_assert_boolean(opts.ShowPlot, 'opts.ShowPlot');
else
    opts.ShowPlot = true;
end

if ml_field_set_to_value(opts, 'StoreStates')
    ml_assert_boolean(opts.StoreStates, 'opts.StoreStates');
else
    opts.StoreStates = false;
end

if ml_field_set_to_value(opts, 'TimePoints')
    ml_assert_posinteger(opts.TimePoints, 'opts.TimePoints');
else
    opts.TimePoints = 500;
end

if ml_field_set_to_value(opts, 'TimeRange')
    ml_assert_vector(opts.TimeRange, 'opts.TimeRange');
    assert(length(opts.TimeRange) <= 2, ...
        'MORLAB:data', ...
        'The parameter opts.TimeRange must be a vector of length <= 2!');

    if length(opts.TimeRange) == 1
        opts.TimeRange = [0, opts.TimeRange];
    end
else
    opts.TimeRange = [0, 10];
end

if ml_field_set_to_value(opts, 'InputFcn')
    assert(isa(opts.InputFcn, 'function_handle'), ...
        'MORLAB:data', ...
        'The parameter opts.InputFcn has to be a function handle!');

    tmp = opts.InputFcn(opts.TimeRange(1));
    assert(isequal(size(tmp), [iosizes{1}(2), 1]), ...
        'MORLAB:data', ...
        'The input function has to generate vectors of length %d!', ...
        iosizes{1}(2));
else
    opts.InputFcn = @(t) ones(iosizes{1}(2), 1);
end

if ml_field_set_to_value(opts, 'InputRange')
    ml_assert_vector(opts.InputRange, 'opts.InputRange');
    assert(length(opts.InputRange) <= 2, ...
        'MORLAB:data', ...
        'The parameter opts.InputRange must be a vector of length <= 2!');

    if length(opts.InputRange) == 1
        opts.InputRange = [opts.InputRange, opts.TimeRange(2)];
    end
else
    opts.InputRange = opts.TimeRange;
end

if ml_field_set_to_value(opts, 'TimeSample')
    if isnumeric(opts.TimeSample)
        ml_assert_vector(opts.TimeSample, 'opts.TimeSample');
    else
        ml_assert_char(opts.TimeSample, 'opts.TimeSample')
        assert(strcmpi(opts.TimeSample, 'equalpts') ...
            || strcmpi(opts.TimeSample, 'equalstep'), ...
            'MORLAB:data', ...
            'The requested time sampling is not implemented!');
    end
else
    opts.TimeSample = 'equalpts';
end

if ml_field_set_to_value(opts, 'TimeStep')
    ml_assert_scalar(opts.TimeStep, 'opts.TimeStep');
else
    opts.TimeStep = (opts.TimeRange(2) - opts.TimeRange(1)) / 500;
end

opts = ml_check_cell_vector(opts, 'Xp0', numSys, orders, @zeros);
opts = ml_check_cell_vector(opts, 'Xv0', numSys, orders, @zeros);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INITIALIZATION.                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Determine kind time grid.
if isnumeric(opts.TimeSample)
    if length(opts.TimeSample) > 1
        testDist = opts.TimeSample(1:end-1) - opts.TimeSample(2:end);
        tau      = testDist(1);

        if all(testDist(1) <= (testDist - eps)) ...
                || all(testDist(1) <= (testDist + eps))
            uniformGrid = 1;
        else
            uniformGrid = 0;
        end
    else
        tau         = 0;
        uniformGrid = 1;
    end

    t  = opts.TimeSample(:);
    nt = length(t) - 1;
else
    if strcmpi(opts.TimeSample, 'equalpts')
        nt  = opts.TimePoints;
        tau = (opts.TimeRange(2) - opts.TimeRange(1)) / nt;
    else
        nt  = ceil(abs(opts.TimeRange(2) - opts.TimeRange(1)) ...
            / opts.TimeStep);
        tau = opts.TimeStep;
    end

    uniformGrid = 1;
    t           = linspace(opts.TimeRange(1), opts.TimeRange(2), nt + 1)';
end

if tau == 0
    nt = 0;
end

% Precompute LU decomposition for uniform grids.
spSolves = zeros(1, numSys);
L        = cell(1, numSys);
U        = cell(1, numSys);
p        = cell(1, numSys);
q        = cell(1, numSys);
R        = cell(1, numSys);

for j = 1:numSys
    if uniformGrid
        spSolves(j) = issparse(systems{j}.M) ...
            && issparse(systems{j}.E) ...
            && issparse(systems{j}.K);

        if spSolves(j)
            [L{j}, U{j}, p{j}, q{j}, R{j}] = lu(systems{j}.M ...
                + tau * systems{j}.E + tau^2 * systems{j}.K, 'vector');
        else
            [L{j}, U{j}, p{j}] = lu(systems{j}.M ...
                + tau * systems{j}.E + tau^2 * systems{j}.K, 'vector');
        end
    end
end

% Initial states and outputs.
u = opts.InputFcn;
y = zeros(iosizes{1}(1), nt + 1, numSys);

if opts.StoreStates
    Xp = cell(1, numSys);
    Xv = cell(1, numSys);

    for j = 1:numSys
        Xp{j} = zeros(orders{j}, nt + 1);
        Xv{j} = zeros(orders{j}, nt + 1);
    end
end

if not(strcmpi(opts.DiffMode, 'off'))
    abserr = zeros(iosizes{1}(1), nt + 1, numSys - 1);
    relerr = zeros(iosizes{1}(1), nt + 1, numSys - 1);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% IMPLICIT EULER INTEGRATION SCHEME.                                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for j = 1:numSys
    if opts.Info
        fprintf(1, 'Simulate System %2d\n', j);
        fprintf(1, '===================\n');
    end

    xp         = opts.Xp0{j};
    xv         = opts.Xv0{j};
    y(:, 1, j) = systems{j}.Cp * xp + systems{j}.Cv * xv;

    if (t(1) >= min(opts.InputRange)) ...
            && (t(1) <= max(opts.InputRange))
        y(:, 1, j) = y(:, 1, j) + systems{j}.D * u(t(1));
    end

    if opts.StoreStates
        Xp{j}(:, 1) = xp;
        Xv{j}(:, 1) = xv;
    end

    for k = 1:nt
        if opts.Info
            fprintf(1, 'SIMULATE_IE Step %4d / %d\n', k, nt);
        end

        if not(uniformGrid)
            tau = t(k+1) - t(k);
        end

        tmp = systems{j}.M * xv - tau * systems{j}.K * xp;
        if (t(k+1) >= min(opts.InputRange)) ...
                && (t(k+1) <= max(opts.InputRange))
            tmp = tmp + tau * systems{j}.Bu * u(t(k+1));
        end

        if uniformGrid
            if spSolves(j)
                tmp(q{j}, :) = U{j} \ (L{j} \ (R{j}(:, p{j}) \ tmp));
            else
                tmp = U{j} \ (L{j} \ tmp(p{j}));
            end
        else
            tmp = (systems{j}.M + tau * systems{j}.E ...
                + tau^2 * systems{j}.K) \ tmp;
        end

        xp = xp + tau * tmp;
        xv = tmp;

        y(:, k+1, j) = systems{j}.Cp * xp + systems{j}.Cv * xv;
        if (t(k+1) >= min(opts.InputRange)) ...
                && (t(k+1) <= max(opts.InputRange))
            y(:, k+1, j) = y(:, k+1, j) + systems{j}.D * u(t(k+1));
        end

        if opts.StoreStates
            Xp{j}(:, k+1) = xp;
            Xv{j}(:, k+1) = xv;
        end
    end

    if opts.Info && (j < numSys)
        fprintf(1, '\n');
    end
end

% Compute errors if requested.
if not(strcmpi(opts.DiffMode, 'off'))
    maxY            = max(abs(y(:, :, 1)), [], 2);
    maxY(maxY == 0) = eps;

    for j = 2:numSys
        abserr(:, :, j-1) = abs(y(:, :, 1) - y(:, :, j));

        for kp = 1:iosizes{1}(1)
            relerr(kp, :, j-1) = abserr(kp, :, j-1) / maxY(kp);
        end
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CONSTRUCTION OF PLOTS.                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Save hold status for figure window.
if opts.ShowPlot
    holdStatus = ishold();
else
    holdStatus = 1;
end

% Set line specifications for plots.
lineSpecs = cell(1, numSys);
for j = 1:numSys
    if numLineSpecs == 1
        lineSpecs{j} = opts.LineSpecs{1};
    elseif j <= numLineSpecs
        lineSpecs{j} = opts.LineSpecs{j};
    else
        lineSpecs{j} = '-';
    end
end

% Correct color choices for 'all' plots by setting first one black.
if strcmpi(opts.DiffMode, 'all')
    setColor  = zeros(1, numSys);

    for colors = {'r', 'g', 'b', 'c', 'm', 'y', 'k', 'w'}
        setColor = setColor + cellfun(@(s) ismember(colors{1}, s), ...
            opts.LineSpecs);
    end

    if not(any(setColor))
        lineSpecs{1} = [lineSpecs{1} 'k'];
    end
end

% Plot time simulation.
if opts.ShowPlot
    switch lower(opts.DiffMode)
        case 'off'
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys
                    plot(t, y(k, :, j)', lineSpecs{j});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

        case 'abs'
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys-1
                    plot(t, abserr(k, :, j)', lineSpecs{j+1});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

        case 'rel'
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys-1
                    plot(t, relerr(k, :, j)', lineSpecs{j+1});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

        case 'all'
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys
                    plot(t, y(k, :, j)', lineSpecs{j});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

            figure();
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys-1
                    plot(t, abserr(k, :, j)', lineSpecs{j+1});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

            figure();
            for k = 1:iosizes{1}(1)
                subplot(iosizes{1}(1), 1, k)

                for j = 1:numSys-1
                    plot(t, relerr(k, :, j)', lineSpecs{j+1});
                    hold on;
                end

                if k == iosizes{1}(1)
                    xlabel('Time');
                end

                if iosizes{1}(1) > 1
                    ylabel(sprintf('Amplitude Out(%d)', k));
                else
                    ylabel('Amplitude');
                end
            end

    end
end

% Turn hold status to normal.
if not(holdStatus)
    hold off;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT INFORMATION.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if strcmpi(opts.DiffMode, 'off')
    [abserr, relerr] = deal([]);
end

if not(opts.StoreStates)
    [Xp, Xv] = deal([]);
end

info = struct( ...
    'AbsErr', abserr, ...
    'Npts'  , nt, ...
    'RelErr', relerr, ...
    'Xp'    , {Xp}, ...
    'Xv'    , {Xv});
