%% MORLAB Demo: Partial Stabilization
% This demo script contains the application of the MORLAB partial
% stabilzation routines.
%
% See also ml_ct_d_dss_partstab, ml_ct_d_ss_partstab, ml_dt_d_dss_partstab,
% ml_dt_d_ss_partstab.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%% Standard System Case
% The MORLAB toolbox implements partial stabilization methods for
% first-order system types, e.g., considering the control system
%
%  x'(t) = A*x(t) + B*u(t),
%
% with the matrix A having possible stable and anti-stable eigenvalues.
% In partial stabilization, a feedback matrix K is constructed such that
% the matrix A-B*K has only stable eigenvalues, while the original stable
% eigenvalues stay uneffected by the matrix K.

%%
% For demonstration reasons we load a prepared data file containing an
% unstable standard system:

if exist('OCTAVE_VERSION', 'builtin')
    orig_warn = warning('off', 'Octave:data-file-in-path');
    load morlab_data_std_unstab.mat;
    warning(orig_warn);
else
    load morlab_data_std_unstab.mat;
end

%%
% The matrix A has 90 stable and 10 anti-stable eigenvalues. The
% construction of the matrix K is done by calling the appropriate MORLAB
% routine.

[K, info] = ml_ct_d_ss_partstab(A, B);

%%
% The struct info contains information, collected about the stabilization
% process.

disp(info);

%%
% We see that the method found the 90 stable and 10 anti-stable
% eigenvalues. Also it shows that the algebraic Bernoulli equation was
% used for the stabilization, i.e., the anti-stable
% eigenvalues have been mirrored into the left open halfplane.
% We take a look on the spectrum of A-B*K and see that now all eigenvalues
% lie in the open left half-plane.

x = eig(A - B * K);
disp(all(real(x) < 0));


%% Descriptor System Case
% The partial stabilization is working a bit differently in case of
% descriptor control systems, e.g.,
%
%  E*x'(t) = A*x(t) + B*u(t),
%
% with the matrix pencil s*E-A having possible stable, anti-stable and
% infinite eigenvalues. The feedback matrix K is now constructed such that
% the new pencil s*E-(A-B*K) has only stable and infinite eigenvalues,
% while the original stable and infinite eigenvalues stay uneffected by the
% matrix K.

%%
% For demonstration reasons we load a prepared data file containing an
% unstable descriptor system:

if exist('OCTAVE_VERSION', 'builtin')
    orig_warn = warning('off', 'Octave:data-file-in-path');
    load morlab_data_desc_infunstab.mat;
    warning(orig_warn);
else
    load morlab_data_desc_infunstab.mat;
end

%%
% The pencil s*E-A has 80 stable, 10 anti-stable and 10 infinite
% eigenvalues. The construction of the matrix K is done by calling the
% appropriate MORLAB routine.

[K, info] = ml_ct_d_dss_partstab(A, B, E);

%%
% The struct info contains information, collected about the stabilization
% process.

disp(info);

%%
% We see that the method found the 80 stable, 10 anti-stable and 10
% infinite eigenvalues could be found. Also it shows that the algebraic
% Bernoulli equation was used for the stabilization, i.e., the anti-stable
% eigenvalues have been mirrored into the left open halfplane.
% We take a look on the spectrum of s*E-(A-B*K).

x = eig(A - B * K, E);
x = x(abs(x) < 1.0e+05);

%%
% After sorting out the numerically infinite eigenvalues, we see that all
% eigenvalues stored in x lie in the left open half-plane.

disp(all(real(x) < 0));


%% Remarks
%
% * All partial stabilization routines are more or less equivalent to each
%   other, which means that all usage examples here also apply to the other
%   routines.
% * For discrete-time standard and descriptor systems, the methods work
%   basically the same with respect to discrete-time stability.
% * The continuous-time partial stabilization routines allow to choose
%   between the Bernoulli equation and Lyapunov equation approach.
