%% MORLAB Demo: Option Constructor Function ml_morlabopts
% This demo script contains the application of the option constructor
% function |ml_morlabopts| and some hints for proper usage.
%
% See also ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%% General Syntax
% In principal, the function |ml_morlabopts| is the option constructor of
% the MORLAB toolbox and can be used for any MORLAB function or datafile.
% The function fulfills two aims:
%
% # documentation of all possible optional parameters of MORLAB functions
% # construction of empty option structs that can be filled by the user
%
% The disctinction of both cases is made by the calling syntax, i.e.,
% calling |ml_morlabopts| with no output argument leads to the
% documentation of the requested function, while asking for an output
% argument creates an empty version of the option struct of the requested
% function.
% Both cases are further explained below.


%% No Output Argument: Optional Parameter Documentation
% As mentioned above, this way of using |ml_morlabopts| gives a
% documentation of the possible optional parameters.
% The general syntax is as follows
%
%   ml_morlabopts('name_of_a_MORLAB_function')
%
% where |name_of_a_MORLAB_function| determines the requested function.
% As an example, we want the documentation of the optional parameters of
% the |ml_ct_bt| function by using

ml_morlabopts('ml_ct_bt');

%%
% This gives as screen output all optional parameters of the function, as
% well as descriptions of the admissible data types, e.g., the parameter
% |StoreProjection| is only allowed to be boolean and |Tolerance| can be
% a scalar value, which is nonnegative. The internal used default values
% are show in curly brackets {}.
% Note that the fields |stabsignmopts| and |stabsylvopts| have an
% additional call of |ml_morlabopts| as default value, which means that
% those fields are option structs of other functions that are used inside
% of the requested function.
%
% In case a MORLAB function doesn't support option structs, the user will
% be informed by a sentence. For example calling
%
%   ml_morlabopts('ml_field_set_to_value')
%
% won't show any parameters but that the hint that there is no option
% struct used by the requested function.

ml_morlabopts('ml_field_set_to_value');


%% One Output Argument: Option Struct Construction
% Requesting an output argmuent of |ml_morlabopts| will change the behavior
% of the function from documenting the optional parameters to creating
% empty versions of the appropriate option structs of a function.
%
% We start with a simple example. To create an option struct of the
% function |ml_lyap_sgn| we used need to use

opts = ml_morlabopts('ml_lyap_sgn');

%%
% This will create the empty option struct with all the fields that can be
% used as optional parameters for the |ml_lyap_sgn| function. If we take a
% look inside the struct

disp(opts);

%%
% We see that all fields are empty. That will cause any MORLAB function to
% use default values that are determined during the function call. To use
% a different optional parameter value than the default one, just change
% the empty array to a suitable value.
% As example, we want to change the tolerance for the relative solution
% error of |ml_lyap_sgn| to 1.0e-10. Therefore, we set

opts.RelTol = 1.0e-10;

%%
% MORLAB supports the complete customization of all subroutines used inside
% the model reduction methods by nested option structs.

opts = ml_morlabopts('ml_ct_d_ss_bt');

%%
% creates the option struct for the standard balanced truncation function.
% If we take a closer look

disp(opts);

%%
% we see that beside empty arrays the struct contains also 1x1 structs for
% fields, which end on _opts_. Those structs contain the optional
% parameters for subroutines used inside the function, e.g.,

disp(opts.lyapopts)

%%
% contains the optional parameters for the dual Lyapunov equation solver
% that is used inside the balanced truncation method.

%%
% For functions that doesn't support option structs, the result of
% |ml_morlabopts| will just be an empty struct, e.g.,

opts = ml_morlabopts('ml_field_set_to_value');
