%% MORLAB Demo: Matrix Equation Solvers
% This demo script contains the application of the MORLAB matrix equation
% solver routines.
%
% See also ml_cabe_sgn, ml_care_nwt_fac, ml_caredl_sgn, ml_caredl_sgn_fac,
% ml_dare_nwt_fac, ml_daredl_sda, ml_daredl_sda_fac, ml_dlyap_smith,
% ml_dlyap_smith_fac, ml_dlyap_smith_ldl, ml_dlyapdl_smith,
% ml_dlyapdl_smith_fac, ml_dlyapdl_smith_ldl, ml_dsylv_smith,
% ml_dsylv_smith_fac, ml_gdlyapdl_smith_fac, ml_icare_ric_fac, ml_lyap_sgn,
% ml_lyap_sgn_fac, ml_lyap_sgn_ldl, ml_lyapdl_sgn, ml_lyapdl_sgn_fac,
% ml_lyapdl_sgn_ldl, ml_pcare_nwt_fac, ml_sylv_sgn, ml_sylv_sgn_fac.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%% General Handling
% The MORLAB toolbox contains a bunch of iterative matrix equation solvers,
% which are used for the implemented model reduction methods. Due to the
% modolarity of the toolbox, the user can use each of the matrix equation
% solver routines by itself.
% MORLAB has implemented solvers for the following types of matrix
% equations:
%
% * Continuous-time Lyapunov equations A*X*E' + E*X*A' + Q = 0:
%   |ml_lyap_sgn|, |ml_lyap_sgn_fac|, |ml_lyap_sgn_ldl|, |ml_lyapdl_sgn|,
%   |ml_lyapdl_sgn_fac|, |ml_lyapdl_sgn_ldl|
% * Continuous-time Riccati equations A'*X*E + E'*X*A + G - E'*X*Q*X*E = 0:
%   |ml_care_nwt_fac|, |ml_caredl_sgn|, |ml_caredl_sgn_fac|,
%   |ml_icare_ric_fac|, |ml_pcare_nwt_fac|
% * Continuous-time Bernoulli equations A'*X*E + E'*X*A - E'*X*Q*X*E = 0:
%   |ml_cabe_sgn|
% * Continuous-time Sylvester equations A*X*F + E*X*B + C = 0:
%   |ml_sylv_sgn|, |ml_sylv_sgn_fac|
% * Discrete-time Lyapunov equations A*X*A' - E*X*E' + Q = 0:
%   |ml_dlyap_smith|, |ml_dlyap_smith_fac|, |ml_dlyap_smith_ldl|,
%   |ml_dlyapdl_smith|, |ml_dlyapdl_smith_fac|, |ml_dlyapdl_smith_ldl|,
%   |ml_gdlyapdl_smith_fac|
% * Discrete-time Riccati equations
%   A'*X*A - E'*X*E + G - A'*X*inv(I - Q*X)*Q*X*A = 0:
%   |ml_dare_nwt_fac|, |ml_daredl_sda|, |ml_daredl_sda_fac|
% * Discrete-time Sylvester equations A*X*B - E*X*F + C = 0:
%   |ml_dsylv_smith|, |ml_dsylv_smith_fac|

%%
% In contrast to the system-based routines in MORLAB, the matrix equation
% solvers just need the coefficient matrices of the equation that has to be
% solved.
% For demonstration, we load a prepared data file containing state-space
% matrices for a stable passive system:

if exist('OCTAVE_VERSION', 'builtin')
    orig_warn = warning('off', 'Octave:data-file-in-path');
    load morlab_data_std_pr.mat;
    warning(orig_warn);
else
    load morlab_data_std_pr.mat;
end

%%
% Now we want to solve a Lyapunov equation with those coefficient matrices.

Q = B * B';
[X, info] = ml_lyap_sgn(A, Q);

%%
% The matrix X is the solution of the Lyapunov equation A*X + X*A' + Q = 0
% as we will show by computing a normalized residual below.
% The structure info instead contains information about the underlying
% iteration like the number of iteration steps, the absolute and relative
% error of the method.

res = norm(A * X + X * A' + Q) / norm(X);
disp(res);


%% Solver Types and Naming Scheme
% There are different types of solvers implemented in MORLAB. The type of
% solver and the underlying method can be determined by the naming scheme
% of the MORLAB function:
%
% * the ending |_fac| stands for the factorized version of a solver, i.e.,
%   certain coefficient matrices are factorized and the solution will also
%   be factorized,
% * the ending |_ldl| stands for the LDL^T version of a solver, i.e., it
%   takes certain coefficient matrices in this format and returns an LDL^T
%   factorized solution,
% * if |dl| is attached to the equation name, two dual matrix equations can
%   be solved by the function at the same time,
% * |_nwt| stands for Newton-type solvers, |_sgn| for the matrix sign
%   function, |_smith| for the Smith or squared Smith iteration, |_sda| for
%   the structure-preserving doubling method and |_ric| for the Riccati
%   iteration.
%
% As example, we will use the solver |ml_lyapdl_sgn_fac|. By the naming
% scheme above, we will solve the dual Lyapunov equations
%
%  A*X + X*A' + B*B' = 0,
%  A'*Y + Y*A + C'*C = 0,
%
% where the solutions are factorized as X = R*R' and Y = L*L'.

[R, L] = ml_lyapdl_sgn_fac(A, B, C);

X = R*R';
Y = L*L';

resX = norm(A * X + X * A' + B * B') / norm(X);
resY = norm(A' * Y + Y * A + C' * C) / norm(Y);

disp(resX);
disp(resY);


%% Remarks
%
% * All matrix equation solver routines are more or less equivalent to each
%   other, which means that all usage examples here also apply to the other
%   equations solvers.
% * The dual Riccati equation solvers have the optional parameter to
%   determine if either the primal, the dual or both equations should be
%   solved.
