% Data script for the construction of an unstable standard system.
%
% DESCRIPTION:
%   This data script generates the matrices of an unstable standard LTI
%   system of the form
%
%       x'(t) = Ax(t) + Bu(t),
%        y(t) = Cx(t) + Du(t),
%
%   with A an (n x n) matrix, B an (n x m) matrix, C a (p x n) matrix, and
%   D a (p x m) matrix.
%
%   The construction has the following dimensions:
%       n = 100,
%       m = 3,
%       p = 2,
%   and the matrix A is constructed such that 10 eigenvalues lie in the
%   open right half-plane and 90 in the open left half-plane.
%   In the datainfo struct the following numbers are saved:
%       - n, number of states,
%       - ns, number of eigenvalues of A in the open left half-plane,
%       - nu, number of eigenvalues of A in the open right half-plane.
%
% NOTES:
%   * The data, generated under Matlab 8.0.0.783 (R2012b) GLNXA64, is saved
%     in the same-named .MAT file.
%   * This script cannot be used under Octave.
%
% See also morlab_data_desc_infunstab.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% System dimensions.
n = 100;
m = 3;
p = 2;

% Random seed.
s = 1.0;

% System generation.
rng(s);
A = rand(n);

rng(s);
B = rand(n, m);

rng(s);
C = rand(p, n);

rng(s);
D = rand(p, m);

% Shift of the matrix to get desired eigenvalue structure.
A = A - 2.1 * eye(n);

% Info about the generated data structure.
x = real(eig(A));

datainfo = struct(...
    'n' , n, ...
    'ns', sum(x < 0), ...
    'nu', sum(x > 0));

% Clear unwanted variables.
clear n m p s x;
