% Data script for the construction of a positive-real standard system.
%
% DESCRIPTION:
%   This data script generates the matrices of a positive-real standard
%   LTI system of the form
%
%       x'(t) = Ax(t) + Bu(t),
%        y(t) = Cx(t) + Du(t),
%
%   with A an (n x n) matrix, B an (n x m) matrix, C an (m x n) matrix, and
%   D an (m x m) matrix. The positive-realness of the system is defined by
%   the following conditions on the system's transfer function
%   G(s) = C(sI - A)^(-1)B + D:
%       - conj(G(s)) = G(conj(s)),
%       - G is analytic in the open right half-plane,
%       - G(s) + G'(conj(s)) >= 0.
%
%   The construction has the following dimensions:
%       n = 100,
%       m = 3,
%   and the matrix C is constructed as B' to ensure the positive-realness
%   of the system. Also the following sufficient conditions were used for
%   the construction:
%       - A is Hurwitz,
%       - C * B = (C * B)' > 0,
%       - C * A * B + (C * A * B)' <= 0,
%       - D + D' > 0 (for existence of Riccati equation).
%   In the datainfo struct the following numbers are saved:
%       - n, number of states,
%       - ns, number of eigenvalues of A in the open left half-plane,
%       - nu, number of eigenvalues of A in the open right half-plane.
%
% NOTES:
%   * The data, generated under Matlab 8.0.0.783 (R2012b) GLNXA64, is saved
%     in the same-named .MAT file.
%   * This script cannot be used under Octave.
%
% See also morlab_data_desc_pr.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% System dimensions.
n = 100;
m = 3;

% Random seed.
s = 1.0;

% System generation.
rng(s);
A = rand(n);

rng(s);
B = rand(n, m);
C = B';

rng(s);
D = rand(m);

% Postprocessing to ensure the sufficient conditions.
x = abs(eigs(D + D', 1, 'lm'));
D = D + 1.01 * x/2 * eye(m);

y = abs(eigs(A, 1, 'lm'));
A = A - 1.01 * y * eye(n);

% To test the positive-realness, the Hamiltonian pencil corresponding to
% the positive-real Riccati equations doesn't have eigenvalues on the
% imaginary axis. Since the example with fixed random seed was already
% tested, this part is commented out.
%
% R = D + D';
% H = [A - B * (R \ C), B * (R \ B'); -C' * (R \ C), -(A - B * (R \ C))'];
% eig(H)

% Info about the generated data structure.
x = real(eig(A));

datainfo = struct(...
    'n' , n, ...
    'ns', sum(x < 0), ...
    'nu', sum(x > 0));

% Clear unwanted variables.
clear n m s x y R H;
