% Data script for the construction of an d-stable standard system.
%
% DESCRIPTION:
%   This data script generates the matrices of a d-stable standard LTI
%   system of the form
%
%       x(t+1) = Ax(t) + Bu(t),
%         y(t) = Cx(t) + Du(t),
%
%   with A an (n x n) matrix, B an (n x m) matrix, C a (p x n) matrix, and
%   D a (p x m) matrix.
%
%   The construction has the following dimensions:
%       n = 100,
%       m = 3,
%       p = 2,
%   and the matrix A is constructed such that all eigenvalues lie inside
%   the open unit disk.
%   In the datainfo struct the following numbers are saved:
%       - n, number of states,
%       - ns, number of eigenvalues of A inside the open unit disk,
%       - nu, number of eigenvalues of A outside the open unit disk.
%
% NOTES:
%   * The data, generated under Matlab 9.1.0.441655 (R2016b) GLNXA64, is
%     saved in the same-named .MAT file.
%   * This script cannot be used under Octave.
%
% See also morlab_data_std_unstab.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% System dimensions.
n = 100;
m = 3;
p = 2;

% Random seed.
s = 1.0;

% System generation.
rng(s);
A = randn(n);

rng(s);
B = rand(n, m);

rng(s);
C = rand(p, n);

rng(s);
D = rand(p, m);

% Shift of the matrix to get desired eigenvalue structure.
A = A / 10.2;

% Set sampling time to 1.
Ts = 1;

% Info about the generated data structure.
x = abs(eig(A));

datainfo = struct(...
    'n' , n, ...
    'ns', sum(x < 1), ...
    'nu', sum(x > 1));

% Clear unwanted variables.
clear n m p s x;
