% Data script for the construction of an bounded-real standard system.
%
% DESCRIPTION:
%   This data script generates the matrices of a bounded-real standard
%   LTI system of the form
%
%       x'(t) = Ax(t) + Bu(t),
%        y(t) = Cx(t) + Du(t),
%
%   with A an (n x n) matrix, B an (n x m) matrix, C a (p x n) matrix, and
%   D a (p x m) matrix. The bounded-realness of the system is defined by
%   the following conditions on the system's transfer function
%   G(s) = C(sI - A)^(-1)B + D:
%       - conj(G(s)) = G(conj(s)),
%       - G is analytic in the open right half-plane,
%       - I - G(s) * G'(conj(s)) >= 0 and I - G'(conj(s)) * G(s) >= 0
%
%   The construction has the following dimensions:
%       n = 100,
%       m = 3,
%       p = 2,
%   and the matrices C, B and D are scaled to ensure the bounded-realness
%   of the system, such that:
%       - ||G||_{\infty} <= 1,
%       - eye(p) - D*D' and eye(m) - D'*D > 0.
%   In the datainfo struct the following numbers are saved:
%       - n, number of states,
%       - ns, number of eigenvalues of A in the open left half-plane,
%       - nu, number of eigenvalues of A in the open right half-plane.
%
% NOTES:
%   * The data, generated under Matlab 8.0.0.783 (R2012b) GLNXA64, is saved
%     in the same-named .MAT file.
%   * This script cannot be used under Octave.
%
% See also morlab_data_desc_br.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% System dimensions.
n = 100;
m = 3;
p = 2;

% Random seed.
s = 1.0;

% System generation.
rng(s);
A = rand(n);

rng(s);
B = rand(n, m);

rng(s);
C = rand(p, n);

rng(s);
D = rand(p, m);

% Postprocessing to ensure the bounded-realness.
y = abs(eigs(A, 1, 'lm'));
A = A - 1.01 * y * eye(n);

x = 1.1 * norm(ss(A, B, C, D), inf);

B = 1 / sqrt(x) * B;
C = 1 / sqrt(x) * C;
D = 1 / x * D;

% Info about the generated data structure.
x = real(eig(A));

datainfo = struct(...
    'n' , n, ...
    'ns', sum(x < 0), ...
    'nu', sum(x > 0));

% Clear unwanted variables.
clear n m p s x y;
