% Data script for the construction of a positive-real descriptor system.
%
% DESCRIPTION:
%   This data script generates the matrices of a positive-real descriptor
%   LTI system of the form
%
%       Ex'(t) = Ax(t) + Bu(t),
%         y(t) = Cx(t) + Du(t),
%
%   with E and A (n x n) matrices, B an (n x m) matrix, C an (m x n)
%   matrix, and D an (m x m) matrix. The positive-realness of the system
%   is defined by the following conditions on the system's transfer
%   function G(s) = C(sE - A)^(-1)B + D:
%       - conj(G(s)) = G(conj(s)),
%       - G is analytic in the open right half-plane,
%       - G(s) + G'(conj(s)) >= 0.
%
%   The construction has the following dimensions:
%       n = 100,
%       m = 3,
%   and the matrix pencil s*E - A is constructed such that 10 eigenvalues
%   are infinite with index 3, also the matrix C is constructed as B' to
%   ensure the positive-realness of the system.
%   Since there is no complete set of sufficient conditions for
%   positive-real descriptor systems known, are variant of the sufficient
%   conditions for standard systems was used.
%   In the datainfo struct the following numbers are saved:
%       - n, number of states,
%       - ni, number of infinite eigenvalues of s*E - A,
%       - np, number of finite eigenvalues of s*E - A in the open left
%         half-plane,
%       - nu, number of finite eigenvalues of s*E - A in the open right
%         half-plane.
%
% NOTES:
%   * The data, generated under Matlab 8.0.0.783 (R2012b) GLNXA64, is saved
%     in the same-named .MAT file.
%   * This script cannot be used under Octave.
%
% See also morlab_data_std_pr.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% System dimensions.
n = 100;
m = 3;

% Random seed.
s = 1.0;

% System generation.
rng(s);
A = blkdiag(rand(n - 10), eye(10));

E         = blkdiag(eye(n - 10), zeros(10));
E(91, 92) = 1;
E(92, 93) = 1;
E(94, 95) = 1;

rng(s);
B = [rand(n - 10, m); zeros(7, m); eye(3, m)];
C = B';

rng(s);
D = rand(m);

% Postprocessing to ensure positive-realness.
x = abs(eigs(D + D', 1, 'lm')) + 1;
D = D + 1.01 * x/2 * eye(m);

y = abs(eigs(A, 1, 'lm'));
A = A - 1.01 * y * blkdiag(eye(n - 10), zeros(10));

% To test the positive-realness, the Hamiltonian pencil corresponding to
% the positive-real Riccati equations doesn't have eigenvalues on the
% imaginary axis. Since the example with fixed random seed was already
% tested, this part is commented out.
%
% M = D - C(: , 91:end) * (A(91:end, 91:end) \ B(91:end, :));
% R = M + M';
% H = [A - B * (R \ C), B * (R \ B'); -C' * (R \ C), -(A - B * (R \ C))'];
% G = blkdiag(E, E');
% eig(H, G)

% System transformation.
rng(s)
U = orth(rand(n));

A = U' * A * U;
E = U' * E * U;
B = U' * B;
C = C * U;

% Info about the generated data structure.
x = real(eig(A));

datainfo = struct(...
    'n' , n, ...
    'ni', 10, ...
    'np', sum(x < 0), ...
    'nu', n - 10 - sum(x < 0));

% Clear unwanted variables.
clear n m s x y M R H G U;
