% Data script for the construction of an d-unstable descriptor system.
%
% DESCRIPTION:
%   This data script generates the matrices of an d-unstable descriptor LTI
%   system of the form
%
%       Ex(t+1) = Ax(t) + Bu(t),
%          y(t) = Cx(t) + Du(t),
%
%   with E and A (n x n) matrices, B an (n x m) matrix, C a (p x n)
%   matrix, and D a (p x m) matrix.
%
%   The construction has the following dimensions:
%       n = 100,
%       m = 3,
%       p = 2,
%   and the matrix pencil s*E - A is constructed such that 10 eigenvalues
%   are infinite with index 3, 8 eigenvalues lie outside the open unit
%   disk and 82 inside the open unit disk.
%   Also, the system is constructed to get a proper transfer function.
%   In the datainfo struct the following numbers are saved:
%       - n, number of states,
%       - ni, number of infinite eigenvalues of s*E - A,
%       - np, number of finite eigenvalues of s*E - A inside the open unit
%         disk,
%       - nu, number of finite eigenvalues of s*E - A outside the open unit
%         disk.
%
% NOTES:
%   * The data, generated under Matlab 9.1.0.441655 (R2016b) GLNXA64, is
%     saved in the same-named .MAT file.
%   * This script cannot be used under Octave.
%
% See also morlab_data_std_unstab.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% System dimensions.
n = 100;
m = 3;
p = 2;

% Random seed.
s = 1.0;

% System generation.
rng(s);
B = rand(n, m);

rng(s);
C = [rand(p, n - 10), zeros(p, 8), eye(p, 2)];

rng(s);
D = rand(p, m);

% Construction of the matrix pencil.
rng(s);
A = blkdiag(rand(n - 10), eye(10));

E         = blkdiag(eye(n - 10), zeros(10));
E(91, 92) = 1;
E(92, 93) = 1;
E(95, 96) = 1;

% System transformation.
rng(s)
U = orth(rand(n));

A = U' * A * U;
E = U' * E * U;
B = U' * B;
C = C * U;

A = A / 1.27;
E = 2 * E;

% Set sampling time to 1.
Ts = 1;

% Info about the generated data structure.
x = abs(eig(A, E));

datainfo = struct(...
    'n' , n, ...
    'ni', 10, ...
    'np', sum(x < 1), ...
    'nu', n - 10 - sum(x < 1));

% Clear unwanted variables.
clear n m p s U x;
