% Data script for the construction of a bounded-real descriptor system.
%
% DESCRIPTION:
%   This data script generates the matrices of a bounded-real descriptor
%   LTI system of the form
%
%       Ex'(t) = Ax(t) + Bu(t),
%         y(t) = Cx(t) + Du(t),
%
%   with E and A (n x n) matrices, B an (n x m) matrix, C a (p x n)
%   matrix, and D a (p x m) matrix. The bounded-realness of the system
%   is defined by the following conditions on the system's transfer
%   function G(s) = C(sE - A)^(-1)B + D:
%       - conj(G(s)) = G(conj(s)),
%       - G is analytic in the open right half-plane,
%       - I - G(s) * G'(conj(s)) >= 0 and I - G'(conj(s)) * G(s) >= 0
%
%   The construction has the following dimensions:
%       n = 100,
%       m = 3,
%       p = 2,
%   and the matrix pencil s*E - A is constructed such that 10 eigenvalues
%   are infinite with index 3, also the matrices C, B and D are scaled to
%   ensure the bounded-realness of the system, such that:
%       - ||G||_{\infty} <= 1,
%       - eye(p) - M * M' and eye(m) - M' * M > 0,
%         with M = D - Cinf * (Ainf \ Binf) where Ainf, Binf, Cinf are the
%         parts of the system matrices corresponding to the infinite
%         eigenvalues.
%   In the datainfo struct the following numbers are saved:
%       - n, number of states,
%       - ni, number of infinite eigenvalues of s*E - A,
%       - np, number of finite eigenvalues of s*E - A in the open left
%         half-plane,
%       - nu, number of finite eigenvalues of s*E - A in the open right
%         half-plane.
%
% NOTES:
%   * The data, generated under Matlab 8.0.0.783 (R2012b) GLNXA64, is saved
%     in the same-named .MAT file.
%   * This script cannot be used under Octave.
%
% See also morlab_data_std_br.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% System dimensions.
n = 100;
m = 3;
p = 2;

% Random seed.
s = 1.0;

% System generation.
rng(s);
A = blkdiag(rand(n - 10), eye(10));

E         = blkdiag(eye(n - 10), zeros(10));
E(91, 92) = 1;
E(92, 93) = 1;
E(94, 95) = 1;

rng(s);
B = [rand(n - 10, m); zeros(7, m); eye(3, m)];

rng(s)
C = rand(p, n);

rng(s);
D = rand(p, m);

% Shift of the spectrum.
y = abs(eigs(A, 1, 'lm'));
A = A - 1.01 * y * blkdiag(eye(n - 10), zeros(10));

% System transformation.
rng(s)
U = orth(rand(n));

A = U' * A * U;
E = U' * E * U;
B = U' * B;
C = C * U;

% Postprocessing to ensure the bounded-realness.
x = 1.1 * norm(dss(A, B, C, D, E), inf);

B = 1 / sqrt(x) * B;
C = 1 / sqrt(x) * C;
D = 1 / x * D;

% Info about the generated data structure.
x = real(eig(A));

datainfo = struct(...
    'n' , n, ...
    'ni', 10, ...
    'np', sum(x < 0), ...
    'nu', n - 10 - sum(x < 0));

% Clear unwanted variables.
clear n m p s x y U;
