function opts = ml_check_messopts(funcs, opts)
%ML_CHECK_MESSOPTS Create default option structures for M-M.E.S.S.
%
% SYNTAX:
%   opts = ML_CHECK_MESSOPTS()
%   opts = ML_CHECK_MESSOPTS(funcs)
%   opts = ML_CHECK_MESSOPTS(funcs, opts)
%
% DESCRIPTION:
%   This function sets default option values for M-M.E.S.S. functions.
%   If the user gives an option struct, it will be extended by the missing
%   default values. Note that this function does only set the necessary
%   parameters; for a full list please see the routines in M-M.E.S.S.
%
% INPUTS:
%   funcs - string or cell array of strings specifying the M-M.E.S.S. for
%           which option fields need to be generated; possible strings are
%            'global' - options shared by all M-M.E.S.S. routines
%            'adi'    - ADI solvers
%            'nm'     - Newton method
%            'radi'   - RADI method
%            'ri'     - Riccati iteration
%            'shifts' - shift computations
%           if left empty, then all optional values are set
%   opts  - struct, options to extend by default parameters
%
% OUTPUTS:
%   opts - constructed option struct
%
% See also ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(0, 2);

% Default values.
if (nargin < 1) || isempty(funcs)
    funcs = {'global', 'adi', 'nm', 'radi', 'ri', 'shifts'};
end

% Convert input into cell array.
if not(isa(funcs, 'cell')), funcs = {funcs}; end

% Check entries.
funcs = ml_check_cell_param(funcs, 'funcs', @ml_assert_char);

% Initial struct.
if nargin == 2
    assert(isa(opts, 'struct'), ...
        'MORLAB:data', ...
        'The input argument opts has to be a struct!');
else
    opts = struct();
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% OPTION CONSTRUCTOR.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Loop over requested functions.
for k = 1:length(funcs)
    switch lower(funcs{k})
        case 'global' % Global options.
            if not(ml_field_set_to_value(opts, 'norm'))
                opts.norm = 'fro';
            end
            if not(ml_field_set_to_value(opts, 'LDL_T'))
                opts.LDL_T = false;
            end

        case 'adi' % ADI options.
            if ml_field_set_to_value(opts, 'adi')
                assert(isa(opts.adi, 'struct'), ...
                    'MORLAB:data', ...
                    'The parameter opts.adi has to be a struct!');
            else
                opts.adi = struct();
            end
            if not(ml_field_set_to_value(opts.adi, 'info'))
                opts.adi.info = false;
            end
            if not(ml_field_set_to_value(opts.adi, 'res_tol'))
                opts.adi.res_tol = 1.0e-12;
            end
            if not(ml_field_set_to_value(opts.adi, 'rel_diff_tol'))
                opts.adi.rel_diff_tol = 0;
            end
            if not(ml_field_set_to_value(opts.adi, 'maxiter'))
                opts.adi.maxiter = 300;
            end

        case 'nm' % Newton options.
            if ml_field_set_to_value(opts, 'nm')
                assert(isa(opts.nm, 'struct'), ...
                    'MORLAB:data', ...
                    'The parameter opts.nm has to be a struct!');
            else
                opts.nm = struct();
            end
            if not(ml_field_set_to_value(opts.nm, 'accumulateRes'))
                opts.nm.accumulateRes = false;
            end
            if not(ml_field_set_to_value(opts.nm, 'inexact'))
                opts.nm.inexact = false;
            end
            if not(ml_field_set_to_value(opts.nm, 'info'))
                opts.nm.info = false;
            end
            if not(ml_field_set_to_value(opts.nm, 'linesearch'))
                opts.nm.linesearch = false;
            end
            if not(ml_field_set_to_value(opts.nm, 'maxiter'))
                opts.nm.maxiter = 50;
            end
            if not(ml_field_set_to_value(opts.nm, 'res_tol'))
                opts.nm.res_tol = 1.0e-12;
            end
            if not(ml_field_set_to_value(opts.nm, 'rel_diff_tol'))
                opts.nm.rel_diff_tol = 0;
            end
            if not(ml_field_set_to_value(opts.nm, 'store_debug'))
                opts.nm.store_debug = false;
            end
            if not(ml_field_set_to_value(opts.nm, 'store_solfac'))
                opts.nm.store_solfac = false;
            end
            if not(ml_field_set_to_value(opts.nm, 'tau'))
                opts.nm.tau = 1;
            end

        case 'radi' % RADI options.
            if ml_field_set_to_value(opts, 'radi')
                assert(isa(opts.radi, 'struct'), ...
                    'MORLAB:data', ...
                    'The parameter opts.radi has to be a struct!');
            else
                opts.radi = struct();
            end
            if not(ml_field_set_to_value(opts.radi, 'compute_res'))
                opts.radi.compute_res = false;
            end
            if not(ml_field_set_to_value(opts.radi, 'compute_sol_fac'))
                opts.radi.compute_sol_fac = 1;
            end
            if not(ml_field_set_to_value(opts.radi, 'get_ZZt'))
                opts.radi.get_ZZt = true;
            end
            if not(ml_field_set_to_value(opts.radi, 'info'))
                opts.radi.info = false;
            end
            if not(ml_field_set_to_value(opts.radi, 'maxiter'))
                opts.radi.maxiter = 300;
            end
            if not(ml_field_set_to_value(opts.radi, 'res_tol'))
                opts.radi.res_tol = 1.0e-12;
            end
            if not(ml_field_set_to_value(opts.radi, 'rel_diff_tol'))
                opts.radi.rel_diff_tol = 0;
            end

        case 'ri' % Riccati iteration options.
            if ml_field_set_to_value(opts, 'ri')
                assert(isa(opts.ri, 'struct'), ...
                    'MORLAB:data', ...
                    'The parameter opts.ri has to be a struct!');
            else
                opts.ri = struct();
            end
            if not(ml_field_set_to_value(opts.ri, 'compres_tol'))
                opts.ri.compres_tol = 0;
            end
            if not(ml_field_set_to_value(opts.ri, 'info'))
                opts.ri.info = false;
            end
            if not(ml_field_set_to_value(opts.ri, 'lqg_solver'))
                opts.ri.lqg_solver = 'radi';
            end
            if not(ml_field_set_to_value(opts.ri, 'maxiter'))
                opts.ri.maxiter = 50;
            end
            if not(ml_field_set_to_value(opts.ri, 'riccati_solver'))
                opts.ri.riccati_solver = 'radi';
            end
            if not(ml_field_set_to_value(opts.ri, 'res_tol'))
                opts.ri.res_tol = 1.0e-12;
            end
            if not(ml_field_set_to_value(opts.ri, 'rel_diff_tol'))
                opts.ri.rel_diff_tol = 0;
            end

        case 'shifts' % Shift computation options.
            if ml_field_set_to_value(opts, 'shifts')
                assert(isa(opts.shifts, 'struct'), ...
                    'MORLAB:data', ...
                    'The parameter opts.shifts has to be a struct!');
            else
                opts.shifts = struct();
            end
            if not(ml_field_set_to_value(opts.shifts, 'method'))
                opts.shifts.method = 'projection';
            end
            if not(ml_field_set_to_value(opts.shifts, 'naive_update_mode'))
                opts.shifts.naive_update_mode = false;
            end
            if not(ml_field_set_to_value(opts.shifts, 'num_desired'))
                opts.shifts.num_desired = 6;
            end
            if not(ml_field_set_to_value(opts.shifts, 'period'))
                opts.shifts.period = 1;
            end

        otherwise
            error('MORLAB:data', ...
                ['The requested M-M.E.S.S. function' ...
                ' does not seem to have options.']);
    end
end
