function opts = ml_check_krylovopts(opts, m, tformat)
%ML_CHECK_KRYLOVOPTS Checks and sets options for Krylov subspace methods.
%
% SYNTAX:
%   opts = ML_CHECK_KRYLOVOPTS([], m)
%   opts = ML_CHECK_KRYLOVOPTS(opts, m)
%
% DESCRIPTION:
%   This function checks given options for Krylov subspace methods and sets
%   the default values if needed.
%
% INPUTS:
%   opts    - struct, contains the interpolation parameters
%   m       - nonnegative integer, dimension of potential tangential
%             directions
%   tformat - string, used to denote the time format, e.g.,
%             'ct' for continuous-time and 'dt' for discrete-time
%
% OUTPUTS:
%   opts - struct, checked and augmented option struct
%
% See also ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(3, 3);

if isempty(opts)
    opts = struct();
else
    assert(isa(opts, 'struct'), ...
        'MORLAB:data', ...
        'The input argument opts has to be a struct!');
end

ml_assert_integer(m, 'm');
ml_assert_nonnegscalar(m, 'm');

ml_assert_char(tformat, 'tformat');

assert(strcmpi(tformat, 'ct') || strcmpi(tformat, 'dt'), ...
    'MORLAB:data', ...
    'Time format can only be ''ct'' or ''dt''!');


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% OPTION CHECKER / CONSTRUCTOR.                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Determine number of interpolation points
if ml_field_set_to_value(opts, 'Points')
    assert(isa(opts.Points, 'double'), ...
        'MORLAB:data', ...
        'The parameter opts.Points must be a vector!');

    assert(isvector(opts.Points), ...
        'MORLAB:data', ...
        'The parameter opts.Points must be a vector!');

    assert(length(opts.Points) == opts.NumPts, ...
        'MORLAB:data', ...
        'The parameter opts.Points has the wrong length!');

    opts.NumPts = length(opts.Points);
elseif ml_field_set_to_value(opts, 'Orders')
    ml_assert_vector(opts.Orders, 'opts.Orders');

    opts.NumPts = length(opts.Orders);
elseif ml_field_set_to_value(opts, 'Directs')
    assert(isa(opts.Directs, 'double'), ...
        'MORLAB:data', ...
        'The parameter opts.Directs must be given as a matrix!');

    [p, opts.NumPts] = size(opts.Directs);

    assert((p == m) || (p == 1), ...
        'MORLAB:data', ...
        'The parameter opts.Directs has the wrong number of rows!');
else
    if ml_field_set_to_value(opts, 'NumPts')
        ml_assert_posinteger(opts.NumPts, 'opts.NumPts');
    else
        opts.NumPts = 50;
    end
end

% Tangential directions.
if ml_field_set_to_value(opts, 'Directs')
    assert(isa(opts.Directs, 'double'), ...
        'MORLAB:data', ...
        'The parameter opts.Directs must be given as a matrix!');

    [p, k] = size(opts.Directs);

    assert((p == m) || (p == 1), ...
        'MORLAB:data', ...
        'The parameter opts.Directs has the wrong number of rows!');

    assert(k == opts.NumPts, ...
        'MORLAB:data', ...
        'The parameter opts.Directs has the wrong number of columns!');
else
    if m > 4
        opts.Directs = randn(m, opts.NumPts);
    else
        opts.Directs = [];
    end
end

% Frequency range.
if ml_field_set_to_value(opts, 'FreqRange')
    ml_assert_vector(opts.FreqRange, 'opts.FreqRange');
    assert(length(opts.FreqRange) == 2, ...
        'MORLAB:data', ...
        'The parameter opts.FreqRange must be of length 2!');
else
    if strcmpi(tformat, 'ct')
        opts.FreqRange = [-4, 4];
    else
        opts.FreqRange = [-4, log10(pi)];
    end
end

% Hermite interpolation orders.
if ml_field_set_to_value(opts, 'Orders')
    ml_assert_vector(opts.Orders, 'opts.Orders');

    assert(length(opts.Orders) == opts.NumPts, ...
        'MORLAB:data', ...
        'The parameter opts.Orders has the wrong length!');
else
    opts.Orders = ones(1, opts.NumPts);
end

% Interpolation points.
if ml_field_set_to_value(opts, 'Points')
    assert(isa(opts.Points, 'double'), ...
        'MORLAB:data', ...
        'The parameter opts.Points must be a vector!');

    assert(isvector(opts.Points), ...
        'MORLAB:data', ...
        'The parameter opts.Points must be a vector!');

    assert(length(opts.Points) == opts.NumPts, ...
        'MORLAB:data', ...
        'The parameter opts.Points has the wrong length!');
else
    if strcmpi(tformat, 'ct')
        opts.Points = 1i * logspace(min(opts.FreqRange), ...
            max(opts.FreqRange), opts.NumPts);
    else
        opts.Points = exp(1i * logspace(min(opts.FreqRange), ...
            max(opts.FreqRange), opts.NumPts));
    end
end

% Basis realification.
if ml_field_set_to_value(opts, 'RealVal')
    ml_assert_boolean(opts.RealVal, 'opts.RealVal');
else
    opts.RealVal = true;
end
