function opts = ml_check_cell_param(param, name, assertFn, default)
%ML_CHECK_CELL_PARAM Performs an assert on all elements in a cell-array.
%
% SYNTAX:
%   opts    = ML_CHECK_CELL_PARAM(opts, field, assertFn, default)
%   cellArr = ML_CHECK_CELL_PARAM(param, name, assertFn)
%
% DESCRIPTION:
%   Checks for the field of a struct or any variable to pass assertFn
%   and packs it into a cell array.
%   If the given variable already is a cell array, all elements are tested
%   and empty values are overwritten with the default.
%
% INPUTS:
%   opts     - struct, having its field  checked
%   field    - character array, name of struct's the field
%   assertFn - function, behaving like ml_assert_*
%   default  - value that is used if the field does not exist
%   param    - the value to be checked
%   name     - character array, used in error messages
%
% OUTPUTS:
%   opts    - structure, copy of the input with the field set and put into
%             a cell array
%   cellArr - the input value wrapped in a cell array
%
% See also ml_field_set_to_value.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

narginchk(3, 4);

assert(ischar(name), ...
    'MORLAB:data', ...
    'The input argument name has to be a character array!');
assert(isa(assertFn, 'function_handle'), ...
    'MORLAB:data', ...
    'The input argument assertFn has to be a function handle!');

if (nargin == 4) && isa(param, 'struct')
    opts  = param;
    field = name;

    if ml_field_set_to_value(opts, field)
        if not(isa(opts.(field), 'cell'))
            % Strings are always arrays.
            assert((length(opts.(field)) == 1) || ischar(opts.(field)), ...
                'MORLAB:data', ...
                'The parameter%s must be a cell-array.', ...
                strcat('opts.',field));

            opts.(field) = {opts.(field)};
        end

        for k = 1:length(opts.(field))
            % Empty values are overwritten with the default.
            if isempty(opts.(field){k})
                opts.(field){k} = default;
            else
                assertFn(opts.(field){k}, sprintf('opts.%s{%d}',field,k));
            end
        end
    else
        opts.(field) = {default};
    end
else
    narginchk(3, 3);

    if not(isa(param, 'cell'))
        % Strings are always arrays.
        assert((length(param) == 1) || ischar(param), ...
            'MORLAB:data', ...
            'The parameter%s must be a cell-array.', ...
            name);

        param = {param};
    end

    for k = 1:length(param)
        assertFn(param{k}, sprintf('%s{%d}',name,k));
    end
    opts = param;
end
