function [] = ml_assert_sosys(sys)
%ML_ASSERT_SOSYS Checks if a given structure is a second-order system.
%
% SYNTAX:
%   ML_ASSERT_SOSYS(sys)
%
% DESCRIPTION:
%   This function checks if a given structure is a second-order system with
%   appropriate matrices and dimensions and throws an error if not.
%
% INPUTS:
%   sys - structure, which shall be checked
%
% OUTPUTS:
%   ---
%
% See also ml_assert_stdsys, ml_assert_descsys.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


narginchk(1, 1);

assert(isa(sys, 'struct'), ...
    'MORLAB:data', ...
    'The parameter sys has to be a struct!');

assert(isfield(sys, 'M'), ...
    'MORLAB:data', ...
    'The system must have an M matrix!');

n = size(sys.M, 1);

assert(isequal(size(sys.M), [n n]), ...
    'MORLAB:data', ...
    'The matrix M has to be square!');

assert(isfield(sys, 'E'), ...
    'MORLAB:data', ...
    'The system must have an E matrix!');

assert(isequal(size(sys.E), [n n]), ...
    'MORLAB:data', ...
    'The matrix E must have the same dimensions as M!');

assert(isfield(sys, 'K'), ...
    'MORLAB:data', ...
    'The system must have an E matrix!');

assert(isequal(size(sys.K), [n n]), ...
    'MORLAB:data', ...
    'The matrix K must have the same dimensions as M!');

assert(isfield(sys, 'Bu'), ...
    'MORLAB:data', ...
    'The system must have an Bu matrix!');

assert(size(sys.Bu, 1) == n, ...
    'MORLAB:data', ...
    'The matrix Bu must have the same number of rows as M!');

assert(ml_field_set_to_value(sys, 'Cp') ...
    || ml_field_set_to_value(sys, 'Cv'), ...
    'MORLAB:data', ...
    'The system must have an output matrix Cp or Cv!');

if ml_field_set_to_value(sys, 'Cp')
    assert(size(sys.Cp, 2) == n, ...
        'MORLAB:data', ...
        'The matrix Cp must have the same number of columns as M!');
end

if ml_field_set_to_value(sys, 'Cv')
    assert(size(sys.Cv, 2) == n, ...
        'MORLAB:data', ...
        'The matrix Cv must have the same number of columns as M!');
end

if ml_field_set_to_value(sys, 'Cp') && ml_field_set_to_value(sys, 'Cv')
    assert(size(sys.Cp, 1) == size(sys.Cv, 1), ...
        'MORLAB:data', ...
        'The matrix Cp must have the same number of rows as Cv!');
end
